"""
File     : Expansions.py
Author   : Ramon Aragues
Creation : 04.03.2004
Contents : main class describing expansions that can be applied to Graph 
Comments :
=======================================================================================================

This is the template class for performing your own expansions to graph objects.

If you want to implement your own expansion class, you  must implement all methods described below.

You can follow the classes implemented in code/Graph/PianaGraph/PianaGraphExpansions.py

"""
# Expansions.py: implements a class describing expansions that can be applied to Graph objects
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues


from GraphEdge import GraphEdge
from GraphNode import GraphNode


class Expansion:
    """
    The base expansion class used as a basis for EdgeExpansion and NodeExpansion
    """
    pass


class NodeExpansion(Expansion):
    """
    The base class for expansion of nodes. All node expansions must be subclasses of this class.

    A node will borrow interactions from other nodes (even if they do not appear in the graph) if condition X is shared between the source node
    and the other nodes existing in the database.
    
    """
    def __init__(self):
        """
        """
        pass

    def get_expansion_name(self):
        """
        This must be overridden with a method that returns the name of the expansion
        """
        raise RuntimeError("method NodeExpansion.get_expansion_name should be overridden by specific method of expansion")


    def create_new_node_attribute(self, node_id= None):
        """
        This must be overridden with a method that creates an node attribute 
        """
        raise RuntimeError("method NodeExpansion.create_new_node_attribute should be overridden by specific method of expansion")


    def create_new_edge_attribute(self, edge_db_id= None, expansion_type= None, propagated_from_node= None, propagated_from_edge_id=None,
                                  propagated_from_db_edge_id=None, extended_from_node= None):
        """
        This must be overridden with a method that creates and attribute of type PianaGraphEdgeAttribute with given characteristics
        """
        raise RuntimeError("method NodeExpansion.create_new_edge_attribute should be overridden by specific method of expansion")

    def get_partner_ids(self, node_id):
        """
        This must be overridden with a method that returns a list of partner_ids of node_id
        """
        raise RuntimeError("method NodeExpansion.get_partner_ids should be overridden by specific method of expansion")


    def get_db_edge_ids_from_node(self, node_id):
        """
        This must be overridden with a method that returns a list of database ids for node_id
        """
        raise RuntimeError("method NodeExpansion.get_db_edge_ids_from_node should be overridden by specific method of expansion")

    def get_edges_attributes_from_node(self, node_id):
        """
        This must be overridden with a method that returns from a database edges attributes of edges where "node_id" appears 
        
        """
        raise RuntimeError("method NodeExpansion.get_node_edges should be overridden by specific method of expansion")
    
    def get_partner_node_id(self, edge_id, node_id):
        """
        This must be overridden with a method that returns from a database the partner node of "node_id" in edge "edge_id"
        
        """
        raise RuntimeError("method NodeExpansion.get_partner_node_id should be overridden by specific method of expansion")
    

    def get_external_code(self, node_id, code_type= None, alternative_code_types= None):
        """
        This must be overridden with a method that returns the external database code for node_id
        
        """
        raise RuntimeError("method NodeExpansion.get_external_code should be overridden by specific method of expansion")


    def class_test(self, node_id = None, class_name = None):
        """
        This must be overridden with a method that tests whether the node_id belongs to class_name or not

        returns 1 if class test is passed, 0 otherwise
        """
        raise RuntimeError("method NodeExpansion.class_test should be overridden by specific method of expansion")
   
    
    def get_sharing_nodes(self, node_id):
        """
        This must be overridden with a method that returns a list of nodes that share a characteristic with node_id

        This characteristic will determine the expansion: edges will be propagated between nodes that share that characteristic.

        This method is the only one that is specific to this characteristic: all the other methods that have to be overridden
        are specific to the external database, but not to the characteristic
        
        """
        raise RuntimeError("method NodeExpansion.get_sharing_nodes should be overridden by specific method of expansion")
    
    def get_partners_of_nodes_sharing_characteristic(self, node_id):
        """
        Returns a list of partner proteins of proteins that share cog with proteinPiana node_id
        
        returns empty list if nothing found

        ATTENTION: currently not being used!!!
        
        """
        raise RuntimeError("method NodeExpansion.get_partners_of_proteins_sharing_characteristic should be overridden by specific method of expansion")


class EdgeExpansion(Expansion):
    """
    The base class for expansion of nodes. All node expansions must be subclasses of this class.
    
    TO CHECK!!! Does it make sense to expand an edge? What characteristics might share two edges that expands one to the other (interchange the nodes
    that they join...)
    """
    def get_expanded_edges(self,edge):
        """
        This should be overridden with a method that returns a list of edges that pass the expansion test

        In its current form this will return 1 if argument "edge" is a GraphNode object or 0 otherwise.
        """
        raise RuntimeError("method EdgeExpansion.get_expanded_edges should be overridden by specific method of expansion")
