"""
File        : GraphCluster.py
Author      : Pablo Boixeda & Ramon Aragues
Creation    : 4.2005
Contents    : Implements a Graph where the nodes are clusters composed of node attributes of another type of graph
Called from : Programs that implements graph clustering

=======================================================================================================

Implements a Graph where the nodes are clusters composed of node attributes of another type of graph

"""

# GraphCluster.py: Implements a Graph where the nodes are clusters composed of node attributes of another type of graph
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

from Graph import *
from GraphNode import *
from GraphClusterNodeAttribute import *

#-----------------------------------------------------------------------------------------------
class GraphCluster(Graph):
#-----------------------------------------------------------------------------------------------

    def __init__(self, graph_id=None):
        """
        Method used to initialise ClusterGraph object
        """
        
        self.old_node_correspondence={}
        
        Graph.__init__(self, graphID= graph_id)

    def do_action(self):
        """
        this is a generic method that can be used by particular GraphCluster subclasses to do
        something after the clustering of this level (eg. in CIR, we fuse as well the clusters
        that have a common cluster partner).

        It is called from Clustering.cluster_graph after each step of the clustering

        In those GraphClusters that nothing has to be done are not affected, since this method won't exist
        and this generic method will be called (which doesn't do anything)
        """
        pass
    
    def create_grouped_node(self, node_id1= None, node_id2= None, new_node_id= None, old_graph= None):
        """
        This method joins to GraphNodes into one GraphNode. These two node attributtes are inserted
        into the new node attribute. This attribute is a list of attributes

        returns the resulting node

        "node_id1" is the id from one of the clusters to join
        
        "node_id2" is the id from one of the clusters to join
        
        "new_node_id" is the id of the new cluster node to create.
        
        "old_graph" is the graph that contains the nodes with node_id1 and node_id2
        
        """
        if node_id1 is None or node_id2 is None:
            raise ValueError("Error: how can I create a clustered node from a None id?")

        print "Joining nodes %s and %s to form a new node with id %s" %(node_id1, node_id2, new_node_id)

        node1=old_graph.get_node(node_id1)
        node2=old_graph.get_node(node_id2)

        NewClusterNodeAttribute= GraphClusterNodeAttribute()
        NewClusterNodeAttribute.add_element_list(list_node_attribute_object= node1.get_node_attribute_object().get_list_elements())
        NewClusterNodeAttribute.add_element_list(list_node_attribute_object= node2.get_node_attribute_object().get_list_elements())
        
        NewNode=GraphNode(nodeID= new_node_id,
                          attribute = NewClusterNodeAttribute,
                          graph = self)

        return NewNode

    def print_cluster_composition(self, output_target= sys.stdout):
        """
        prints the node ids for each cluster in the graph

        This is very general: should be overwritten by a method specific to the clustering being performed
        """
        nodes=self.get_node_object_list()
        
        for node in nodes:
            # node is a cluster, which has an attribute that contains elements (each element is a GraphNodeAttribute of
            #                                                                   the graph we are clustering)
            output_target.write( "cluster %s: " %node.get_node_id() )
            elements_list=node.get_node_attribute_object().get_list_elements()
            for element in elements_list:
                output_target.write("%s --" %(element.get_node_id()) )
            output_target.write("\n")

    def print_pairs_same_cluster(self, output_target):
        """
        prints pairs of elements  that appear in the same cluster

        To be overwritten by the subclass of GraphCluster in case an element is not described by its identifier
        """
        nodes=self.get_node_object_list()
        
        for node in nodes:
            elements_list=node.get_node_attribute_object().get_list_elements()
            number_of_elements = len(elements_list)
            for i in range(number_of_elements):
                for j in range(i+1, number_of_elements):
                    output_target.write( "p;%s-%s\n" %(elements_list[i].get_node_id(), elements_list[j].get_node_id()) )
                # END OF for j in range(i+1, number_of_elements):
            # END OF for i in range(number_of_elements):
        # END OF for node in nodes:
        
    
    def print_cluster_interactions(self, output_target= sys.stdout):
        """
        prints the edge table of the graph

        This is very general: should be overwritten by a method specific to the clustering being performed
        """
        self.output_edges_table(output_target=output_target)
            

    def print_pairs_interactions(self, output_target):
        """
        prints pairs of elements whose clusters interact

        To be overwritten by the subclass of GraphCluster in case an element is not described by its identifier
        """
        for edge in self.get_edge_object_list():

            node_id_start = edge.get_start_node_id()
            node_start = self.get_node(identifier= node_id_start, get_mode="error") 
            list_elements_start = node_start.get_node_attribute_object().get_list_elements()
            
            node_id_end = edge.get_end_node_id()
            node_end = self.get_node(identifier= node_id_end, get_mode="error") 
            list_elements_end = node_end.get_node_attribute_object().get_list_elements()

            for one_element_start in list_elements_start:
                for one_element_end in list_elements_end:
                    output_target.write("i;%s-%s\n" %(one_element_start.get_node_id(), one_element_end.get_node_id()) )
                # END OF for one_element_end in list_elements_end:
            # END OF for one_element_start in list_elements_start:
