"""
File     : GraphEdge.py
Author   : Daniel Jaeggi and Ramon Aragues
Creation : 2003
Contents : main class providing graph processing methods
Comments :

=======================================================================================================
The basic edge class linking teo nodes in a graph.

The edge is initialised with two node objects and optional attributes.
Further attributes may be added later.

"""

# GraphEdge.py: implements a class for edges contained in a Graph object
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

import sys

from GraphEdgeAttribute import GraphEdgeAttribute
#from Filter import EdgeFilter, NodeFilter

verbose = 0

class GraphEdge:
    """
    The basic edge class linking teo nodes in a graph.

    The edge is initialised with two node objects and optional attributes.
    Further attributes may be added later.

    """
    
    def __init__(self, node1_id, node2_id, attribute_object = None, graph= None, original= 1, propagated= 0, extended= 0, hidden = 0):
        """
        GraphEdge is initialized with two node identifiers, optional GraphEdgeAttribute and parameters indicating origin of the edge

        "original" states whether the edge comes from the standard set of edges (loaded from file, database, ...) or not

        "expanded_from" and "expansion_type" indicate if the edge comes from an expansion

        "graph" is the Graph where the edge belongs (one edge can only be in one graph)
        """

        node1_id = int(node1_id)
        node2_id = int(node2_id)
        
        if node1_id < node2_id:     
            self.start_node_id = node1_id
            self.end_node_id = node2_id
            
        else:
            self.start_node_id = node2_id
            self.end_node_id = node1_id
            
        if attribute_object is None or isinstance(attribute_object, GraphEdgeAttribute):   
            self.attribute = attribute_object
        else:
            raise TypeError("Should be passed Attribute")
        
        # _gen_edge_edgeID generates an edgeID from the two nodes (there can only be one edge between two nodes)
        self.edgeID = GraphEdge._gen_edge_edgeID( node_id1= self.get_start_node_id(),
                                                  node_id2= self.get_end_node_id()   )


        self.graph = graph
        
        self.hidden = hidden

        # An edge can be in the graph because it was added by:
        #                  - original: standard graph building 
        #                  - propagated: propagation of edges between nodes that share a characteristic
        #                  - extended: addition of edges for those nodes added in an expansion
        #
        # An edge can be "original", "propagated" and "extended" at the same time when it was originally in the set of edges of the graph
        # and was found as well as a result of an expansion_propagation or expansion_extension
        #
        # Therefore, self.original (1: is original; 0: not original), self.propagated (1: is propagated; 0: not propagated) and
        # self.extended (1: is extended; 0: not extended) can be 1
        #
        self.original_yes = original     # states whether this edge comes from the original set of edges (different from expansions, ...)
        
        self.propagated_yes = propagated # states whether this edge comes from an expansion_propagation operation on the graph
                                         # expansion_propagation operation is the propagation of edges between nodes that share a characteristic
                                         # details on how it was introduced in the graph are described inside the edge attribute, specific
                                         # to each graph type
                                         
        self.extended_yes = extended     # states whether this edge comes from an expansion_extension operation on the graph
                                         # expansion_extension operation is the addition of edges for those nodes added in an expansion
                                         # details on how it was introduced in the graph are described inside the edge attribute, specific
                                         # to each graph type



    # --------------------------
    # basic type operations
    # --------------------------

    def __eq__(self,other):
        return isinstance(other, GraphEdge) and (self.edgeID == other.edgeID)

    def __ne__(self,other):
        return not self.__eq__(other)

    def __str__(self):
        string = "<GraphEdge: "
        string +="Start node %s" %self.get_start_node_id()
        string +=", End node %s" %self.get_end_node_id()
        if self.attribute is not None:
            string +=", Attributes "
            string += str(self.attribute)
        string += ">"
        return string

    def __repr__(self):
        return self.edgeID
    
    # --------------------------
    # basic GraphEdge methods
    # --------------------------

    def _gen_edge_edgeID(node_id1, node_id2):
        """
        Returns a string to be used as an edge edgeID. node_id1 and node_id2 are the two node ids
        defining the edge.
        """
        string = None

        # TO DO!!! A better (smaller in size) edge id would be node1.node2, but with decimals instead of in a string
        #          Would it work????
        
        if node_id1 <= node_id2:
            string = "%s.%s" %(node_id1, node_id2)
        else:
            string = "%s.%s" %(node_id2, node_id1)
        return string

    _gen_edge_edgeID = staticmethod(_gen_edge_edgeID)


    def get_start_node_id(self):
        """
        returns the start node object of this edge
        """
        return self.start_node_id

    def get_end_node_id(self):
        """
        returns the  end node object of this edge
        """
        return self.end_node_id

    def get_node_ids(self):
        """
        returns both nodes in a list [start_node, end_node]
        """
        return [self.get_start_node_id(), self.get_end_node_id()]
    
    def get_edge_id(self):
        """
        returns the edge id internal to Graph (composed by the two node identifiers)
        this is different from the database identifier, which is obtained from the edge attribute
        """
        return self.edgeID


    def get_db_edge_id(self):
        """
        returns the identifier of the external database for this edge. If the edge doesn't come from an external database, it will
        return None

        this method simply calls the get_edge_db_id of the attribute, which is specific to each edge attribute.
        """
        return self.attribute.get_db_edge_id()

    def merge_edges(self, new_edge, ignore_ids=0):
        """
        merges two edges (ie. merges the attribute of the new_edge into the attribute of current edge)
        """
        self.merge_attribute(new_attribute_object= new_edge.get_edge_attribute_object(), ignore_ids= ignore_ids )
    
    def merge_attribute(self, new_attribute_object, ignore_ids=0):
        """
        Merges new_attribute with the existing attribute of the GraphEdge. Since the way of merging attributes
        depends on the kind of attribute, all the merging process is performed by the method merge_attribute of
        the corresponding attribute
        """

        if verbose:
            sys.stderr.write("Calling attribute.merge_attribute for edge_id %s." %(self.get_edge_id()))
            
            sys.stderr.write(" db_id of current attribute is: %s. New attribute db id is: %s\n\n" %(self.get_db_edge_id(),
                                                                                                    new_attribute_object.get_db_edge_id()))
            

        if not new_attribute_object is None and not self.attribute is None:
            
            self.attribute.merge_attribute(new_edge_attribute= new_attribute_object, ignore_ids= ignore_ids)

        
    
    def get_edge_attribute_object(self):
        """
        returns GraphEdgeAttribute object of the GraphEdge
        """
        return self.attribute

    def get_partner_id(self, query_node_id):
        """
        Returns the other partner node of the edge.
        Argument is one node id
        """
        if self.get_start_node_id() == query_node_id:
            return self.get_end_node_id()
        
        elif self.get_end_node_id() == query_node_id:
            return self.get_start_node_id()
        
        else:
            raise ValueError("Asking for the partner of %s in edge between %s and %s\n" %(query_node_id,
                                                                                          self.get_start_node_id(),
                                                                                          self.get_end_node_id()))

    def distance(self):
        """
        Returns a distance/length like value.
        """
        # TO DO!!!  LOTS OF WORK NEEDED HERE!
        
        # return a distance like value
        dist = 0.
        if self.attribute is not None:
            dist = self.attribute.distance()
        else:
            dist = 1.0
            
        return dist
    
    def direction(self):
        """
        Returns the direction of the edge
           0 --> Bidirectional
           1 --> A to B
          -1 --> B to A

        """

        direction = 0
        
        if self.attribute is not None:
            
            direction = self.attribute.direction()
        else:
            direction = 0 # default direction: Bidirectional

        return direction

    def set_attribute(self, attribute_object):
        """
        Adds an attribute to the edge.
        """
        if isinstance(attribute_object, GraphEdgeAttribute):
            self.attribute = attribute_object
        else:
            raise TypeError("Must be passed an GraphEdgeAttribute object")

        
    def is_hidden(self):
        """
        if node is hidden, returns 1
        if node is not hidden returns 0
        """
        if self.hidden:
            return 1
        else:
            return 0

    # TO DO!!!
    # I am not currently hiding/unhidding, but if I do, self.start_node will produce an error, because it doesn't exist anymore...
    # in case I need to access to the real node, I think I would have to get a graph reference so I can get_node from graph
    def hide(self):
        """
        Hides the edge and checks the two nodes defining the edge to see if they
        need to be hidden too.
        """
        if not self.hidden:
            self.hidden = 1
            start_node_object = self.graph.get_node(identifier=self.get_start_node_id())
            end_node_object = self.graph.get_node(identifier=self.get_end_node_id())
            
            if start_node_object.get_degree() == 0:
                if not start_node_object.is_hidden():
                    start_node_object.hide()
                    
            if end_node_object.get_degree() == 0:
                if not end_node_object.is_hidden():
                    end_node_object.hide()

    def unhide(self):
        """
        Unhides the edge and the nodes associated with it.
        """
        self.hidden = 0
        self.start_node.unhide()
        self.end_node.unhide()

    def set_original(self, original):
        """
        changes the status of the edge

        original == 1 means the edge was added to the graph from the original set of root nodes
        """
        self.original_yes = original

    def set_propagated(self, propagated):
        """
        changes the status of the edge

        propagated == 1 means the edge was added to the graph as a propagation of the edge of another node

        details on the kind of propagation can be obtained from the edge_attribute
        """
        self.propagated_yes = propagated

    def set_extended(self, extended):
        """
        changes the status of the edge

        extended == 1 means the edge was added to the graph as an edge of a node added to the graph as a result of an expansion

        details on the kind of extension can be obtained from the edge_attribute
        """
        self.extended_yes = extended

    def is_original(self):
        """
        returns whether the edge is original (1) or not (0)
        """
        return self.original_yes

    def is_propagated(self):
        """
        returns whether the edge is propagated (1) or not (0)

        more details about the propagation can be obtained from the edge_attribute
        """
        return self.propagated_yes

    def is_extended(self):
        """
        returns whether the edge is extended (1) or not (0)

        more details about the propagation can be obtained from the edge_attribute
        """
        return self.extended_yes
