"""
File     : GraphEdgeAttribute.py
Author   : Daniel Jaeggi and Ramon Aragues
Creation : 2003
Contents : class GraphEdgeAttribute

=======================================================================================================

Skeleton attribute classes for edges in a graph.

Edges in a graph are just abstract containers and only have
"real world" meanings by containing attributes. For each particular
problem, custon edge attributes must be defined.

For example, we may be modeling the network formed by Hollywood actors.
Actors form the nodes and films in which both actors have starred form
the edges. We could then define custom edge and node attributes thus:

class FilmLink(EdgeAttribute):
    # class defining edge between 2 actors based on appearance in the same film
    def __init__(self, film, director, year):
        self.film = film
        self.director = director
        self.year = year
    

Attributes in this way can be arbitrarily complex
"""

# GraphEdgeAttribute.py: implements a class for edge attribute contained in a GraphEdge object
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues



class GraphEdgeAttribute:
    """
    The master class for all GraphEdge attributes. It is envisaged that this
    class will form the basis for custom attribute classes.
    """
    def __eq__(self,other):
        """
        When defining new Attribute classes, the equality operator MUST be overridden.
        Otherwise, adding further attributes to existing nodes in the graph may cause
        duplicated attributes.
        """
        if self == other:
            return 1
        else:
            return 0

    def get_db_edge_id(self):
        """
        returns the external database identifier of the edge, in case the graph has been constructed from an external database

        Must be overridden by a method specific to the subclass of GraphEdgeAttribute that interfaces with the database
        
        """
        
        raise Error("This method should be overriden by specific database method of subclasses of GraphEdgeAttribute")   
        
    def distance(self):
        """
        To be overridden. Returns a distance like value.
        Defaults to return 1
        """
        return 1.0

    def direction(self):
        """
        To be overridden. Returns a direction like value.
        Defaults to return 0 (i.e. bidirectional)
        """
        return 0
 
    def merge_attribute(self, new_edge_attribute, ignore_ids):
        """
        must be overwritten by a method specific to the kind of Edge being used by the graph
        
        merges the attribute of the object with the new edge attribute passed as argument
        """
        pass
