"""
File        : IRGraph.py
Author      : Ramon Aragues
Creation    : 6.2.2006
Contents    : implementation of graph handling graphs of interacting regions
Called from : programs/classes that find CIRs 

=======================================================================================================


IRGraph is the intermediate step between a PPI network and a CIR-CIR network.


The process is as follows:

1. given a ppi network
2. create its IRGraph (each interaction --> one IR for each protein)
3. use Clustering to cluster the IRGraph into CIRs
   
"""

# IRGraph.py :  is the intermediate step between a PPI network and a CIR-CIR network.
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues


from Graph import *
from PianaDBaccess import *
from IRGraphNodeAttribute import *

#----------------------------------------------------------------------------------------------
class IRGraph(Graph):
#----------------------------------------------------------------------------------------------

    """
    Implements a Graph where nodes are Interacting Regions of proteins
    """
    def __init__(self, graphID = None, piana_access_object = None):

        self.graph_id = graphID
        self.piana_access = piana_access_object
        self.node_id_counter = 0

        Graph.__init__(self, self.graph_id)


    def _get_new_node_id(self):
        """
        This method is created to give an id to new nodes
        """
        new_node_id = self.node_id_counter
        self.node_id_counter += 1
        return new_node_id


    
    def initialize_IRGraph_from_PianaGraph(self, piana_graph = None):
        """
        Method that initializes this IRGraph from a PianaGraph
        Creates a two linked nodes (with IRGraphNodeAttribute) for each  interaction
        
 
        "piana_graph": PianaGraph object that we want to create the IRGraph from
        """

        if piana_graph is None:
            raise ValueError ("No PianaGraph given to transform into IRGraph")


        if verbose:
            sys.stderr.write("Initializing IRGraph from pianagraph\n")
            
        for one_edge in piana_graph.get_edge_object_list():
            # for each edge, create two IR nodes with a link between them and add everything to self

            # 1. get ids for nodes in IRGraph (each node is a 'potential interacting region'
            ir_id_a = self._get_new_node_id()
            ir_id_b = self._get_new_node_id()

            # 2. get proteinPianas and node object (ie protein object) from the original interaction in piana_graph
            proteinPiana_a = one_edge.get_start_node_id()
            proteinPiana_b = one_edge.get_end_node_id()
            
            protein_object_a = piana_graph.get_node(identifier= proteinPiana_a, get_mode="error" )
            protein_object_b = piana_graph.get_node(identifier= proteinPiana_b, get_mode="error" )

            # 3. create nodes for IRGraph (standard GraphNode with a IRGraphNodeAttribute)
            ir_node_attribute_a = IRGraphNodeAttribute(ir_id= ir_id_a,
                                                       proteinPiana_value= proteinPiana_a,
                                                       is_root_value= protein_object_a.is_root())
            ir_node_attribute_b = IRGraphNodeAttribute(ir_id= ir_id_b,
                                                       proteinPiana_value= proteinPiana_b,
                                                       is_root_value= protein_object_b.is_root())
        

            ir_node_a = GraphNode(nodeID= ir_id_a, attribute= ir_node_attribute_a, alternative_id= str(ir_id_a) + "." + str(proteinPiana_a) )
            ir_node_b = GraphNode(nodeID= ir_id_b, attribute= ir_node_attribute_b, alternative_id= str(ir_id_b) + "." + str(proteinPiana_b) )

            # 4. create edge for IRGraph (standard GraphEdge)
            
            new_edge_attribute = GraphEdgeAttribute() # empty attribute: nothing to do with it at the moment

            new_edge= GraphEdge(node1_id= ir_id_a, node2_id= ir_id_b,
                                attribute_object= new_edge_attribute) # I am sure this edge doesn't exist (new ir_ids
                                                                      # are generated at each iteration: therefore, no
                                                                      # new to do get_edge or create_edge

            self.add_node(ir_node_a)
            self.add_node(ir_node_b)
            self.add_edge(new_edge)
        # END OF for one_edge in piana_graph.get_edge_object_list():




    #
    # THESE METHODS HAVE NOT BEEN ADAPTED TO IR GRAPHS
    #
    def print_IR_node_values(self,go_node = None):
        """
        prints the label of one GO node
        
        "go_node": node from a go graph
        """

        node_attribute=go_node.get_node_attribute_object()

        return "%s        proteinPiana %s : IRTerm %s"  %(node_attribute.get_go_name(),
                                                         node_attribute.get_proteinPiana(),
                                                         node_attribute.get_term_id())

    def print_IR_dot_file(self, output_target= None):
        """
        method that prints IRGraph in dot format to a file object "output_target"
        """
        
        output_target.write("graph G { graph [orientation=portrait, pack=true, overlap=scale]\n")
        output_target.write(" node [shape=box,fontsize=12,width=0.15,height=0.15,style=filled,fillcolor=lightblue];\n")

        for node in self.get_node_object_list():
            if node.get_node_attribute_object().get_is_root():
                output_target.write(""""%s" [fillcolor = %s]\n""" %(self.print_go_node_values(go_node = node),"yellow"))

        for edge in self.get_edge_object_list():
            start_node= edge.get_start_node_id()
            end_node= edge.get_end_node_id()
            output_target.write(""" "%s" -- "%s" [len=2];\n""" %(self.print_go_node_values(go_node = self.get_node(start_node)),
                                                                 self.print_go_node_values(go_node = self.get_node(end_node))))
        # END OF for edge in self.edges:
                                 
        # print graph termination    
        output_target.write( "}\n")   

        
                    

                        
        
    


