"""
 File       : PianaSelectSQL.py
 Author     : R. Aragues & D. Jaeggi
 Creation   : 2003
 Contents   : subclasses for generating select sql statements
 Called from: PianaDBaccess

=======================================================================================================

This class is used in the same way as PianaInsertSQL

Each of the subclasses creates an sql query string that can be used to select information from Piana

This is done this way to generalize as much as possible the interaction of users with the database pianaDB.

A user that needs to select information into pianaDB will not create the SQL statement himself, but will
use one of these classes, together with method select_db_content() of class PianaDB

class PianaDBaccess integrates into a single and clear interface all the access to piana databases

"""

# PianaSelectSQL.py: subclasses for generating select sql statements
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

import sys

import PianaGlobals
import utilities

verbose = 0


class PianaSelectSQL:
    """
    General class for Piana SQL SELECT queries. All SQL queries reading piana should be extensions of this class
    """
    def get_sqlquery():
        """
        Override this with the sql query that returns the value being researched
        """
        return None
    
    get_sqlquery = staticmethod(get_sqlquery) 

#---------------------------------------------------------------------
# MISCELANEOUS SQL ACTIONS 
#---------------------------------------------------------------------
class LockTables(PianaSelectSQL):
    """
    Generates the SQL statement that locks tables indicated in argument

    Method used to return the SQL query that locks access to mysql tables. Insertions are faster if tables are previously locked.

    table_list is a list of table names
    """
    def get_sqlquery(table_list):

        tables_to_lock = ""

        number_of_tables = len(table_list)
        
        for i in range(number_of_tables - 1):
            
            tables_to_lock +=table_list[i]
            tables_to_lock +=" WRITE, "

        # adding last table to lock. This looks weird because apparently we
        # are adding the same element as in the loop, but in fact range() returns
        # numbers from 0 to number_of_elements -1
        # therefore, number_of_tables-1 has not been added in the loop
        tables_to_lock += table_list[number_of_tables-1]
        tables_to_lock +=" WRITE "

        sqlquery = """lock tables %s """  %tables_to_lock

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class UnlockTables(PianaSelectSQL):
    """
    Generates the SQL statement that unlocks tables previously locked

    Method used to return the SQL query that locks access to mysql tables. Insertions are faster if tables are previously locked.

    table_list is a list of table names
    """
    def get_sqlquery():

        sqlquery = """unlock tables""" 

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

#---------------------------------------------------------------------
# PROTEIN RELATED CLASSES
#---------------------------------------------------------------------

class SelectAllProteinPiana(PianaSelectSQL):
    """
    Generates the SQL statement that obtains all proteinPiana in database

    """
    def get_sqlquery():

        sqlquery = """select %s from %s """ %(PianaGlobals.proteinPiana_col,
                                              PianaGlobals.protein_table)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)


def _normalize_pdb(code_value):
    """
    returns a normalized pdb.chain code for string code_value
    """

    # if table is pdb, there is a bit of preprocessing to do
    #   old pdb codes used NULL chain values
    #   pianaDB table pdb contains a column (pdb_chain) where NULL chain values have been transformed to "A"
    #   therefore, if the code coming in has chain value NULL, we'll look in piana for chain value "A"
    # I also protect against wrong usage of pdb code (not writing the "." for empty chain values)

    if len(code_value) == 6:
        # code was correctly passed (ie. pdb.chain)
        pdb_fields = code_value.split(".")
        pdb_code = pdb_fields[0].strip()
        pdb_chain = pdb_fields[1].strip()

    elif len(code_value) == 5:
        # code was uncorrectly passed: two different incorrect options: pdb. or pdbchain

        if code_value[-1] == ".":
            pdb_code = code_value[:-1]
            pdb_chain = "A"
        else:
            pdb_code = code_value[:-1]
            pdb_chain = code_value[-1]

    elif len(code_value) == 4:
        # code was uncorrectly passed: no chain given (we will consider it to be the only chain in the file)

        pdb_code = code_value
        pdb_chain = "A"

    else:
        raise ValueError("Weird pdb code used: %s\n" %(code_value))

    if pdb_chain == "" or pdb_chain == "_":
        pdb_chain = "A"

    pdb_code = pdb_code + "." + pdb_chain

    return pdb_code

    
    
class SelectProteinPiana(PianaSelectSQL):
    """
    Generates the SQL statement that obtains proteinPiana from any other protein identifier (swissprot, ...)

    Method get_sqlquery used to return the SQL query that retrieves the proteinPianas equivalent to "protein_code" of type "code_type"

    Valid "code_type" value are those in PianaGlobals.xxxx_col that are in a table where there is as well a proteinPiana_col

    if protein_code being passed is a pdb code, the format of the code must be pdb_code.chain_id  If the chain_id is None, write pdb_code. (leaving the dot)

        "source_db_info" determines if information about the sourceDB that inserted the proteinPiana is returned or not
           - "no" will simply return a list of proteinPianas
           - "yes" will return a list of tuples (proteinPiana, sourceDBID)

    """
           
    def get_sqlquery(protein_code= None, code_type= None, source_db_info= "no"):

        if protein_code is None or code_type is None:
            raise ValueError("Trying to Select the proteinPiana of None arguments")

        protein_code_table = utilities.get_code_table(code_type_col= code_type)

        if protein_code_table is None:
            raise ValueError("trying to Select the proteinPiana for a type that doesn't have a table associated: %s" %(code_type))

        if protein_code_table == PianaGlobals.pdb_table:
            # if dealing with pdb codes, normalization required...
            protein_code = _normalize_pdb(code_value=protein_code)
        # END OF if protein_code_table == PianaGlobals.pdb_table:

        if source_db_info == "no":
            # add empty string to select: do not get source db
            get_source_db = ""
        elif source_db_info == "yes":
            if protein_code_table == PianaGlobals.protein_table:
                # if proteinPiana to be returned from protein table (which has no source_db) we
                # have to modify the query so it returns the name of the table...
                get_source_db = ', "protein"'
            else:
                get_source_db = ", %s" %(PianaGlobals.sourceDBID_col)


        
        sqlquery = """select %s%s from %s where %s = "%s" """ %(PianaGlobals.proteinPiana_col,
                                                                get_source_db,
                                                                protein_code_table,
                                                                code_type,
                                                                protein_code)

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)



class SelectProteinPianaTax(PianaSelectSQL):
    """
    Generates the SQL statement that obtains proteinPiana (of a specific taxonomy) from any other protein identifier (swissprot, ...)

    Method get_sqlquery is used to return the SQL query that retrieves the proteinPianas (of a specific taxonomy)
    equivalent to "protein_code" of type "code_type"

    Valid "code_type" value are those in PianaGlobals.xxxx_col that are in a table where there is as well a proteinPiana_col

    Valid taxonomy_value values are those in NCBI taxonomy database

    "source_db_info" determines if information about the sourceDB that inserted the proteinPiana is returned or not
           - "no" will simply return a list of proteinPianas
           - "yes" will return a list of tuples (proteinPiana, sourceDBID)


    """
    def get_sqlquery(protein_code= None, code_type= None, taxonomy_value= None, source_db_info= "no"):

        if protein_code is None or code_type is None or taxonomy_value is None:
            raise ValueError("Trying to call SelectProteinPianaTax with None arguments")

        if taxonomy_value == 0:
            return SelectProteinPiana.get_sqlquery(protein_code= protein_code, code_type= code_type, source_db_info= source_db_info)

        protein_code_table = utilities.get_code_table(code_type_col= code_type)

        if protein_code_table is None:
            raise ValueError("trying to Select the proteinPiana for a type that doesn't have a table associated")


        if protein_code_table == PianaGlobals.pdb_table:
            # if dealing with pdb codes, normalization required...
            protein_code = _normalize_pdb(code_value=protein_code)
        # END OF if protein_code_table == PianaGlobals.pdb_table:

        
        if source_db_info == "no":
            # add empty string to select: do not get source db
            get_source_db = ""
        elif source_db_info == "yes":
            if protein_code_table == PianaGlobals.protein_table:
                # if proteinPiana to be returned from protein table (which has no source_db) we
                # have to modify the query so it returns the name of the table...
                get_source_db = ', "protein"'
            else:
                get_source_db = ", code_table.%s" %(PianaGlobals.sourceDBID_col)
            

        # the query in this method is built so it only retrieves proteinPianas that have taxonomy taxonomy_value
        #    --> code_table and species_table are aliases used for simplifying the query...
        sqlquery = """select distinct code_table.%s%s from %s code_table, %s species_table \
            where code_table.%s="%s" and code_table.%s=species_table.%s and species_table.%s= %s """ %(PianaGlobals.proteinPiana_col,
                                                                                                       get_source_db,
                                                                                                       protein_code_table,
                                                                                                       PianaGlobals.proteinSpecies_table,
                                                                                                       code_type,
                                                                                                       protein_code,
                                                                                                       PianaGlobals.proteinPiana_col,
                                                                                                       PianaGlobals.proteinPiana_col,
                                                                                                       PianaGlobals.speciesNCBI_col,
                                                                                                       taxonomy_value)

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)



class SelectProteinSimilarity(PianaSelectSQL):
    """
    Generates the SQL statement that obtains a pair of similar proteins, if they exist

    Selects two elements, must be taken into account when making the call to PianaDB
    """

    def get_sqlquery(proteinPianaA_value, proteinPianaB_value):

        # make sure proteinPianaA < proteinPianaB
        if proteinPianaA_value > proteinPianaB_value:
            temp = proteinPianaA_value
            proteinPianaA_value = proteinPianaB_value
            proteinPianaB_value = temp
        
        sqlquery = """select %s, %s from %s where %s = %s and %s=%s""" %(PianaGlobals.proteinPianaA_col, PianaGlobals.proteinPianaB_col,
                                                                         PianaGlobals.proteinSimilarity_table,
                                                                         PianaGlobals.proteinPianaA_col, proteinPianaA_value,
                                                                         PianaGlobals.proteinPianaB_col, proteinPianaB_value)

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectSimilarProteins(PianaSelectSQL):
    """
    Generates the SQL statement that obtains a list of proteins that are similar to a proteinPiana

    "mode" can be:

       - "higher" : searches in the column proteinPianaB
       - "lower"  : searches in the column proteinPianaA

    To get all similar proteins you need to do two queries, one with mode higher and another with mode lower. PianaDBaccess.get_similar_proteins_dic does it...
    """
    def get_sqlquery(proteinPiana_value, mode= None):

        if mode == "higher":
            sqlquery = """select %s from %s where %s = %s""" %(PianaGlobals.proteinPianaB_col, 
                                                                         PianaGlobals.proteinSimilarity_table,
                                                                         PianaGlobals.proteinPianaA_col, proteinPiana_value)
        elif mode == "lower":
            sqlquery = """select %s from %s where %s = %s""" %(PianaGlobals.proteinPianaA_col, 
                                                                         PianaGlobals.proteinSimilarity_table,
                                                                         PianaGlobals.proteinPianaB_col, proteinPiana_value)
        else:
            raise ValueError("Incorrent mode (%s) used: valid values are higher and lower" %(mode))
            

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)


class SelectProteinCode(PianaSelectSQL):
    """
    Generates the SQL statement that obtains protein codes from protein tables

    This is a generalization that can be used for any protein table that has at least these two fields:

       proteinPiana
       proteinCodeX
    
    Currently, this class can be used for tables: swissProt,  proteinName, geneName, ec, emblAcc, gi, pirEntry, pirAcc, interpro

    if you need to select swissAccession primary number you cannot use this class... use instead SelectSwissAccession (because of isPrimary)
    """
    def get_sqlquery(proteinCode_table, proteinCode_col, proteinPiana_value, source_db_info="no"):

        if source_db_info == "no":
            sqlquery = """select %s from %s where %s = %s""" %(proteinCode_col, proteinCode_table
                                                               , PianaGlobals.proteinPiana_col , proteinPiana_value)
        else:
            sqlquery = """select %s, %s from %s where %s = %s""" %(proteinCode_col, PianaGlobals.sourceDBID_col,
                                                                   proteinCode_table
                                                                   , PianaGlobals.proteinPiana_col , proteinPiana_value)
            

            
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectSwissAccession(PianaSelectSQL):
    """
    Generates the SQL statement that obtains swiss accession numbers corresponding to proteinPiana

    get_sqlquery method used to return the SQL query that retrieves the swiss accession number corresponding to proteinPiana

        if return_primary == 1, then returns only primary swiss accessions
        if return_primary == 0, then returns all swiss accession numbers
    """
    def get_sqlquery(proteinPiana_value, return_primary=1, source_db_info="no"):

        if source_db_info == "no":
            if return_primary == 1:
                sqlquery = """select %s from %s where %s = %s and %s = %s""" %(PianaGlobals.swissAccessionID_col, PianaGlobals.swissAccession_table,
                                                                               PianaGlobals.proteinPiana_col , proteinPiana_value,
                                                                               PianaGlobals.isPrimary_col, return_primary )
            else:
                sqlquery = """select %s from %s where %s = %s""" %(PianaGlobals.swissAccessionID_col, PianaGlobals.swissAccession_table,
                                                                   PianaGlobals.proteinPiana_col , proteinPiana_value)
        else:
            # source_db_info required
            if return_primary == 1:
                sqlquery = """select %s, %s from %s where %s = %s and %s = %s""" %(PianaGlobals.swissAccessionID_col, PianaGlobals.sourceDBID_col,
                                                                                   PianaGlobals.swissAccession_table,
                                                                                   PianaGlobals.proteinPiana_col , proteinPiana_value,
                                                                                   PianaGlobals.isPrimary_col, return_primary )
            else:
                sqlquery = """select %s, %s from %s where %s = %s""" %(PianaGlobals.swissAccessionID_col, PianaGlobals.sourceDBID_col,
                                                                       PianaGlobals.swissAccession_table,
                                                                       PianaGlobals.proteinPiana_col , proteinPiana_value)
            

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)


class SelectProteinTableColumn(PianaSelectSQL):
    """
    Generates the SQL statement that obtains a column value of table protein, given a proteinPiana identifier

    This is a generalization that can be used for any column of table protein
    """
    def get_sqlquery(proteinTable_column_value, proteinPiana_value):
        
        sqlquery = """select %s from %s where %s = %s""" %(proteinTable_column_value, PianaGlobals.protein_table
                                                           , PianaGlobals.proteinPiana_col , proteinPiana_value)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)



class SelectProteinPianaCorrespondence(PianaSelectSQL):
    """
    Generates the SQL statement that obtains the proteinPiana associated to a proteinMD5 in table proteinCorrespondence

    This class is use to keep consistency of proteinPiana <--> proteinSequence between different versions of piana databases
    """
    def get_sqlquery(proteinMD5_value, tax_id_value):
        
        sqlquery = """select %s from %s where %s = "%s" and %s=%s """ %(PianaGlobals.proteinPiana_col, PianaGlobals.proteinCorrespondence_table,
                                                                        PianaGlobals.proteinMD5_col , proteinMD5_value,
                                                                        PianaGlobals.speciesNCBI_col, tax_id_value)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

    
class SelectNewProteinPiana(PianaSelectSQL):
    """
    Generates the SQL statement that obtains a new proteinPiana identifier from proteinPianaCounter table

    After getting the new proteinPiana identifier, the method calling this class must make sure that the counter in increased by 1
    """
    def get_sqlquery():
        
        sqlquery = """select %s from %s""" %(PianaGlobals.proteinPianaCounter_col, PianaGlobals.proteinPianaCounter_table)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

    
class SelectCommonCodeProteins(PianaSelectSQL):
    """
    Generates the SQL statement that obtains those proteins that share a code with a given protein

    The list will not contain the protein passed as reference
    """
    def get_sqlquery( proteinCode_table, proteinCodeID_col, proteinPiana_value):

        sqlquery = """select t2.%s from %s t1, %s t2 where t1.%s = %s and t1.%s=t2.%s and t2.%s <> %s""" %(PianaGlobals.proteinPiana_col,
                                                                                                           proteinCode_table,
                                                                                                           proteinCode_table,
                                                                                                           PianaGlobals.proteinPiana_col, proteinPiana_value,
                                                                                                           proteinCodeID_col, proteinCodeID_col,
                                                                                                           PianaGlobals.proteinPiana_col, proteinPiana_value)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectCommonCharacteristicProteins(PianaSelectSQL):
    """
    Generates the SQL statement that obtains those proteins with characteristics described to get_sqlquery

    This class is used for example to get all proteins that belong to a certain species, by passing the approapiate arguments
    as shown:  get_sqlquery(proteinCode_table= PianaGlobals.proteinSpecies_table,
                            proteinCodeID_col= PianaGlobals.speciesNCBI_col,
                            proteinCodeID_value = taxonomy_value)

              will retrieve all those proteinPiana of table proteinSpecies_table where column speciesNCBI_col is taxonomy_value
    """
    # TO CHECK!!! I am not sure this will work for string and int values!!! Do I need to put the "" around %s???
    #             (I think integers do not mind to be between quotes)


    def get_sqlquery(proteinCode_table=None, proteinCodeID_col= None, proteinCodeID_value=None ):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        
        if proteinCodeID_value is None:
            raise ValueError("code value needed to retrieve proteins from table")
        
        sqlquery = """select %s from %s where %s="%s" """ %(PianaGlobals.proteinPiana_col,
                                                         proteinCode_table,
                                                         proteinCodeID_col,
                                                         proteinCodeID_value)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)


#---------------------------------------------------------------------
# PROTEIN ATTRIBUTES RELATIONSHIP RELATED CLASSES
#---------------------------------------------------------------------

class SelectProteinAttributeRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about protein attributes relationship tables
    
    the tables must follow the structure:
    
    (currently for tables:
    
    proteinCog_table 
    proteinEC_table 
    proteinSpecies_table 
    proteinDescription_table 
    proteinKeyword_table 
    proteinFunction_table 
    proteinSubcellularLocation_table
    )
    """
    def get_sqlquery( attribute_relationship_table, attribute_relationship_col, proteinPiana_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s from %s where %s = %s """ \
        %(attribute_relationship_col, attribute_relationship_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectAllDBAliClusterRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about dbali cluster information
    """
    def get_sqlquery( clustering_method_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        
        sqlquery = """SELECT %s, %s from %s where %s="%s" """ \
        %(PianaGlobals.proteinPiana_col, PianaGlobals.DBAliclusterID_col,
          PianaGlobals.proteinDBAliCluster_table,
          PianaGlobals.DBAliclusteringMethod_col, clustering_method_value)

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectDBAliClusterRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about dbali cluster information
    """
    def get_sqlquery( proteinPiana_value, clustering_method_value, source_db_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error

        if source_db_value == "all":
            sqlquery = """SELECT %s from %s where %s = %s and %s="%s"  """ \
                       %(PianaGlobals.DBAliclusterID_col,
                         PianaGlobals.proteinDBAliCluster_table,
                         PianaGlobals.proteinPiana_col, proteinPiana_value,
                         PianaGlobals.DBAliclusteringMethod_col, clustering_method_value)
        else:
            # if a specific source_db has been chosen, fix it in the sql query
            sqlquery = """SELECT %s from %s where %s = %s and %s="%s" and %s="%s"  """ \
                       %(PianaGlobals.DBAliclusterID_col,
                         PianaGlobals.proteinDBAliCluster_table,
                         PianaGlobals.proteinPiana_col, proteinPiana_value,
                         PianaGlobals.DBAliclusteringMethod_col, clustering_method_value,
                         PianaGlobals.sourceDBID_col, source_db_value)
            
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectPairMethodDBAliClusterRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about dbali cluster information
    """
    def get_sqlquery( proteinPiana_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s, %s from %s where %s = %s  """ \
        %(PianaGlobals.DBAliclusterID_col, PianaGlobals.DBAliclusteringMethod_col,
          PianaGlobals.proteinDBAliCluster_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
     
    
class SelectProteinsWithScop(PianaSelectSQL):
    """
    Generates the sql statement to select proteins that have a particular scop information
    """
    def get_sqlquery( cf, sf, fa):

        # create the condition that the proteins to return have to respect by looking to the cf, sf, fa values
        condition_to_respect = ""
        
        if cf is not None:
            condition_to_respect += "%s=%s" %(PianaGlobals.cf_col, cf)

        if sf is not None:
            if condition_to_respect != "":
                # cf was not None and something was written to condition_to_respect: write the 'and'
                condition_to_respect += " and "
            condition_to_respect += "%s=%s" %(PianaGlobals.sf_col, sf)

        if fa is not None:
            if condition_to_respect != "":
                # cf was not None and something was written to condition_to_respect: write the 'and'
                condition_to_respect += " and "
            condition_to_respect += "%s=%s" %(PianaGlobals.fa_col, fa)

        # use the condition_to_respect to build the sql query that will return all proteinPianas that respect the arguments set by user
        sqlquery = """SELECT %s from %s where %s; """ \
        %(PianaGlobals.proteinPiana_col,
          PianaGlobals.proteinScop_table,
          condition_to_respect)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
   
    
class SelectProteinScopRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about scop information
    """
    def get_sqlquery( proteinPiana_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s,%s,%s from %s where %s = %s """ \
        %(PianaGlobals.cf_col, PianaGlobals.sf_col, PianaGlobals.fa_col,
          PianaGlobals.proteinScop_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class SelectProteinScopCfRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about scop fold information
    """
    def get_sqlquery( proteinPiana_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s from %s where %s = %s """ \
        %(PianaGlobals.cf_col,
          PianaGlobals.proteinScop_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class SelectProteinScopSfRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about scop superfamily information
    """
    def get_sqlquery( proteinPiana_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s from %s where %s = %s """ \
        %(PianaGlobals.sf_col,
          PianaGlobals.proteinScop_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class SelectProteinScopFaRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about scop family information
    """
    def get_sqlquery( proteinPiana_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s from %s where %s = %s """ \
        %(PianaGlobals.fa_col,
          PianaGlobals.proteinScop_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectProteinResiduesCathRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about cath information
    """
    def get_sqlquery( proteinPiana_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s, %s, %s from %s where %s = %s """ \
        %(PianaGlobals.cathID_col, PianaGlobals.res_start_col, PianaGlobals.res_end_col,
          PianaGlobals.proteinCath_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectProteinCathRelationship(PianaSelectSQL):
    """
    Generates the sql statement to select information about cath information
    """
    def get_sqlquery( proteinPiana_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s from %s where %s = %s """ \
        %(PianaGlobals.cathID_col,
          PianaGlobals.proteinCath_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectSpeciesNCBIfromSpeciesName(PianaSelectSQL):
    """
    Generates the sql statement to select the speciesNCBI value for a given speciesName speciesName_value
    """
    def get_sqlquery(speciesName_value= None ):

        if speciesName_value is None:
            raise ValueError("speciesName cannot be None")
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s from %s where %s = "%s" """ \
        %(PianaGlobals.speciesNCBI_col, PianaGlobals.species_table,
          PianaGlobals.speciesName_col, speciesName_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectSpeciesNamefromSpeciesNCBI(PianaSelectSQL):
    """
    Generates the sql statement to select the speciesName value for a given speciesNCBI_value
    """
    def get_sqlquery(speciesNCBI_value= None ):

        if speciesNCBI_value is None:
            raise ValueError("speciesNCBI cannot be None")
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s from %s where %s = %s """ \
        %(PianaGlobals.speciesName_col, PianaGlobals.species_table,
          PianaGlobals.speciesNCBI_col, speciesNCBI_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectProteinKingdom(PianaSelectSQL):
    """
    Generates the sql statement to select the kingdom value for a given proteinPiana
    """
    def get_sqlquery(proteinPiana_value= None ):
        
        sqlquery = """SELECT s.%s from %s s, %s ps where ps.%s=%s and ps.%s=s.%s """ \
        %(PianaGlobals.speciesKingdom_col,
          PianaGlobals.species_table, PianaGlobals.proteinSpecies_table,
          PianaGlobals.proteinPiana_col, proteinPiana_value,
          PianaGlobals.speciesNCBI_col,PianaGlobals.speciesNCBI_col )
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

#---------------------------------------------------------------------
# PROTEIN EXTERNAL DATABASES RELATED CLASSES
#---------------------------------------------------------------------
class SelectFitnessScore(PianaSelectSQL):
    """
    Generates the SQL statement that obtains the 

    After getting the new proteinPiana identifier, the method calling this class must make sure that the counter in increased by 1
    """
    def get_sqlquery(proteinPiana_value):
        
        sqlquery = """select %s from %s where %s=%s """ %(PianaGlobals.fitnessScore_col, PianaGlobals.cellFitness_table, PianaGlobals.proteinPiana_col,
                                                         proteinPiana_value )
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class SelectFitnessReaction(PianaSelectSQL):
    """
    Generates the SQL statement that obtains the 

    After getting the new proteinPiana identifier, the method calling this class must make sure that the counter in increased by 1
    """
    def get_sqlquery(proteinPiana_value):
        
        sqlquery = """select %s from %s where %s=%s""" %(PianaGlobals.reaction_col, PianaGlobals.cellFitness_table, PianaGlobals.proteinPiana_col,
                                                         proteinPiana_value )
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

    
class SelectFitnessConditions(PianaSelectSQL):
    """
    Generates the SQL statement that obtains the 

    After getting the new proteinPiana identifier, the method calling this class must make sure that the counter in increased by 1
    """
    def get_sqlquery(proteinPiana_value):
        
        sqlquery = """select %s from %s where %s=%s""" %(PianaGlobals.conditions_col, PianaGlobals.cellFitness_table, PianaGlobals.proteinPiana_col,
                                                         proteinPiana_value )
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectFitnessScoreReactionConditions(PianaSelectSQL):
    """
    Generates the SQL statement that obtains the 

    After getting the new proteinPiana identifier, the method calling this class must make sure that the counter in increased by 1
    """
    def get_sqlquery(proteinPiana_value):
        
        sqlquery = """select %s,%s,%s from %s where %s=%s""" %(PianaGlobals.fitnessScore_col, PianaGlobals.reaction_col, PianaGlobals.conditions_col,
                                                               PianaGlobals.cellFitness_table, PianaGlobals.proteinPiana_col, proteinPiana_value )
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)


#---------------------------------------------------------------------
# INTERACTION RELATED CLASSES
#---------------------------------------------------------------------

def build_sql_db_restriction(list_source_dbs= None, inverse_dbs="no"):
    """
    function used to build a restriction from a list of databases

    this restriction will limit the sql query to retrieve only those interactionPiana that belong to specific databases
    """

    # depending on whether the user wants interactions from a set of dbs, or all interactions except for those in a set of dbs, the
    # sign and logic to use when building the restriction is different
    if inverse_dbs == "yes":
        sign_to_use = "!="
        logic_to_use = "and"
    else:
        sign_to_use = "="
        logic_to_use = "or"
    
    dbs_restriction = "("
    
    if not list_source_dbs or not isinstance(list_source_dbs, list):
        raise ValueError("Doesn't make sense to restrict interactions to those from list %s source database" %(list_source_dbs) )
    else:

        # build sql restriction to get only interactions from the list of databases in list_source_dbs
        #   -> to build restriction, start with first database and add the others afterwards
        dbs_restriction += """ s.sourceDBID%s"%s" """ %(sign_to_use, list_source_dbs[0])

        if len(list_source_dbs) > 1:
            for database in list_source_dbs[1:]:
                dbs_restriction += """ %s s.sourceDBID%s"%s" """ %(logic_to_use, sign_to_use, database)
            # END OF for database in list_source_dbs[1:]:
        # END OF if len(list_source_dbs) >1:
    # END OF else: (if not list_source_dbs:)

    dbs_restriction += ")"

    return dbs_restriction

def build_sql_method_restriction(list_source_methods= None, inverse_methods="no"):
    """
    function used to build a restriction from a list of methods

    this restriction will limit the sql query to retrieve only those interactionPiana that where found by specific methods
    """

    # depending on whether the user wants interactions from a set of methods, or all interactions except for those in a set of methods, the
    # sign and logic to use when building the restriction is different
    if inverse_methods == "yes":
        sign_to_use = "!="
        logic_to_use = "and"
    else:
        sign_to_use = "="
        logic_to_use = "or"
        
    methods_restriction = "("
    if not list_source_methods or not isinstance(list_source_methods, list):
        raise ValueError("Doesn't make sense to restrict interactions to those from list %s source methods" %list_source_methods)
    else:

        # build sql restriction to get only interactions from the list of methods in list_source_methods
        #   -> to build restriction, start with first database and add the others afterwards
        methods_restriction += """ m.methodID%s"%s" """ %(sign_to_use, list_source_methods[0])

        if len(list_source_methods) > 1:
            for method in list_source_methods[1:]:
                methods_restriction += """ %s m.methodID%s"%s" """ %(logic_to_use, sign_to_use, method)
            # END OF for method in list_source_methods[1:]:
        # END OF if len(list_source_methods) >1:
    # END OF else: (if not list_source_methods:)

    methods_restriction += ")"

    return methods_restriction

class SelectListInteractionPiana(PianaSelectSQL):
    """
    Generates the SQL statement that obtains interaction internal  ids where a given protein is involved
    """
    def get_sqlquery(proteinPiana_value, list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):

        # instead of generalizing with one single query, speed up things by separating queries for all databases and those
        # that need to take into account the source database for the interaction

        if verbose:
            sys.stderr.write("in PianaSelectSQL: list source dbs is %s and list source methods is %s\n" %(list_source_dbs,list_source_methods))

        if list_source_dbs == "all" and list_source_methods == "all":
            # ----------------------------------------------
            # no restrictions to apply: get all interactions
            # ----------------------------------------------
            sqlquery = """select %s from %s where %s=%s or %s=%s""" %(PianaGlobals.interactionPiana_col, PianaGlobals.interaction_table
                                                                      , PianaGlobals.proteinPianaA_col, proteinPiana_value
                                                                      , PianaGlobals.proteinPianaB_col, proteinPiana_value)

        else:
            # ------------------------------------
            # there are restrictions to be applied
            # ------------------------------------

            #set them to "no": will be changed depending on which list is not empty
            apply_db_restriction= "no"
            apply_method_restriction= "no"

            # build restrictions for lists that have elements in them
            if list_source_dbs != "all" and list_source_dbs:
                db_restriction = build_sql_db_restriction(list_source_dbs= list_source_dbs,
                                                          inverse_dbs=inverse_dbs)
                apply_db_restriction = "yes"
                
            if list_source_methods != "all" and list_source_methods:
                method_restriction = build_sql_method_restriction(list_source_methods,
                                                                  inverse_methods=inverse_methods)
                apply_method_restriction = "yes"


            if apply_db_restriction == "yes" and apply_method_restriction == "yes":

                #
                # SQL query when there are source db and method restrictions
                #
                sqlquery = """select i.%s from %s i, %s s, %s m where (i.%s=%s or i.%s=%s) and ( (s.%s=i.%s and m.%s=i.%s) """ %(
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.interactionMethod_table,
                    PianaGlobals.proteinPianaA_col,
                    proteinPiana_value,
                    PianaGlobals.proteinPianaB_col,
                    proteinPiana_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col )


                sqlquery += "and " + db_restriction
                sqlquery += "and " + method_restriction
                
            elif  apply_db_restriction == "yes":
                
                #
                # SQL query when there is source db restriction
                #
                sqlquery = """select i.%s from %s i, %s s where (i.%s=%s or i.%s=%s) and ( (s.%s=i.%s) """ %(
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.proteinPianaA_col,
                    proteinPiana_value,
                    PianaGlobals.proteinPianaB_col,
                    proteinPiana_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col)


                sqlquery += "and " + db_restriction
                
            elif  apply_method_restriction == "yes":
                
                #
                # SQL query when there is source method restriction
                #
                sqlquery = """select i.%s from %s i, %s m where (i.%s=%s or i.%s=%s) and ( (m.%s=i.%s) """ %(
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionMethod_table,
                    PianaGlobals.proteinPianaA_col,
                    proteinPiana_value,
                    PianaGlobals.proteinPianaB_col,
                    proteinPiana_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col)
                
                sqlquery += "and " + method_restriction
                
            # END OF elif  apply_method_restriction == "yes": (if apply_db_restriction == "yes" and apply_method_restriction == "yes":)
            else:
                raise ValueError("Trying to apply restrictions with empty lists of source dbs (%s) and methods (%s)" %(list_source_dbs, list_source_methods)) 

            sqlquery += ")"
            
        # END OF else (if list_source_dbs == "all" and list_source_methods == "all":)

        if verbose:
            sys.stderr.write("SelectListInteractionPiana returning query \n%s\n" %sqlquery)
 
        return sqlquery
     
    get_sqlquery = staticmethod(get_sqlquery)

class SelectInteractionPiana(PianaSelectSQL):
    """
    Generates the SQL statement that obtains interaction internal id interactionPiana for the interaction between two specific proteins

    Before looking for the interaction, it makes sure that the order proteinPianaA < proteinPianaB is respected. If it is not the case, it
    changes the query (not the objects passed, they remain unchanged) to execute it in the right order
    """
    def get_sqlquery(proteinPianaA_value, proteinPianaB_value, list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):
 
        if proteinPianaA_value > proteinPianaB_value:
            temp = proteinPianaA_value
            proteinPianaA_value = proteinPianaB_value
            proteinPianaB_value = temp

        if list_source_dbs == "all" and list_source_methods == "all":
            # ---------------------------------------------------------------------------------
            # no restrictions to apply: get interactionPiana regardless of db and method source
            # ---------------------------------------------------------------------------------
            sqlquery = """select %s from %s where (%s=%s) and (%s=%s)""" %(PianaGlobals.interactionPiana_col, PianaGlobals.interaction_table
                                                                           , PianaGlobals.proteinPianaA_col, proteinPianaA_value
                                                                           , PianaGlobals.proteinPianaB_col, proteinPianaB_value)
        else:
            # ------------------------------------
            # there are restrictions to be applied
            # ------------------------------------

            #set them to "no": will be changed depending on which list is not empty
            apply_db_restriction= "no"
            apply_method_restriction= "no"

            # build restrictions for lists that have elements in them
            if list_source_dbs != "all" and list_source_dbs:
                db_restriction = build_sql_db_restriction(list_source_dbs,
                                                          inverse_dbs=inverse_dbs)
                apply_db_restriction = "yes"
                
            if list_source_methods != "all" and list_source_methods:
                method_restriction = build_sql_method_restriction(list_source_methods,
                                                                  inverse_methods=inverse_methods)
                apply_method_restriction = "yes"

            if apply_db_restriction == "yes" and apply_method_restriction == "yes":

                #
                # SQL query when there are source db and method restrictions
                #
                sqlquery = """select i.%s from %s i, %s s, %s m where (i.%s=%s and i.%s=%s) and ( (s.%s=i.%s and m.%s=i.%s) """ %(
                   PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.interactionMethod_table,
                    PianaGlobals.proteinPianaA_col,
                    proteinPianaA_value,
                    PianaGlobals.proteinPianaB_col,
                    proteinPianaB_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col  )


                sqlquery += "and " + db_restriction
                sqlquery += "and " + method_restriction
                
            elif  apply_db_restriction == "yes":
                
                #
                # SQL query when there is source db restriction
                #
                sqlquery = """select i.%s from %s i, %s s where (i.%s=%s and i.%s=%s) and ( (s.%s=i.%s) """ %(
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.proteinPianaA_col,
                    proteinPianaA_value,
                    PianaGlobals.proteinPianaB_col,
                    proteinPianaB_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col)


                sqlquery += "and " + db_restriction
                
            elif  apply_method_restriction == "yes":
                
                #
                # SQL query when there is source method restriction
                #
                sqlquery = """select i.%s from %s i, %s m where (i.%s=%s and i.%s=%s) and ( (m.%s=i.%s) """ %(
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionMethod_table,
                    PianaGlobals.proteinPianaA_col,
                    proteinPianaA_value,
                    PianaGlobals.proteinPianaB_col,
                    proteinPianaB_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col)
                
                sqlquery += "and " + method_restriction
                
            # END OF elif  apply_method_restriction == "yes": (if apply_db_restriction == "yes" and apply_method_restriction == "yes":)
            else:
                raise ValueError("Trying to apply restrictions with empty lists of source dbs (%s) and methods (%s)" %(list_source_dbs, list_source_methods)) 

            sqlquery += ")"
            
        # END OF else: (if list_source_dbs == "all" and list_source_methods == "all":)
        
        return sqlquery
     
    get_sqlquery = staticmethod(get_sqlquery)

class SelectAllProteinProteinInteractions(PianaSelectSQL):
    """
    Generates the SQL statement that obtains all triplets (proteinPianaA, proteinPianaB, interactionPiana) for list_source_dbs and list_source_methods
    """
    def get_sqlquery(list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):

        if list_source_dbs == "all" and list_source_methods == "all":
            # ---------------------------------------------------------------------------------
            # no restrictions to apply: get interactionPiana regardless of db and method source
            # ---------------------------------------------------------------------------------
            sqlquery = """select %s, %s, %s from %s""" %(PianaGlobals.proteinPianaA_col, PianaGlobals.proteinPianaB_col, PianaGlobals.interactionPiana_col,
                                                         PianaGlobals.interaction_table)
 
        else:
            # ------------------------------------
            # there are restrictions to be applied
            # ------------------------------------

            #set them to "no": will be changed depending on which list is not empty
            apply_db_restriction= "no"
            apply_method_restriction= "no"

            # build restrictions for lists that have elements in them
            if list_source_dbs != "all" and list_source_dbs:
                db_restriction = build_sql_db_restriction(list_source_dbs,
                                                          inverse_dbs=inverse_dbs)
                apply_db_restriction = "yes"
                
            if list_source_methods != "all" and list_source_methods:
                method_restriction = build_sql_method_restriction(list_source_methods,
                                                                  inverse_methods=inverse_methods)
                apply_method_restriction = "yes"

            if apply_db_restriction == "yes" and apply_method_restriction == "yes":

                #
                # SQL query when there are source db and method restrictions
                #
                sqlquery = """select i.%s, i.%s, i.%s from %s i, %s s, %s m where  ( (s.%s=i.%s and m.%s=i.%s) """ %(
                    PianaGlobals.proteinPianaA_col,
                    PianaGlobals.proteinPianaB_col,
                   PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.interactionMethod_table,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col  )


                sqlquery += "and " + db_restriction
                sqlquery += "and " + method_restriction
                
            elif  apply_db_restriction == "yes":
                
                #
                # SQL query when there is source db restriction
                #
                sqlquery = """select i.%s, i.%s, i.%s  from %s i, %s s where ( (s.%s=i.%s) """ %(
                    PianaGlobals.proteinPianaA_col,
                    PianaGlobals.proteinPianaB_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col)


                sqlquery += "and " + db_restriction
                
            elif  apply_method_restriction == "yes":
                
                #
                # SQL query when there is source method restriction
                #
                sqlquery = """select i.%s, i.%s, i.%s  from %s i, %s m where ( (m.%s=i.%s) """ %(
                    PianaGlobals.proteinPianaA_col,
                    PianaGlobals.proteinPianaB_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionMethod_table,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col)
                
                sqlquery += "and " + method_restriction
                
            # END OF elif  apply_method_restriction == "yes": (if apply_db_restriction == "yes" and apply_method_restriction == "yes":)
            else:
                raise ValueError("Trying to apply restrictions with empty lists of source dbs (%s) and methods (%s)" %(list_source_dbs, list_source_methods)) 

            sqlquery += ")"
            
        # END OF else: (if list_source_dbs == "all" and list_source_methods == "all":)
        
        return sqlquery
     
    get_sqlquery = staticmethod(get_sqlquery)

class SelectPartners(PianaSelectSQL):
    """
    Generates the SQL statement that obtains interaction partners of a given protein


        Used to return the SQL query that retrieves the interaction partners of a protein (only those on side A or B)
        
        This class is not to be used externally: the side (A or B)
        where the protein is placed should be trasparent to the
        external user. Piana makes this trasparent to the user by
        creating a method (get_all_partners) under PianaDBaccess that
        performs two queries to obtain all partners.  PianaDBaccess.get_all_partners
        will call twice select_db_content with SelectPartners
        

        partner_side is either "A" or "B", indicating if the partner will be searched on side A or B.
        this is done this way because there is no single sqlquery that will return a complete list
        of partners of a given protein, given the fact that those partners can be on side A or B

        this is an artifact of the database design: an interaction is formed by a sideA and a sideB.
 
    """
    def get_sqlquery(proteinPiana_value, partner_side, list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):

        if partner_side == "A":

            partner_column = PianaGlobals.proteinPianaA_col
            this_protein_column = PianaGlobals.proteinPianaB_col
            
        elif partner_side == "B":

            partner_column = PianaGlobals.proteinPianaB_col
            this_protein_column = PianaGlobals.proteinPianaA_col
        else:
            raise ValueError("partner_side incorrect: it must be either A or B")
        
        if list_source_dbs == "all" and list_source_methods == "all":

            # -------------------------------------------------------------------------
            # no restrictions to apply: get partners regardless of db and method source
            # -------------------------------------------------------------------------
            
            sqlquery = """select %s from %s where %s = %s""" %(partner_column, PianaGlobals.interaction_table
                                                               ,this_protein_column , proteinPiana_value)
            
        else:

            # ------------------------------------
            # there are restrictions to be applied
            # ------------------------------------

            #set them to "no": will be changed depending on which list is not empty
            apply_db_restriction= "no"
            apply_method_restriction= "no"

            # build restrictions for lists that have elements in them
            if list_source_dbs != "all" and list_source_dbs:
                db_restriction = build_sql_db_restriction(list_source_dbs,
                                                          inverse_dbs=inverse_dbs)
                apply_db_restriction = "yes"
                
            if list_source_methods != "all" and list_source_methods:
                method_restriction = build_sql_method_restriction(list_source_methods,
                                                                  inverse_methods=inverse_methods)
                apply_method_restriction = "yes"

            if apply_db_restriction == "yes" and apply_method_restriction == "yes":

                #
                # SQL query when there are source db and method restrictions
                #
   
                sqlquery = """ select i.%s from %s i, %s s, %s m where (i.%s = %s) and ( (s.%s=i.%s and m.%s=i.%s) """ %(
                    partner_column,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.interactionMethod_table,
                    this_protein_column,
                    proteinPiana_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col  )


                sqlquery += "and " + db_restriction
                sqlquery += "and " + method_restriction

                      
            elif  apply_db_restriction == "yes":
                
                #
                # SQL query when there is source db restriction
                #
   
                sqlquery = """ select i.%s from %s i, %s s where (i.%s = %s) and ( (s.%s=i.%s) """ %(
                    partner_column,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionSourceDB_table,
                    this_protein_column,
                    proteinPiana_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col )


                sqlquery += "and " + db_restriction
          
                  
            elif  apply_method_restriction == "yes":
                
                #
                # SQL query when there is source method restriction
                #
   
                sqlquery = """ select i.%s from %s i, %s m where (i.%s = %s) and ( (m.%s=i.%s) """ %(
                    partner_column,
                    PianaGlobals.interaction_table,
                    PianaGlobals.interactionMethod_table,
                    this_protein_column,
                    proteinPiana_value,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col )


                sqlquery += "and " + method_restriction
                 
            # END OF elif  apply_method_restriction == "yes": (if apply_db_restriction == "yes" and apply_method_restriction == "yes":)
            else:
                raise ValueError("Trying to apply restrictions with empty lists of source dbs (%s) and methods (%s)" %(list_source_dbs, list_source_methods)) 

            sqlquery += ")"
            
        # END OF else: (if list_source_dbs == "all" and list_source_methods == "all":)

        if verbose:
            sys.stderr.write("SelectPartners returning query \n%s\n" %sqlquery)
             
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectPartnersProteinSharingCog(PianaSelectSQL):
    """
    Generates the SQL statement that obtains interaction partners for proteins that share cog with a certain proteinPiana

        This class is not to be used externally: the side (A or B)
        where the protein is placed should be trasparent to the
        external user. Piana makes this trasparent to the user by
        creating a method (get_partners_of_proteins_sharing_cog) under PianaDBaccess that
        performs two queries to obtain all partners.  PianaDBaccess.get_partners_of_proteins_sharing_cog
        will call twice select_db_content with SelectPartnersProteinSharingCog
        

        partner_side is either "A" or "B", indicating if the partner will be searched on side A or B.
        this is done this way because there is no single sqlquery that will return a complete list
        of partners of a given protein, given the fact that those partners can be on side A or B

        this is an artifact of the database design: an interaction is formed by a sideA and a sideB.

    """
    def get_sqlquery(proteinPiana_value, partner_side, list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):

        if partner_side == "A":

            partner_column = PianaGlobals.proteinPianaA_col
            this_protein_column = PianaGlobals.proteinPianaB_col
            
        elif partner_side == "B":

            partner_column = PianaGlobals.proteinPianaB_col
            this_protein_column = PianaGlobals.proteinPianaA_col
        else:
            raise ValueError("partner_side incorrect: it must be either A or B")

        if list_source_dbs == "all" and list_source_methods == "all":

            # -------------------------------------------------------------------------
            # no restrictions to apply: get partners regardless of db and method source
            # -------------------------------------------------------------------------
            
            sqlquery = """select i.%s from %s i, %s pc1, %s pc2 where pc1.%s = %s and pc1.%s = pc2.%s and pc2.%s = i.%s""" %(
                partner_column,
                PianaGlobals.interaction_table,
                PianaGlobals.proteinCog_table,
                PianaGlobals.proteinCog_table,
                PianaGlobals.proteinPiana_col,
                proteinPiana_value,
                PianaGlobals.cogID_col,
                PianaGlobals.cogID_col,
                PianaGlobals.proteinPiana_col,
                this_protein_column )
            
        else:
            # ------------------------------------
            # there are restrictions to be applied
            # ------------------------------------

            #set them to "no": will be changed depending on which list is not empty
            apply_db_restriction= "no"
            apply_method_restriction= "no"

            # build restrictions for lists that have elements in them
            if list_source_dbs != "all" and list_source_dbs:
                db_restriction = build_sql_db_restriction(list_source_dbs,
                                                          inverse_dbs=inverse_dbs)
                apply_db_restriction = "yes"
                
            if list_source_methods != "all" and list_source_methods:
                method_restriction = build_sql_method_restriction(list_source_methods,
                                                                  inverse_methods=inverse_methods)
                apply_method_restriction = "yes"

            if apply_db_restriction == "yes" and apply_method_restriction == "yes":

                #
                # SQL query when there are source db and method restrictions
                #

                sqlquery = """select i.%s from %s i, %s pc1, %s pc2, %s s, %s m where (pc1.%s = %s and pc1.%s = pc2.%s and pc2.%s = i.%s) \
                and ( (s.%s=i.%s and m.%s=i.%s)""" %(
                    partner_column,
                    PianaGlobals.interaction_table,
                    PianaGlobals.proteinCog_table,
                    PianaGlobals.proteinCog_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.interactionMethod_table,
                    PianaGlobals.proteinPiana_col,
                    proteinPiana_value,
                    PianaGlobals.cogID_col,
                    PianaGlobals.cogID_col,
                    PianaGlobals.proteinPiana_col,
                    this_protein_column,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col  )

                sqlquery += "and " + db_restriction
                sqlquery += "and " + method_restriction
                      
            elif  apply_db_restriction == "yes":
                
                #
                # SQL query when there is source db restriction
                #

                sqlquery = """select i.%s from %s i, %s pc1, %s pc2, %s s where (pc1.%s = %s and pc1.%s = pc2.%s and pc2.%s = i.%s) \
                and ( (s.%s=i.%s)""" %(
                    partner_column,
                    PianaGlobals.interaction_table,
                    PianaGlobals.proteinCog_table,
                    PianaGlobals.proteinCog_table,
                    PianaGlobals.interactionSourceDB_table,
                    PianaGlobals.proteinPiana_col,
                    proteinPiana_value,
                    PianaGlobals.cogID_col,
                    PianaGlobals.cogID_col,
                    PianaGlobals.proteinPiana_col,
                    this_protein_column,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col )


                sqlquery += "and " + db_restriction
                  
            elif  apply_method_restriction == "yes":
                
                #
                # SQL query when there is source method restriction
                #

                sqlquery = """select i.%s from %s i, %s pc1, %s pc2, %s m where (pc1.%s = %s and pc1.%s = pc2.%s and pc2.%s = i.%s) \
                and ( (m.%s=i.%s)""" %(
                    partner_column,
                    PianaGlobals.interaction_table,
                    PianaGlobals.proteinCog_table,
                    PianaGlobals.proteinCog_table,
                    PianaGlobals.interactionMethod_table,
                    PianaGlobals.proteinPiana_col,
                    proteinPiana_value,
                    PianaGlobals.cogID_col,
                    PianaGlobals.cogID_col,
                    PianaGlobals.proteinPiana_col,
                    this_protein_column,
                    PianaGlobals.interactionPiana_col,
                    PianaGlobals.interactionPiana_col  )


                sqlquery += "and " + method_restriction

            # END OF elif  apply_method_restriction == "yes": (if apply_db_restriction == "yes" and apply_method_restriction == "yes":)
            else:
                raise ValueError("Trying to apply restrictions with empty lists of source dbs (%s) and methods (%s)" %(list_source_dbs, list_source_methods)) 

            sqlquery += ")"
            
        # END OF else: (if list_source_dbs == "all" and list_source_methods == "all":)

        if verbose:
            sys.stderr.write("SelectPartnersProteinSharingCog returning query \n%s\n" %sqlquery)
           
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectInteractionPartner(PianaSelectSQL):
    """
    Generates the SQL statement that obtains the interaction partner of a protein for a given interactionID
    """

    def get_sqlquery(proteinPiana_value, interactionPiana_value, partner_side):

        if partner_side == "A":

            partner_column = PianaGlobals.proteinPianaA_col
            this_protein_column = PianaGlobals.proteinPianaB_col
            
        elif partner_side == "B":

            partner_column = PianaGlobals.proteinPianaB_col
            this_protein_column = PianaGlobals.proteinPianaA_col
        else:
            raise ValueError("partner_side incorrect: it must be either A or B")

        
        sqlquery = """select %s from %s where (%s=%s) and (%s=%s)""" \
                   %(partner_column, PianaGlobals.interaction_table
                     ,this_protein_column , proteinPiana_value, PianaGlobals.interactionPiana_col, interactionPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectInteractionSourceDB(PianaSelectSQL):
    """
    Generates the SQL statement that obtains entries from table interactionSourceDB

    if interactionPiana_value not None --> get the list of sourceDBID for this interactionPiana

    if sourceDBID_value not None --> get the list of interactionPiana for this sourceDBID

    Only one of the arguments can be "not None"
    """

    def get_sqlquery(interactionPiana_value=None, sourceDBID_value=None):

        if interactionPiana_value is not None and sourceDBID_value is not None:
            raise ValueError("in SelectInteractionSourceDB, both arguments cannot be 'not None'")
        
        elif interactionPiana_value is not None:
            sqlquery = """select %s from %s where (%s=%s)""" %(PianaGlobals.sourceDBID_col, PianaGlobals.interactionSourceDB_table
                                                               , PianaGlobals.interactionPiana_col, interactionPiana_value)
        elif sourceDBID_value is not None:
            sqlquery = """select %s from %s where (%s="%s")""" %(PianaGlobals.interactionPiana_col, PianaGlobals.interactionSourceDB_table
                                                               , PianaGlobals.sourceDBID_col, sourceDBID_value)
        else:
            raise ValueError("in SelectInteractionSourceDB, both arguments cannot be 'None' Why here????")
            

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectInteractionMethod(PianaSelectSQL):
    """
    Generates the SQL statement that obtains a list  of methodID of an interactionPiana

    if sourceDB_value == None, then it returns the SQL statement that obtains all methodIDs for interactionPiana
    if sourceDB_value is a database, then it returns the SQL statement that obtains all methodIDs for interactionPiana for a particular sourceDB

    Attention: list_source_dbs and inverse_dbs not being used!!! 
    """

    # TO DO!!! Remove sourceDBID and get methods for all dbs in list_source_dbs (taking into account inverse_dbs)
    def get_sqlquery(interactionPiana_value=None, sourceDBID_value=None, list_source_dbs= "all", inverse_dbs="no"):

        if interactionPiana_value is not None and sourceDBID_value is not None:
            # return methods for a given interaction in a particular database

 
                
            sqlquery = """select %s from %s where (%s=%s) and (%s="%s") """ %(PianaGlobals.methodID_col, PianaGlobals.interactionMethod_table,
                                                                              PianaGlobals.interactionPiana_col, interactionPiana_value ,
                                                                              PianaGlobals.sourceDBID_col, sourceDBID_value )

        elif interactionPiana_value is not None:
            # return methods for a given interaction, regardless of the database
            sqlquery = """select %s from %s where (%s=%s)""" %(PianaGlobals.methodID_col, PianaGlobals.interactionMethod_table
                                                               , PianaGlobals.interactionPiana_col, interactionPiana_value)

        else:
            raise ValueError("in SelectInteractionMethod, interactionPiana_value cannot be 'None' ")
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
  
class SelectXFromGoTermId(PianaSelectSQL):
    """
    Generates the sql statement to select something (proteinPiana_col or go_acc_col)
    from one of the two go tables (proteinGo_table or go_table) from a
    given GO term_id
    """
    def get_sqlquery( x_col, term_id_value):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """SELECT %s from %s where %s = %s """ \
        %(x_col, PianaGlobals.go_table,
          PianaGlobals.goID_col, term_id_value)
      
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class SelectGoTermIdUsingTermType(PianaSelectSQL):
    """
    Generates the sql statement to select go term id associated to a proteinPiana specifing the term type value
    because one proteinPiana could be associated to more than one Go term id depending the term type:

          - "molecular_function"
          - "biological_process"
          - "cellular_component"
    """
    def get_sqlquery( proteinPiana,term_type):

        sqlquery= """SELECT protGo.%s from  %s protGo, %s go \
        where protGo.%s = %s  and protGo.%s = go.%s and go.%s = "%s" """ \
        %(PianaGlobals.goID_col, PianaGlobals.proteinGo_table, PianaGlobals.go_table,
          PianaGlobals.proteinPiana_col, proteinPiana,
          PianaGlobals.goID_col,PianaGlobals.goID_col,
          PianaGlobals.go_term_type_col, term_type)

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class Select_go_term2term_distance(PianaSelectSQL):
   """
   Generates the sql statement to select the distance between two go term ids
   """
   def get_sqlquery(term1_id, term2_id):
       
       if term1_id is None or term2_id is None:
           raise ValueError("Either term1(%s) or term2(%s) are None" %(term1_id, term2_id))
       
       if term1_id > term2_id:
           temp = term1_id
           term1_id = term2_id
           term2_id = temp
           
       sqlquery="""SELECT %s from %s where %s = %s and %s = %s """\
                 %(PianaGlobals.go_term2term_distance_col, PianaGlobals.go_term2term_distance_table,
                   PianaGlobals.go_term2term_distance_term1_col, term1_id, PianaGlobals.go_term2term_distance_term2_col, term2_id)

       return sqlquery

   get_sqlquery = staticmethod(get_sqlquery)
        

