"""
 File       : train_and_test_cir.py
 Author     : R. Aragues
 Creation   : 13.02.2006
 Contents   : based on files produced by command train-cir, does a cross validation of results
 Called from: 

=======================================================================================================

This script evaluates the CIR method for both protein decomposition
(--task=protein_cir) and CIR-CIR interactions (--task=cir_ints).

Different n-folds can be used with flag --n-fold=n_fold

Input dir must contain results from PIANA command train-cirs. it should not contain any other results, since
this script will use all results files in there regardless of when they were generated.


See --help for details on which PIANA database to use for this script

"""
# train_and_test_cir.py: based on files produced by command train-cir, does a cross validation of results
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues"

import sys
import os
import getopt

import re
import readline
from sets import *

from plplot import *

import random

import utilities

from PianaDBaccess import *

verbose = 1
verbose_detailed = 0

# ----------------------
# Function usage()
# ----------------------
def usage():
    print "\n--------------------------------------------------------------------------------------------------------------"
    print "Based on files produced by command train-cir, does a cross validation of results\n"
    print "\nUsage: python train_and_test_cir.py  --piana-dbname=piana_dbname --piana-dbhost=piana_dbhost "
    print "                                       --piana-dbuser=piana_dbuser --piana-dbpass=piana_dbpass"
    print "                                       --task=task --input-dir=input_dir --class-type=class_type --n-fold=n_fold"
    print "                                       --remove-redundancy=remove_redundancy"
    print "                                       [--help] [--verbose]"
    print "\nwhere:"
    print "     piana_dbname : name of database piana to be used"
    print "                    -> this db must contain the same proteinPianas as the db used to generate files in input-dir"
    print "                       Attention! But it should not be the same db, because for those results we didn't use all interactions, just experimental y2h"
    print "                    -> db must contain domain info for proteins"
    print "                    -> db must contain interaction predictions based on domain ints"
    print "     piana_dbhost : name of host where database piana to be used is placed"
    print "     piana_dbuser : username accessing the database"
    print "     piana_dbpass : password of username accessing the database"
    print "     task         : write here which task you wish to evaluate"
    print "                    valid values are:"
    print "                       - protein_cir: to evaluate the method on the protein decomposition task (ie. predicting CIRs of a protein)"
    print "                       - cir_ints: to evaluate the method on the CIR-CIR interaction task (ie. predicting CIR-CIR interactions)"
    print "     input_dir    : directory with the results from command train-cir"
    print "     class_type   : type of domain categorization that is going to be used for evaluating proteins' CIRs "
    print "                    and to get interaction predictions based on domain interactions (these ints are used to evaluate the CIR-CIR ints)"
    print "                    valid values are scop_fa, ..."
    print "     n_fold       : N of the N-fold cross validation you are performing"
    print "     remove_redundancy : determines whether all proteins are used for evaluating the method or redundant proteins are removed "
    print "                         (redundant proteins are those that appear in table proteinSimilarity)"
    print "                         - yes: only one representant of each group of 'redundant' proteins is used for evaluation "
    print "                         - no: all proteins are used "
    print "     --help       : prints this message and exits"
    print "     --verbose    : prints process info to stdout"
    print "--------------------------------------------------------------------------------------------------------------"
        

   
# ---------------------------
# Function parseArguments()                                               
# --------------------------- 

def parseArguments():
    
    global piana_dbname
    global piana_dbhost
    global piana_dbuser
    global piana_dbpass

    global input_dir
    global class_type
    global n_fold
    global task
    global remove_redundancy
    
    global verbose
    
    try:
        opts, args = getopt.getopt(sys.argv[1:], "", ["verbose","help", "piana-dbname=", "piana-dbhost=", "piana-dbuser=", "piana-dbpass=",
                                                      "input-dir=", "class-type=", "n-fold=", "task=" , "remove-redundancy=" ])
    except getopt.GetoptError, bad_opt:
        # print help information and exit:
        sys.stderr.write( bad_opt.__str__() )
        usage()
        sys.exit(2)
     
    for option,value in opts:
        
        if option == "--piana-dbname":
            piana_dbname = value
            
        elif option == "--piana-dbhost":
            piana_dbhost = value
            
        elif option == "--piana-dbuser":
            piana_dbuser = value
             
        elif option == "--piana-dbpass":
            piana_dbpass = value
             
        elif option == "--input-dir":
            input_dir =  value
             
        elif option == "--class-type":
            class_type =  value
             
        elif option == "--n-fold":
            n_fold =  int(value)
             
        elif option == "--task":
            task = value
             
        elif option == "--remove-redundancy":
            remove_redundancy = value
             
        elif option == "--verbose":
            verbose = 1
            
        elif option == "--help":
            # print help information and exit
            usage()
            sys.exit(2)
    # END OF for option,value in opts:

    if class_type is None or input_dir is None or n_fold is None or remove_redundancy is None:
        raise ValueError("n_fold, class_type and input_dir and remove_redundancy arguments are mandatory\n")

    if task != "protein_cir" and task != "cir_ints":
        raise ValueError("Invalid values for task: check valid values using flag --help\n")
        
        
            
def get_list_proteins_sharing_class(protein, class_type, piana_access):
    """
    returns a list of proteins that have the same class "class_type" as protein
       (eg. list of proteins that have a scop that also appears in protein)

    "class_type" determines which class type is going to be used
       - valid class_type values are scop_fa, ...
    """

    if class_type == "scop_fa":
        list_classes = piana_access.get_proteins_sharing_scop(proteinPiana_value= protein)

    return list_classes

def get_list_partners_according_to_class(protein, class_type, piana_access):
    """
    returns a list of proteins that interact with "protein" according to a prediction method based on "class_type"
    ( eg. from scop-scop interactions we can predict protein-protein interactions. These predictions are tested
          versus the interactions inferred from the method CIR-CIR interactions)

    "class_type" determines which class type is going to be used
       - valid class_type values are scop_fa, ...
    """

    if class_type == "scop_fa":
        list_partners = piana_access.get_all_partners(proteinPiana_value= protein,
                                                      list_source_dbs= "all", inverse_dbs="no", list_source_methods=["scop_fa_pred"], inverse_methods="no",
                                                      threshold = 0)

    return list_partners

    
def calculate_tp_fp_fn(predicted_set_of_proteins, real_set_of_proteins, remove_redundancy, piana_access):
    """
    given two sets of proteins calculates tp, fp and fn for prediction vs reality


    "remove_redundancy" is used to avoid using redundant proteins for calculating stats
      -> 'yes': doesn't use redundant proteins
      -> 'no': uses all the proteins

    returns a list [number of true positives, number of false positives, number of false negatives]

    """

    if remove_redundancy == "no":
        # use all proteins for calculating the stats
        number_of_tp = len( predicted_set_of_proteins.intersection(real_set_of_proteins) )
        number_of_fp = len( predicted_set_of_proteins.difference(real_set_of_proteins)   )
        number_of_fn = len( real_set_of_proteins.difference(predicted_set_of_proteins)   )

    elif remove_redundancy == "yes":
        # we must avoid counting more than once the 'same' protein (ie proteins that appear in table proteinSimilarity)
    

        # transform the original sets to sets of proteins where we make sure there are no redundancies
        #  -> moreover, the sets are 'synchronized' so that the representants used in both sets are
        #     the same
        (predicted_set_of_representants, real_set_of_representants) = utilities.get_non_redundant_sets(piana_access=piana_access,
                                                                                                       set_1=predicted_set_of_proteins,
                                                                                                       set_2=real_set_of_proteins)
        # And now, use these representants sets for calculating the stats
        number_of_tp = len( predicted_set_of_representants.intersection(real_set_of_representants) )
        number_of_fp = len( predicted_set_of_representants.difference(real_set_of_representants)   )
        number_of_fn = len( real_set_of_representants.difference(predicted_set_of_representants)   )
        

    else:
        raise ValueError("valid remove_redundancy values are 'yes' and 'no'\n")
        


    return [number_of_tp, number_of_fp, number_of_fn]


def get_tp_fp_fn_from_list_results(dic_results, score_threshold):
    """
    returns the total tp, fp, fn from a list of results given a score threshold

    "dic_results" is a dictionary that follows the format described below for dic_all_results
                  --> usually, dic_results will be a subset of dic_all_results, ie the test set extracted from dic_all_results

    "score_threshold" is the stop condition being applied for this evaluation
        (ie. when the clustering reaches a score below this threshold, clustered graph is returned)

    returns a tuple (tp, fp, fn) where tp, fp and fn are totals for evaluating proteins in the dic_results
    """

    total_tps = 0
    total_fps = 0
    total_fns = 0

    for protein in dic_results:
        # for each protein, perform evaluation
        
        #   --> dic_results[protein] is a list [ (score, {"tp":X; "fp":Y, "fn":Z}), (score, {"tp":X; "fp":Y, "fn":Z}), (...) ]
        #                              -> the list has been created in the same order that the clustering followed to create the clusters
        
        for one_tuple in dic_results[protein]:
            # we do an ordered search on the list, to make sure we follow the same order in which the clusters were created
            if one_tuple[0] < score_threshold:
                total_tps += one_tuple[1]["tp"]
                total_fps += one_tuple[1]["fp"]
                total_fns += one_tuple[1]["fn"]
                break # break the loop: we already got the result for this protein
            
        # END OF for one_tuple in dic_results[protein]:
    # END OF for protein in dic_results:

    return (total_tps, total_fps, total_fns)

def calculate_spec_sens_fval(tp, fp, fn):
    """
    calculates specificity, sensibility and f value from tp, fp, fn

    if 0/0, returns a string "unknown" as value

    returns a tuple (spec, sens, fval)
    """

    if tp or fp:     this_spec = tp / float(tp + fp)
    else:            this_spec = "unknown" # this is not a 0... it is a 0/0, which means nothing was found but no mistakes were made...

    if tp or fn:     this_sens = tp / float(tp + fn)
    else:            this_sens = "unknown" # this is not a 0... it is a 0/0, which means nothing was found but there was nothing to be found

    if this_spec!="unknown" and this_sens!="unknown":
        if this_spec == 0 and this_sens == 0:
            this_fval = 0
        else:
            this_fval = 2*this_spec*this_sens/(this_spec + this_sens)
    else:   this_fval = "unknown" # this is not a 0... it is a 0/0, which means no stats avalaible for this case

    return (this_spec, this_sens, this_fval)
    

def get_threshold_from_list_results(dic_results, max_score):
    """
    finds which are the score thresholds that maximize spec, sens and fval for proteins in dic_results
    
    returns a tuple (max_spec_threshold, max_sens_threshold, max_fvalue_threshold) 

    "dic_results" is a dictionary that follows the format described below for dic_all_results
                  --> usually, dic_results will be a subset of dic_all_results, ie the test set extracted from dic_all_results


    "max_score" is the high score on which to start the training (all scores between max_score and 0 will be tested for training

    """
    # init stats
    pair_max_spec_score = [0.0, None]  # 2 elements list: [max spec obtained, which score threshold was used to get it]
    pair_max_sens_score = [0.0, None]  # 2 elements list: [max sens obtained, which score threshold was used to get it]
    pair_max_fval_score = [0.0, None]  # 2 elements list: [max fval obtained, which score threshold was used to get it]

    # this is complicated to calculate... because we cannot just look at the scores for each protein, since the stats depend as well
    # on which were the scores in the previous clustering levels... therefore, I can only think of one way to test this: starting from
    # the highest score in all the proteins/results, test tp, fp, fn for all scores below that (using get_tp_fp_fn_from_list_results).
    # then, keep the score threshold that performed best... Very costly in time, but I think it is the only way to do it correctly...


    for score_to_test in range(max_score, 0, -1):

        if verbose_detailed:
            sys.stderr.write("-----------------\n")
            sys.stderr.write("testing score %s\n" %(score_to_test))
            sys.stderr.write("previous maxs are %s, %s, %s\n" %(pair_max_spec_score[0], pair_max_sens_score[0], pair_max_fval_score[0]))
            
        # score_to_test will take values from max_score to 1
        (tp, fp, fn) = get_tp_fp_fn_from_list_results(dic_results=dic_results, score_threshold=score_to_test)
        (this_spec, this_sens, this_fval) = calculate_spec_sens_fval(tp=tp, fp=fp, fn=fn)
            
        if verbose_detailed:   sys.stderr.write("results are: spec=%s, sens=%s, fval=%s\n" %(this_spec, this_sens, this_fval ))
        
        if this_spec!="unknown" and this_spec >= pair_max_spec_score[0]:
            if verbose_detailed:  sys.stderr.write("updating max_spec\n")
            pair_max_spec_score = [this_spec, score_to_test]
            
        if this_sens!="unknown" and this_sens >= pair_max_sens_score[0]:
            if verbose_detailed:  sys.stderr.write("updating max_sens\n")
            pair_max_sens_score = [this_sens, score_to_test]
            
        if this_fval!="unknown" and this_fval >= pair_max_fval_score[0]:
            if verbose_detailed:  sys.stderr.write("updating max_fval\n")
            pair_max_fval_score = [this_fval, score_to_test]
            
        if verbose_detailed:   sys.stderr.write("-----------------\n")
    # END OF for score_to_test in range(max_score, 0, -1):

    # return which were the score thresholds that obtained the best spec, sens and fval
    return (pair_max_spec_score[1], pair_max_sens_score[1], pair_max_fval_score[1])


def print_dic_with_results(dic, output_target):
    """
    prints to output_target file object in a human readable way a dictionary that follows format described below for dic_all_results
    """

    for protein in dic:
        output_target.write("-----protein %s-----\n" %(protein))
        for pair in dic[protein]:
            output_target.write(" --> score %s: tp=%s fp=%s fn=%s\n" %(pair[0], pair[1]["tp"], pair[1]["fp"], pair[1]["fn"]))
        

# ----
# MAIN
# ----

input_dir= None
class_type= None
n_fold= None
task= None

remove_redundancy = None

piana_dbname = None
piana_dbuser = None
piana_dbhost = None
piana_dbpass = None

# parsing arguments from the command line
parseArguments()

# initialisating connection to piana
piana_access = PianaDBaccess(dbname=piana_dbname, dbhost=piana_dbhost, dbuser=piana_dbuser, dbpassword= piana_dbpass)


current_root_protein = None
current_score = None
max_score = 0 # used to know which is the max score that has been obtained among all proteins
              #  -> we need to know this max score to then be able to test all scores when training

dic_all_results = {}    # this is a dictionary where the keys are the root proteins values for each key are an ordered list of (scores, stats)
                        #
                        #        { root_protein: [ (score1, { "tp":X; "fp":Y, "fn":Z},
                        #                          (score2, { "tp":X; "fp":Y, "fn":Z},
                        #                          ..................................
                        #                        ],
                        #          root_protein: [ (score1, {.........}),
                        #                          ..................
                        #                        ],
                        #          ....................................
                        #        }


dic_all_roots = {} # just to keep track of all the roots processed



"""
A. Read each results file in input_dir and load information (scores, tps, fp, ... for each protein) in dic_all_results
"""
for file_in_directory in utilities.GlobDirectoryWalker(input_dir , "*.results"):

    file_fd = file(file_in_directory, "r")

    # 0. initialize sets
    set_proteins_sharing_cir_with_root = Set([])
    set_proteins_interacting_with_root = Set([])

    # 1. get all data from this file
    
    for line in file_fd:

        if line.strip() == "":
            continue

        line_fields = line.split()

        if line_fields[0] == "description":
            # line describing the parameters for this results file
            current_root_protein = int(line_fields[1].split("=")[1])
            current_score = int(float(line_fields[3].split("=")[1]))  # don't know why, but python doesn't let me do int(xxx.y)

            if current_score > max_score:
                # keep this current score as maximum...
                max_score = current_score
                
            dic_all_roots[current_root_protein] = None # just keeping track of all the roots processed

        elif line_fields[0] == "share":
            # line describing results for proteins that have the same CIR
            protein_1 = int(line_fields[2])
            protein_2 = int(line_fields[3])

            if protein_1 == current_root_protein:
                set_proteins_sharing_cir_with_root.add(protein_2)
                
            if protein_2 == current_root_protein:
                set_proteins_sharing_cir_with_root.add(protein_1)
                
        elif line_fields[0] == "int":
            # line describing results for proteins whose CIRs interact
            protein_1 = int(line_fields[2])
            protein_2 = int(line_fields[3])

            if protein_1 == current_root_protein:
                set_proteins_interacting_with_root.add(protein_2)
                
            if protein_2 == current_root_protein:
                set_proteins_interacting_with_root.add(protein_1)
                
        # END OF elif line_fields[0] == "int":
            
            
    # END OF for line in file_fd:

    # 2. now that the file has been read completely, calculate TPs, FPs, FNs and write them to the dictionary with all results
    #
    # --> depending on the task that was set on the command line, evaluate CIR decomposition or CIR-CIR interactions

    if task == "protein_cir":
        # get 'reality' for root protein domains
        set_proteins_sharing_class_with_root = Set(get_list_proteins_sharing_class(protein=current_root_protein,
                                                                                   class_type=class_type,
                                                                                   piana_access=piana_access))

        # and compare that reality with our prediction
        (tp, fp, fn) = calculate_tp_fp_fn(predicted_set_of_proteins= set_proteins_sharing_cir_with_root,
                                          real_set_of_proteins= set_proteins_sharing_class_with_root,
                                          remove_redundancy= remove_redundancy,
                                          piana_access=piana_access)
    elif task == "cir_ints":
        # get 'reality' for root protein interactions
        set_partners_root = Set(get_list_partners_according_to_class(protein=current_root_protein,
                                                                     class_type=class_type,
                                                                     piana_access=piana_access))
        # and compare that reality with our prediction
        (tp, fp, fn) = calculate_tp_fp_fn(predicted_set_of_proteins= set_proteins_interacting_with_root,
                                          real_set_of_proteins= set_partners_root,
                                          remove_redundancy= remove_redundancy,
                                          piana_access=piana_access)
    # END OF elif task == "cir_ints":

    # if there are stats, keep them in the dictionary with all results
    #
    # Attention!!! This tp, fp, fn correspond to the clustering results obtained when current_score was the first score under the threshold fixed
    #              
    if tp != 0 or fp!=0 or fn != 0:

        if dic_all_results.has_key(current_root_protein):
            # append the new entry to the list for current_protein 
            dic_all_results[current_root_protein].append( (current_score, {"tp":tp, "fp":fp, "fn":fn}) )

        else:
            # nothing was previously read for this protein: create a list with one entry with the current score and tp, fp, fn
            dic_all_results[current_root_protein] = [ (current_score, {"tp":tp, "fp":fp, "fn":fn}) ]
    # END OF if tp != 0 or fp!=0 or fn != 0:

    file_fd.close()
    current_protein = None
    current_score = None
# END OF for file_in_directory in utilities.GlobDirectoryWalker(input_dir , "*.*"):

if verbose:
    print_dic_with_results(dic= dic_all_results, output_target= sys.stderr)
    sys.stderr.write( "number of root proteins: %s -- number of proteins with stats %s\n" %(len(dic_all_roots), len(dic_all_results)))


"""
B. Use dic_all_results to do a cross-validation
"""
# at this point, we've got in the dictionary the results for all the proteins that have associated cir result files
# --------------------------------------------------------------------------------------------------------
# now, do the cross validation
#  0. create subgroups of proteins based on the N-fold chosen by user
#  1. loop N times (ie. number of cross validation == number of different subgroups)
#     2. train of N-1 groups
#     3. use the previous training to evaluate on the Nth group
#  4. average and std deviation of the N rounds
# --------------------------------------------------------------------------------------------------------

# ---
#  0. create subgroups of proteins based on the N-fold chosen by user
# ---
all_proteins = dic_all_results.keys()

number_of_proteins = len(all_proteins)

subsets_dic = {} # subsets_dic is a dictionary that contains a list of proteins for each subset
                 # follows structure:
                 #                     { 0: dic following structure of dic_all_results,
                 #                       1: dic following structure of dic_all_results,
                 #                       .....................,
                 #                       n_fold-1: dic following structure of dic_all_results
                 #                     }


if n_fold == 0:
    # if user chose 0 as n_fold, he meant to do a leave-one-out (ie. each set is just one file)
    n_fold = number_of_proteins

# initialize subsets to empty dics
for i in range(n_fold):   subsets_dic[i] = {}

# Create random groups of equal size (each 
end = 0
random.seed()
while not end:
    for i in range(n_fold):
        try:
            protein_to_insert = random.choice(all_proteins)
            subsets_dic[i][protein_to_insert] = dic_all_results[protein_to_insert]
            all_proteins.remove(protein_to_insert)
        except:
            # when there are no more items an exception will be raised and we will exit the loop: all items have been assigned to a group
            end = 1
            break
    # END OF for i in range(n_fold):
# while not end:


if verbose_detailed:
    sys.stderr.write("\nSizes of subsets are: \n")
    for i in range(n_fold):
        sys.stderr.write("subset %s of lenght %s-" %(i, len(subsets_dic[i])))
    sys.stderr.write("\n--------------\n" + str(subsets_dic) + "\n--------------\n")
    
# ---
#  1. loop N times (ie. number of cross validation == number of different subgroups)
# ---

all_values_max_spec_has_spec = []# keeps all values found for specificity when maximizing specificity (used to calculate std deviation)
total_max_spec_has_spec = [0,0]  # [0] is total added specificity when maximizing specificity
                                 # [1] is number of tests for which this stat could be calculated
                                 # [0] / [1] is the average specificity of the cross validation when maximizing specificity
            
all_values_max_spec_has_sens = []# keeps all values found for sensitivity when maximizing specificity (used to calculate std deviation)                  
total_max_spec_has_sens = [0, 0] # [0] is total added sensitivity when maximizing specificity
                                 # [1] is number of tests for which this stat could be calculated
                                 # [0] / [1] is the average sensitivity of the cross validation when maximizing specificity
                                 
all_values_max_spec_has_fval = []# keeps all values found for f value when maximizing specificity (used to calculate std deviation)  
total_max_spec_has_fval = [0, 0] # [0] is total added fvalue when maximizing specificity
                                 # [1] is number of tests for which this stat could be calculated
                                 # [0] / [1] is the average fvalue of the cross validation when maximizing specificity 

all_values_max_sens_has_spec = []# see comments above...
total_max_sens_has_spec = [0, 0] # see comments above...
all_values_max_sens_has_sens = []# see comments above...
total_max_sens_has_sens = [0, 0] # see comments above...
all_values_max_sens_has_fval = []# see comments above...
total_max_sens_has_fval = [0, 0] # see comments above...

all_values_max_fval_has_spec = []# see comments above...
total_max_fval_has_spec = [0, 0] # see comments above...
all_values_max_fval_has_sens = []# see comments above...
total_max_fval_has_sens = [0, 0] # see comments above...
all_values_max_fval_has_fval = []# see comments above...
total_max_fval_has_fval = [0, 0] # see comments above...


for test_number in subsets_dic:
    # this is the loop in charge of running a test for each subset (using the other subsets for training)

    test_dic = subsets_dic[test_number]
    train_dic = {}
    for train_number in subsets_dic:
        if train_number != test_number:
            train_dic.update(subsets_dic[train_number])
    # END OF for train_number in subsets_dic:

    
    (thres_spec, thres_sens, thres_fval) = get_threshold_from_list_results(dic_results= train_dic, max_score= max_score)

    if verbose:
        sys.stderr.write("For subset %s (proteins=%s), optimal trained thresholds are: %s, %s, %s\n" %(test_number, subsets_dic[test_number].keys(),
                                                                                                       thres_spec, thres_sens, thres_fval))

    (spec_tp, spec_fp, spec_fn) = get_tp_fp_fn_from_list_results(dic_results= test_dic, score_threshold= thres_spec)
    (sens_tp, sens_fp, sens_fn) = get_tp_fp_fn_from_list_results(dic_results= test_dic, score_threshold= thres_sens)
    (fval_tp, fval_fp, fval_fn) = get_tp_fp_fn_from_list_results(dic_results= test_dic, score_threshold= thres_fval)

    (max_spec_has_spec, max_spec_has_sens, max_spec_has_fval) = calculate_spec_sens_fval(tp=spec_tp, fp=spec_fp, fn=spec_fn)
    (max_sens_has_spec, max_sens_has_sens, max_sens_has_fval) = calculate_spec_sens_fval(tp=sens_tp, fp=sens_fp, fn=sens_fn)
    (max_fval_has_spec, max_fval_has_sens, max_fval_has_fval) = calculate_spec_sens_fval(tp=fval_tp, fp=fval_fp, fn=fval_fn)


    if max_spec_has_spec != "unknown":
        total_max_spec_has_spec[0] += max_spec_has_spec
        all_values_max_spec_has_spec.append(max_spec_has_spec)
        total_max_spec_has_spec[1] += 1
        
    if max_spec_has_sens != "unknown":
        total_max_spec_has_sens[0] += max_spec_has_sens
        all_values_max_spec_has_sens.append(max_spec_has_sens)
        total_max_spec_has_sens[1] += 1
        
    if max_spec_has_fval != "unknown":
        total_max_spec_has_fval[0] += max_spec_has_fval
        all_values_max_spec_has_fval.append(max_spec_has_fval)
        total_max_spec_has_fval[1] += 1
        
    if max_sens_has_spec != "unknown":
        total_max_sens_has_spec[0] += max_sens_has_spec
        all_values_max_sens_has_spec.append(max_sens_has_spec)
        total_max_sens_has_spec[1] += 1
        
    if max_sens_has_sens != "unknown":
        total_max_sens_has_sens[0] += max_sens_has_sens
        all_values_max_sens_has_sens.append(max_sens_has_sens)
        total_max_sens_has_sens[1] += 1
        
    if max_sens_has_fval != "unknown":
        total_max_sens_has_fval[0] += max_sens_has_fval
        all_values_max_sens_has_fval.append(max_sens_has_fval)
        total_max_sens_has_fval[1] += 1
        
    if max_fval_has_spec != "unknown":
        total_max_fval_has_spec[0] += max_fval_has_spec
        all_values_max_fval_has_spec.append(max_fval_has_spec)
        total_max_fval_has_spec[1] += 1
        
    if max_fval_has_sens != "unknown":
        total_max_fval_has_sens[0] += max_fval_has_sens
        all_values_max_fval_has_sens.append(max_fval_has_sens)
        total_max_fval_has_sens[1] += 1
        
    if max_fval_has_fval != "unknown":
        total_max_fval_has_fval[0] += max_fval_has_fval
        all_values_max_fval_has_fval.append(max_fval_has_fval)
        total_max_fval_has_fval[1] += 1

    
    if verbose:
        sys.stderr.write("----\n")
        sys.stderr.write("test %s spec=%s, sens=%s, fval=%s (max_spec)\n" %(test_number, max_spec_has_spec, max_spec_has_sens, max_spec_has_fval))
        sys.stderr.write("test %s spec=%s, sens=%s, fval=%s (max_sens)\n" %(test_number, max_sens_has_spec, max_sens_has_sens, max_sens_has_fval))
        sys.stderr.write("test %s spec=%s, sens=%s, fval=%s (max_fval)\n" %(test_number, max_fval_has_spec, max_fval_has_sens, max_fval_has_fval))
        sys.stderr.write("----\n")
# END OF for test_number in subsets_dic:

sys.stdout.write("  ----------------------------------------------------\n")
sys.stdout.write("     MAXIMIZING SPECIFICITY task=%s \n" %task)
sys.stdout.write("  ----------------------------------------------------\n")
try:    total_max_spec_has_av_spec = total_max_spec_has_spec[0]/float(total_max_spec_has_spec[1])
except: total_max_spec_has_av_spec = "unknown"
try:    total_max_spec_has_av_sens = total_max_spec_has_sens[0]/float(total_max_spec_has_sens[1])
except: total_max_spec_has_av_sens = "unknown"
try:    total_max_spec_has_av_fval = total_max_spec_has_fval[0]/float(total_max_spec_has_fval[1])
except: total_max_spec_has_av_fval = "unknown"
sys.stdout.write("  task=%s average_specificity=%s (std_dev=%s)\n" %(task, total_max_spec_has_av_spec,
                                                                     utilities.calculate_std_deviation( all_values= all_values_max_spec_has_spec )))
sys.stdout.write("  task=%s average_sensitivity=%s (std_dev=%s)\n" %(task, total_max_spec_has_av_sens ,
                                                                     utilities.calculate_std_deviation( all_values= all_values_max_spec_has_sens)))
sys.stdout.write("  task=%s average_fvalue=%s (std_dev=%s)\n\n" %(task,total_max_spec_has_av_fval ,
                                                                  utilities.calculate_std_deviation( all_values= all_values_max_spec_has_fval)))
sys.stdout.write("  ---------------------------------------------------\n")
sys.stdout.write("     MAXIMIZING SHARED SENSITIVITY task=%s \n" %task)
sys.stdout.write("  ---------------------------------------------------\n")
try:    total_max_sens_has_av_spec = total_max_sens_has_spec[0]/float(total_max_sens_has_spec[1])
except: total_max_sens_has_av_spec = "unknown"
try:    total_max_sens_has_av_sens = total_max_sens_has_sens[0]/float(total_max_sens_has_sens[1])
except: total_max_sens_has_av_sens = "unknown"
try:    total_max_sens_has_av_fval = total_max_sens_has_fval[0]/float(total_max_sens_has_fval[1])
except: total_max_sens_has_av_fval = "unknown"
sys.stdout.write("  task=%s average_specificity=%s (std_dev=%s)\n" %(task, total_max_sens_has_av_spec,
                                                                     utilities.calculate_std_deviation( all_values= all_values_max_sens_has_spec )))
sys.stdout.write("  task=%s average_sensitivity=%s (std_dev=%s)\n" %(task, total_max_sens_has_av_sens ,
                                                                     utilities.calculate_std_deviation( all_values= all_values_max_sens_has_sens)))
sys.stdout.write("  task=%s average_fvalue=%s (std_dev=%s)\n\n" %(task,total_max_sens_has_av_fval ,
                                                                  utilities.calculate_std_deviation( all_values= all_values_max_sens_has_fval)))
sys.stdout.write("  -----------------------------------------------\n")
sys.stdout.write("     MAXIMIZING SHARED F VALUE task=%s \n" %task)
sys.stdout.write("  -----------------------------------------------\n")
try:    total_max_fval_has_av_spec = total_max_fval_has_spec[0]/float(total_max_fval_has_spec[1])
except: total_max_fval_has_av_spec = "unknown"
try:    total_max_fval_has_av_sens = total_max_fval_has_sens[0]/float(total_max_fval_has_sens[1])
except: total_max_fval_has_av_sens = "unknown"
try:    total_max_fval_has_av_fval = total_max_fval_has_fval[0]/float(total_max_fval_has_fval[1])
except: total_max_fval_has_av_fval = "unknown"
sys.stdout.write("  task=%s average_specificity=%s (std_dev=%s)\n" %(task, total_max_fval_has_av_spec,
                                                                     utilities.calculate_std_deviation( all_values= all_values_max_fval_has_spec )))
sys.stdout.write("  task=%s average_sensitivity=%s (std_dev=%s)\n" %(task, total_max_fval_has_av_sens ,
                                                                     utilities.calculate_std_deviation( all_values= all_values_max_fval_has_sens)))
sys.stdout.write("  task=%s average_fvalue=%s (std_dev=%s)\n\n" %(task,total_max_fval_has_av_fval ,
                                                                  utilities.calculate_std_deviation( all_values= all_values_max_fval_has_fval)))
