"""
File        : Graph.py
Author      : Daniel Jaeggi and Ramon Aragues
Creation    : 8.2003
Contents    : main class providing graph processing methods
Called from : programs/classes that need to use graphs

=======================================================================================================

This file implements class Graph, which has methods for graph creation, analysis and manipulation.

The easiest way to create a Graph would be:

from Graph import *
graph = Graph("1")
node1 = graph.get_node(1)
node2= graph.get_node(2)
edge1 = graph.get_edge(node1, node2)
graph.add_node(node1)
graph.add_node(node2)
graph.add_edge(edge1)

If you want to use a database with edges (eg. interactions) to build
your graph, then you need to use a GraphBuilder. If this is the case,
you'll need to implement method GraphBuilder.get_links(node). This
method gets all the links for a node, and it is problem specific since
the links are differently retrieved depending on the problem being faced.

For example, look at PianaGraph, a subclass of Graph that handles
protein-protein interaction networks (piana_access is just an
interface to the DB, but it is not required if you are implementing your own type of Graph):

"""

# Graph.py: implements a class for managing graphs
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

import sys
import numarray     
import LinearAlgebra

import piana_configuration_parameters

if piana_configuration_parameters.piana_mode == "developer" or piana_configuration_parameters.piana_mode == "advanced_user":
    import fgraph
    
from Filter import *
from Expansion import *
from EdgeToPropagate import *
from GraphNode import GraphNode
from GraphEdge import GraphEdge
from GraphNodeAttribute import *
from GraphEdgeAttribute import *
import PianaGlobals


verbose = 0
verbose_get_node = 0
verbose_get_edge = 0
verbose_expansion_minimal = 1
verbose_expansion_detailed = 0
verbose_expansion_shallow = 0
verbose_join = 0
verbose_propagate = 0
verbose_propagate_print= 0
verbose_add_edge_shallow= 0
verbose_add_edge_detailed= 0

root_nodes ={}  # keeps record of which are the root nodes of the network (i.e. the source nodes that generated the network)


class Graph(object):
    """
    The main graph class. Implements all methods related to creation and analysis of general-purpose graphs

    Other Graphs (eg. PianaGraph) will be subclasses of this class.
    """
    def __init__(self, graphID = None):
        """
        Optionally, give a graphID to your graph.
        """

        # graph attributes and components
	self.graphID = graphID
        
        self.nodes_dic = {}  # dictionary with key=node_id and content the GraphNode
        self.edges_dic = {}  # dictionary with key=edge_id and content the GraphEdge

        # graph characteristics
        self.adjacency = None
        self.connectivity = None
        self.degree = None
        self.ave_degree = None
        self.laplacian = None
        self.clus_coef = None

        # graph attributes related to filtering
        self.has_hidden = 0
        self.hidden_nodes = []
        self.unhidden_nodes = []
        self.hidden_edges = []
        self.unhidden_edges = []

        self.list_root_node_ids = []
        
        self.applied_filters = None
        self.saved_filters = None
        
        # graph attributes related to expanding
        
        self.applied_expansions = []
        self.saved_expansions = None


    def __str__(self):
        string = "<Graph: Graph with graphID " + str(self.graphID) 
	string += ", containing " + str(len(self.nodes_dic.keys()))
        string += " nodes and " + str(len(self.edges_dic.keys())) + " edges>"
        return string

    def build_graph(self, user_graph_builder):
        
        user_graph_builder.set_graph(graph_object= self)

        user_graph_builder.build_graph()
        
    def get_graph_id(self):
        
        return self.graphID
        
    def set_graph_id(self, new_graph_id):
        
        self.graphID = new_graph_id

    #------------------------------------------
    # Graph Methods: joining Graphs
    #------------------------------------------

    def join_graphs(self, graph_object):
        """

        adds to current graph the nodes and edges from graph passed as argument "graph_object"
        
        """
        if verbose_join:
            i = 0
            j = 0
            number_of_nodes = len(graph_object.get_node_ids_list())
            number_of_edges = len(graph_object.get_edge_ids_list())
            
        # adding nodes (add_node makes sure of mergin attributes and not duplicating entries)
        for node_id in graph_object.get_node_ids_list():

            self.add_node(node_object= graph_object.nodes_dic[node_id])
            
            if verbose_join:
                sys.stderr.write("joining %s nodes of %s--"%(i, number_of_nodes))
                i += 1
        # END OF for node_id in graph_object.get_node_ids_list():
        if verbose_join:
                sys.stderr.write("end joining nodes\n")


        # adding edges (add_edge makes sure of mergin attributes and not duplicating entries)
        for edge_id in graph_object.get_edge_ids_list():

            self.add_edge(edge_object= graph_object.edges_dic[edge_id])
            
            if verbose_join:
                sys.stderr.write("joining %s edges of %s--" %(j, number_of_edges))
                j += 1
        # END OF for edge_id in graph_object.get_edge_ids_list():
        
        if verbose_join:
                sys.stderr.write("end joining edges\n" )

    #------------------------------------------
    # Graph Methods: loading a graph with data from files
    #------------------------------------------

    def load_graph_from_text_file(self, file_name):
        """

        loads graph data described in a text file into the current graph object

        input file must follow format:

        node--node
        node--node

        where node are identifiers of nodes and each line is a link between those identifiers
        
        
        """
        file_fd = file(file_name, "r")
        
        for line in file_fd:

            line_fields = line.split("--")

            node1_id = line_fields[0]
            node2_id = line_fields[1]

            new_node1= self.get_node(identifier= node1_id , get_mode="new")
            new_node2= self.get_node(identifier= node2_id , get_mode="new")
            self.add_node(node_object= new_node1)
            self.add_node(node_object= new_node2)
            
            new_edge = self.get_edge(identifier1= node1_id, identifier2=node2_id , get_mode = "new" )
            self.add_edge(edge_object= new_edge)
    #------------------------------------------
    # Graph Methods: Adding and removing nodes and edges
    #------------------------------------------
    
    def add_node(self, node_object):
        """
        Adds a node to the graph.

        "node_object" must be previously created from GraphNode
        
        """
        if not isinstance(node_object, GraphNode):
            raise TypeError("Must be passed a GraphNode object")
        
        node_object_id = node_object.get_node_id()

        if self.nodes_dic.has_key(node_object_id):
            # node existed previously: merge them (merge the nodes and their attributes)
            self.nodes_dic[node_object_id].merge_nodes(node_object= node_object)
            
        else:
            # node didn't exist previosly: create it and add it to graph
            self.nodes_dic[node_object_id] = node_object

        #not being used self._changed()

    def rm_node(self, node_object):
        """
        Removes a node from the graph.

        "node_object" is a  GraphNode object of the graph
        
        """
        if not isinstance(node_object, GraphNode):
            raise TypeError("Must be passed a GraphNode object")
        
        if not self.nodes_dic.has_key(node_object.get_node_id()):
            raise ValueError("Node %s not found in graph" %(node_object.get_node_id()))
        
        # before removing node, need to go through its edges and remove them
        # An edge which node has disappeared cannot be used by any other node or process
        for edge_id_to_remove in node_object.edge_ids_dic.keys():
            self.rm_edge(edge_id= edge_id_to_remove)
        
        del self.nodes_dic[node_object.get_node_id()]
        #not being used self._changed()


 
    def rm_edge(self, edge_id):
        """
        Removes a edge from the graph.

        "edge_object" is a  GraphEdge object of the graph
        """
        
        if not self.edges_dic.has_key(edge_id):
            raise ValueError("Edge %s not found in graph" %edge_id)
     

        del self.edges_dic[edge_id]
        # not being used self._changed()

    #------------------------------------------
    # Graph Methods: Getting nodes 
    #------------------------------------------

            

    def get_node(self, identifier, attribute = None, get_mode = "new"):
        """
        Returns a node object.

        "identifier" can be a nodeID or a GraphNode object.

        Attention: get_node doesn't add the node to the graph!!!
        
        If the node nodeID already exists in the graph, returns that node.
        Otherwise the behaviour is defined by parameter get_mode (see below)

        get_mode can be "new" or "error" (default is "new")
          - "new" returns a new node in case node.nodeID doesn't exist
          - "error" returns an error in case node.nodeID doesn't exist
      
        """
                
        node_id = None

        # If the parameter was an identifier, find the corresponding nodeID
        # and set the variable accordingly
        
        if isinstance(identifier, GraphNode):
            node_id = identifier.get_node_id()
        else:
            node_id = identifier

            
        if self.nodes_dic.has_key(node_id):
            # the node already exists in the graph

            # If attribute existing in the parameters, set it to be the
            # attribute of the node being returned
            if attribute != None:
                self.nodes_dic[node_id].set_attribute(attribute_object= attribute)
                
            # force a return of the existing node
            
            return self.nodes_dic[node_id]
        
        else:
            # the node does not exist in the graph
            
            if get_mode == "new":
                return GraphNode(nodeID= node_id, attribute= attribute)
            
            elif get_mode == "error":
                raise ValueError("Node %s not found\n" %(node_id))
            
            else:
                raise ValueError("argument get_mode set to incorrect value")
            
        
    def get_hidden_nodes(self):
        """
        Returns a list of hidden node objects in the graph.
        """
        # currently, when the user asks for hidden nodes, a new list is generated...
        # if in the future we manage this list so it is always updated, the following commented lines are important
        #
        #if not self.has_hidden:
        #    return []
        
        #elif self.hidden_nodes is not None:
        #   return self.hidden_nodes
        
        #else:
        self._gen_hidden_lists()

        return self.hidden_nodes
        

    def get_unhidden_nodes(self):
        """
        Returns a list of unhidden nodes objects in the graph.
        """
        # currently, when the user asks for unhidden nodes, a new list is generated...
        # if in the future we manage this list so it is always updated, the following commented lines are important
        #
        # if not self.has_hidden:
        #    self.unhidden_nodes = self.get_node_object_list()
        #    
        #elif self.unhidden_nodes is not None:
        #    pass
        #else:
        self._gen_hidden_lists()
            
        return self.unhidden_nodes


    # TO DO!!! Unify queries to get node deep copy with the others, just by adding an argument...

    def get_node_ids_list(self):
        """
        Returns a list with node ids of current graph
        """
        return self.nodes_dic.keys()
    
    def get_node_ids_list_deep_copy(self):
        """
        Returns a new list (a deep copy, not just a reference to node_ids) with node ids of current graph
        """
        return self.nodes_dic.keys()[:]

    def get_node_object_list(self):
        """
        Returns a list with node objects of the graph
        """
        return self.nodes_dic.values()
    
    def get_root_node_ids(self):
        """
        Returns a list of root node ids of the graph.
        """

        # TO DO!!! This is not a very efficient way of keeping a list of root nodes...
        
        self.list_root_node_ids = []
        
        for node in self.get_node_object_list():
            
            if node.is_root():
                self.list_root_node_ids.append( node.get_node_id() )
        # END OF for node in self.get_node_object_list():

        return self.list_root_node_ids
      
    def get_root_node_ids_deep_copy(self):
        """
        Returns a list of root nodes (a deep copy, not just a reference to node objects) of the graph.
        """

        # TO DO!!! This is not a very efficient way of keeping a list of root nodes...
        
        self.list_root_node_ids = []
        
        for node in self.get_node_object_list():
            
            if node.is_root():
                self.list_root_node_ids.append( node.get_node_id() )
        # END OF for node in self.get_node_object_list():

        list_to_return = self.list_root_node_ids[:]

        return list_to_return
      


    def get_node_root_connectivity(self, node):
        """
        returns the number of root nodes that node "node" connects
        """
        connectivity = 0
        node_neihgbour_ids = node.get_neighbour_ids()
        list_root_node_ids = self.get_root_node_ids()
        
        for root_node_id in list_root_node_ids:
            
            if root_node_id in node_neihgbour_ids:
                connectivity += 1
            # END OF if root_node_id in node_neihgbour_ids:
            
        # END OF for root_node_id in list_root_node_ids:

        return connectivity


    def get_nodes_connectivities(self, distance= 1):
        """
        returns a dictionary of connectivities of all nodes in the graph

        the dictionary returned follows format:
           {node_id 1: list of node ids at distance 1 of node id 1,
            node_id 2: list of node ids at distance 1 of node id 2,
            .............................    }

        "distance" is currently not being used

        """
        node_connectivities_dic = {}
        
        for node in self.get_node_object_list():
            node_neihgbour_ids = node.get_neighbour_ids()

            node_connectivities_dic[ node.get_node_id()] = node_neihgbour_ids

        # END OF  for node in self.get_node_object_list():

        return node_connectivities_dic
            

    def get_connecting_nodes_dic(self, distance = 1):
        """
        Returns a dictionary of node_ids that connect root_nodes of the graph
        format is: {connecting_node_id:[list of root nodes that connects], connecting_node_id:[], ...}

        These are the nodes that belong to the path between two root nodes, at a maximun distance "distance"

        Note: it doesn't return nodes that are only connected to one root node. The minimum number of root nodes is 2.
        """
        
        # TO DO!!!! Make use of distance... right now, only works for distance == 1

        connecting_node_ids = {} # dictionary with keys=connecting nodes and content=list of connected root nodes
        list_root_node_ids = self.get_root_node_ids()

        # for each node, create a list with 
        for node in self.get_node_object_list():
            node_neihgbour_ids = node.get_neighbour_ids()

            connected_root_partners = []

            for root_node_id in list_root_node_ids:
                
                if root_node_id in node_neihgbour_ids:
                    connected_root_partners.append(root_node_id)
                # END OF if root_node_id in node_neihgbour_ids:
                
            # END OF for root_node_id in list_root_node_ids:
            if len(connected_root_partners) > 1:
                connecting_node_ids[node.get_node_id()]= connected_root_partners
        # END OF for node in self.get_node_object_list():
            
        return connecting_node_ids

    #------------------------------------------
    # Graph Methods: Getting edges
    #------------------------------------------
    def get_node_node_links(self):
        """
        returns a list of node_id pairs that have an edge between them

        returned list looks like this: [ [node1, node2], [node1, node3], [node2,node4], ...]
        
        """
        list_node_node = []
        
        for edge in self.get_edge_object_list():
            list_node_node.append(edge.get_node_ids())

        return list_node_node
        
    def get_edge_object_list(self):
        """
        Returns a list with edge objects of the graph
        """
        return self.edges_dic.values()

    def get_edge_ids_list(self):
        """
        Returns a list with edge objects of the graph
        """
        return self.edges_dic.keys()

    
    def edge_exists(self, identifier1= None, identifier2= None):
        """
        Returns a 1 if edge already exists. 0 otherwise

        if identifier1 and identifier2 are not None, both identifiers are node_id
        if identifier2 is None, then identifier1 is an edge_id
        """
        if identifier2 is not None:
            return self.edges_dic.has_key( GraphEdge._gen_edge_edgeID(node_id1= identifier1, node_id2= identifier2) )
        else:
            return self.edges_dic.has_key( identifier1 )



    def get_edge(self, identifier1, identifier2= None, attribute_object= None, get_mode="new" ):
        """
        Returns an edge object.

        "identifier1" and "identifier2" can be edgeIDs,  edge objects GraphEdge, nodeIDs or nodes objects GraphNode.

        If the edge described in the arguments already exists in the graph, returns that edge object.
        
        Otherwise the behaviour is defined by parameter get_mode (see below)

        get_mode can be "new" or "error" (default is "new")
          - "new" - returns a new edge if it doesn't exist
          - "error" - raises error if it doesn't exist

        Both get_modes return a reference to the edge when it already existed

        Attention: get_edge does not add the edge to the Graph!!!!!
        """

        # TO DO!!!! isinstance for node_id and edge_id to do type comparisons depending on the type used as identifier 
        
        edge_id = None

        if identifier2 is not None:
            # two nodes were passed: check whether they are node identifiers or objects

            # TO DO!!! Improve type checking
            #          when isinstance will exist for node_id and edge_id this type comparisons have to be improved and errors raised
            
            if isinstance(identifier1, GraphNode) and isinstance(identifier2, GraphNode):
                node_id1= identifier1.get_node_id()
                node_id2= identifier2.get_node_id()
                edge_id = GraphEdge._gen_edge_edgeID(node_id1= node_id1, node_id2= node_id2)

            else:
                # two nodes identifiers have been passed: get (or create) the nodes
               
                node_id1= identifier1
                node_id2= identifier2
                edge_id = GraphEdge._gen_edge_edgeID(node_id1= node_id1, node_id2= node_id2)
            # END OF if isinstance(identifier1, GraphNode) and isinstance(identifier2, GraphNode): ... elif ( not is... : .... else:
                        
        elif isinstance(identifier1,GraphEdge):
            edge_id = identifier1.edgeID
            
        else:
            # if type was not previously identified, identifier1  has to be a edge_id
            edge_id = identifier1
            
        # END OF if identifier2 is not None: .... elif isinstance(identifier1,GraphEdge): .....  else: ...

        
        """
        at this point:

        ed_id is an edge identifier
        node_id1 is a node identifier
        node_id2 is a node identifier
        """


        if self.edge_exists(identifier1= edge_id ):
            # If the edge already exists in the graph, then all we have to do is set the attribute (in case it exists)

            if attribute_object is not None:                               
                self.edges_dic[edge_id].merge_attribute(attribute_object)
                
            # force a return of the existing edge (for both modes new and error)
            return self.edges_dic[edge_id]
        
        else:
        # if edge doesn't exist already in the graph, create a new one (doesn't add it to the graph... must be done by user)
        # (unless the get_mode was "error", which returns an error)
        
            if get_mode == "new":
                return GraphEdge(node1_id= node_id1, node2_id= node_id2,
                                 attribute_object= attribute_object)
            
            elif get_mode == "error":
                raise ValueError("Edge %s not found\n" %(edge_id))
            
            else:
                raise ValueError("argument get_mode set to incorrect value")
        # END OF else: (if self.edge_exists(identifier1= edge_id ):)
        
        
    def create_edge(self, node_id1= None, node_id2= None, attribute_object= None):
        """
        Returns a tuple [edge object, new], where new= 1 means edge didn't exist in the graph before

        If edge already existed, in edge returned attribute_object is merged with attribute of existing edge
        
        "node_id1" and "node_id2" are node identifiers

        Attention!!! The edge is not added to the graph! That must be done by the user if he wants to do so...
        """
        edge_id = GraphEdge._gen_edge_edgeID(node_id1= node_id1, node_id2= node_id2)
        
        if self.edge_exists(identifier1= edge_id):
            # If the edge already exists in the graph, then all we have to do is set the attribute

            if attribute_object is not None:                               
                self.edges_dic[edge_id].merge_attribute(new_attribute_object= attribute_object)
                
            # return the existing edge: the 0 indicates that the edge existed before the call
            return [self.edges_dic[edge_id], 0]
        
        else:
            # if edge doesn't exist already in the graph, create a new one (doesn't add it to the graph... must be done by user)
            # (unless the get_mode was "error", which returns an error)

            new_edge = GraphEdge(node1_id= node_id1,
                                 node2_id= node_id2,
                                 attribute_object= attribute_object)
            
            # return the new edge: the 1 indicates that the edge is a new one
            return [new_edge, 1]
        # END OF else (if self.edge_exists(identifier1=node_id1, identifier2= node_id2 ):)

       
    def add_edge(self, edge_object):
        """
        Adds an edge to the graph. Argument is a GraphEdge object.

        Attention! The nodes have to be added to the graph before adding the edge!!!!
          -> add_edge does not add the nodes!
        
        adds the edge to the graph after checking its existance. Therefore, it can be used wihtout checking if it was in the network
        """
        if not isinstance(edge_object, GraphEdge):
            raise TypeError("Must be passed a GraphEdge object")

        edge_id = edge_object.get_edge_id()
       
        
        if self.edge_exists(identifier1= edge_id):
            # the edge is already in the graph: merge attributes
            self.edges_dic[edge_id].merge_attribute(new_attribute_object= edge_object.get_edge_attribute_object())
            
        else:
            # if edge was not in the graph,  add the edge, letting the nodes know that they have this edge
            # The user is obliged to adding the nodes before adding the edge: yield error if nodes not existing
            start_node_id, end_node_id= edge_object.get_node_ids()
            
            start_node_object = self.get_node(identifier= start_node_id, get_mode="error" )
            start_node_object._has_edge(edge_id= edge_id, partner_id= end_node_id)

            end_node_object = self.get_node(identifier= end_node_id, get_mode="error"    )
            end_node_object._has_edge(edge_id= edge_id, partner_id= start_node_id )

            self.edges_dic[edge_id] = edge_object
        # END OF else: (if self.edge_exists(identifier1= edge_id):)

        #not being used self._changed() I am currentlyt not using this
    
    def get_hidden_edges(self):
        """
        Returns a list of hidden edges of the graph.
        """
        # currently, when the user asks for hidden edges, a new list is generated...
        # if in the future we manage this list so it is always updated, the following commented lines are important
        #
        #if not self.has_hidden:
        #    return []
        #elif self.hidden_edges is not None:
        #    return self.hidden_edges
        
        #        else:
        self._gen_hidden_lists()
        return self.hidden_edges
        

    def get_unhidden_edges(self):
        """
        Returns a list of unhidden edges of the graph.
        """
        # currently, when the user asks for unhidden edges, a new list is generated...
        # if in the future we manage this list so it is always updated, the following commented lines are important
        #
        #if not self.has_hidden:
        #    self.unhidden_edges = self.edges_dic.values()
            
        #elif self.unhidden_nodes is not None:
        #    pass
        
        #else:
        self._gen_hidden_lists()
            
        return self.unhidden_edges
    
    # ------------------------------------------
    # Graph Methods: Expanding
    # ------------------------------------------
    
    def _propagate_edge(self,
                        node_source_id= None,
                        node_target_id= None,
                        new_partner_id= None,
                        expansion= None,
                        expansion_name= None,
                        source_edge_db_id= None,
                        expansion_mode = "all",
                        exp_output_mode= "add",
                        output_target = None,
                        code_type= None,
                        alternative_code_types= None,
                        class_name= "all"):
        """
        method that (adds or prints) and edge between nodes node_target_id and new_partner_id
         -> it will set this new edge as propagated from node "node_source_id" and database edge "source_edge_db_id"

        It is called from Graph.expand() once all propagated edges have been determined

        "expansion_name" is given as attribute (instead of getting it here from expansion) to speed up things

        "expansion_mode"  (see expansion_mode in expand() )

        "exp_output_mode"  (see exp_output_mode in expand() )

        "output_target"  (see output_target in expand() )

        "code_type" (see code_type in expand() )

        "alternative_code_types" (see alternative_code_types in expand() )

        "class_name" (see class_name in expand() )
        
        
         If "exp_output_mode" is 'print' then info printed follows format:

           node1<TAB>node2<TAB>expansion_type<TAB>source_db_edge_id<TAB>node_source_id

         If "exp_output_mode" is 'add' then: process to follow to create the new edge is...
        
           1.0. create attribute for this new edge (set as propagated)
           1.1. look if there is already an edge joining these two nodes
                --> method Graph.create_edge returns a tuple indicating if edge exists
           1.2. In case there is already one, modify existing attribute to be set as "propagated"
                with characteristics needed (node_from, ...)
                --> method GraphEdge.merge_attributes (that calls merge_attribute from PianaGraphEdgeAttribute)
                    PianaGraphEdgeAttributes.merge_attributes (attribute1, attribute2) will
                    integrate the values of attribute2 into attribute 1, and then set the attribute to be
                    the attribute of this Edge
           1.3. In case there was no edge, create a new one with get_edge and add it to the list of edges to be added to graph
                the characteristics of this new edge are NOT those of the attribute we are reading
        

        """
        global root_nodes


        # Only adding/printing the edge if a root node is involved in it (assumming that predictions are wanted only for root_nodes)
        if (expansion_mode == "all" ) or ( (expansion_mode== "root") and ( root_nodes.has_key(node_target_id) or root_nodes.has_key(new_partner_id))):
            

            if node_source_id is not None and node_source_id != node_target_id:
                # -> if node_source_id is None, it means this is not a prediction but a real interaction
                #
                # -> predictions that come from the node_target_id are actually not new predictions but just a consequence of
                # a previous expansion. Do not add/print them, since node_target_id already has that interaction
            
                if exp_output_mode == "print":
                    # print information to output_target

                    target_passes_class_test = expansion.class_test(node_id= node_target_id, class_name= class_name)
                    partner_passes_class_test = expansion.class_test(node_id= new_partner_id, class_name= class_name)

                    if target_passes_class_test and partner_passes_class_test:
                        # only printing expansion if it passes the class test
                        
                        if verbose_propagate_print:
                            sys.stderr.write("Printing expanded interaction between %s and %s, source node being %s and source db %s\n" %(node_target_id,
                                                                                                                                          new_partner_id,
                                                                                                                                          node_source_id,
                                                                                                                                          source_edge_db_id))


                        node_target_external_code = expansion.get_external_code(node_id = node_target_id,
                                                                                code_type=code_type,
                                                                                alternative_code_types=alternative_code_types)

                        new_partner_external_code = expansion.get_external_code(node_id = new_partner_id,
                                                                                code_type=code_type,
                                                                                alternative_code_types=alternative_code_types)
                        
                        node_source_external_code = expansion.get_external_code(node_id = node_source_id,
                                                                                code_type=code_type,
                                                                                alternative_code_types=alternative_code_types)


                        output_target.write("%s\t%s\t%s\t%s\t%s\n" %(node_target_external_code,
                                                                     new_partner_external_code,
                                                                     expansion_name,
                                                                     source_edge_db_id,
                                                                     node_source_external_code  ) )
                    # END OF if target_passes_class_test and partner_passes_class_test:
                    else:
                        # nodes are of different class name than the required one
                        if verbose_propagate_print:
                            sys.stderr.write("Target (passes=%s) or Partner  (passes=%s) node did not pass the class test\n" %(target_passes_class_test,
                                                                                                                               partner_passes_class_test))
                    
                # END OF if exp_output_mode == "print":

                elif exp_output_mode == "add":

                    # generating edge_id of source_edge_db_id (the one that originated this new edge we are creating)
                    source_edge_id = GraphEdge._gen_edge_edgeID(node_id1= node_source_id, node_id2= new_partner_id)

                    # 1.0 first of all, we create an edge attribute describing the propagation of the edge
                    new_edge_attribute = expansion.create_new_edge_attribute(edge_db_id= None,
                                                                             expansion_type= expansion_name,
                                                                             propagated_from_node= node_source_id,
                                                                             propagated_from_edge_id= source_edge_id,
                                                                             propagated_from_db_edge_id= source_edge_db_id )


                    # create_edge returns a tuple [edge_object, new] where new==1 means edge didn't exist in the graph before
                    #             -> it doesn't add the edge to the graph, we have to do it afterwards with add_edge
                    expanded_edge = self.create_edge(node_id1= node_target_id,
                                                     node_id2= new_partner_id,
                                                     attribute_object= new_edge_attribute)



                    if expanded_edge[1] == 1:

                        if verbose_expansion_detailed:
                            sys.stderr.write(" adding ")

                        expanded_edge[0].set_propagated(1)

                        # edge has to be added: create nodes, add them and then add the edge
                        new_node1_attribute = expansion.create_new_node_attribute(node_id=node_target_id)
                        new_node2_attribute = expansion.create_new_node_attribute(node_id=new_partner_id)

                        new_node1 = self.get_node(identifier=node_target_id, attribute=new_node1_attribute)
                        new_node2 = self.get_node(identifier=new_partner_id, attribute=new_node2_attribute)

                        # we do not know if the nodes are already in the graph: add them (add_node takes care of merging attributes if different)
                        self.add_node(new_node1)
                        self.add_node(new_node2)
                        self.add_edge(expanded_edge[0])

                        if verbose_expansion_detailed:
                            sys.stderr.write(" added ")
                    else:
                        if verbose_expansion_detailed:
                            sys.stderr.write(" existing ")

                    # END OF if expanded_edge[1] == 1: ... else: ...
                # END OF elif exp_output_mode == "add":
                else:
                    raise ValueError("Incorrect exp_output_mode set to %s\n" %(exp_output_mode) )
                
            # END OF if node_source_id != node_target_id:
                           
        # END OF if (expansion_mode == "all" ) or ((expansion_mode== "root") and (root_nodes.has_key(node_target_id) or root_nodes.has_key(new_partner_id))):
                
        else:
            if verbose_expansion_shallow:
                sys.stderr.write(" skipped ")
            if verbose_expansion_detailed:
                sys.stderr.write(" skipped because expansion_mode= %s and node_target %s and node_partner %s not in root nodes %s " %(expansion_mode,
                                                                                                                            node_target_id,
                                                                                                                            new_partner_id,
                                                                                                                            root_nodes.keys()))
        # END OF else: (if (expansion_mode == "all" ) or ((expansion_mode== "root") and (roo ....)


    def expand(self, expansion= None, behaviour="fresh", expansion_mode="all", exp_output_mode="add",
               output_target= None, code_type= None, alternative_code_types= None, expansion_threshold=0, hub_threshold=0,
               class_name= "all"):
        """
        Expands node and edges of the graph based on behaviour defined by an object Expansion

          -> expand is used to "propagate" edges of one node to another node in case they share a certain characteristic
              - if node A has interactions a, b and c
              - if node B has interactions d, e and f
              - if node A and node B share characteristic X (defined by object expansion)
              - if we perform a expand on both nodes, the new network will be:
                 - A will have interactions a, b, c, d, e and f
                 - B will have interactions a, b, c, d, e and f

        "expansion" is an expansion object, created using a subclass of Expansion (and specific to the database you're working with)

        "expansion_mode" determines if all nodes are expanded or only those that were used as root to create the network
          - expansion_mode == "all" creates expanded edges for all nodes in the graph
          - expansion_mode == "root" creates expanded edeges only for root nodes

        "expansion_threshold" sets the threshold determining whether to expand a node or not
          - if a node shares a certain characteristic with lots of other nodes, it might not be a relevant characteristic
            for these cases, we introduce a threshold that determines that if there are more than "expansion_threshold" nodes sharing the characteristic
            with the node, then edges from those nodes will not be propagated to the node.
              - if 0, no thresholds will be applied

        "hub_threshold" sets the threshold for how many edges can a node have in order to be added to the network
          - if a node has more than "hub_threshold" edges, these edges will not be added to the network.
             - if 0, no threshold is applied

        "output_mode" determines if expansions performed will be added to the current graph or printed to stdout
          - output_mode == "add" adds expansions to graph
          - output_mode == "print" will print expansions to stdout

        "code_type" is a code type used in database, and is the code type that will be used for printing out propagated edges (in exp_output_mode "print")

        "alternative_code_types" are the alternative code types in case nothing is found for "code_type" (identifier will be preceded by "type:xxxx")

        "class_name" is used in output_mode print to choose the class of nodes we want to print out
           - "all" will print all classes of nodes
           - other classes are defined by the subclass of Graph you are using (eg. in PianaGraph, class_name is the protein taxonomy id)

           - expansions will only be printed out if both nodes in the edge are of class "class_name"
        
        """
          
        global root_nodes
        
        # TO DO!!!! expansion should not be applied to hidden nodes!!!!!! And edges should not be expanded when a filter was applied previously...
        #           maybe one solution is to perform the expansion normally, and then reapply the filters to the resulting graph.
        #           this is easier than checking for each edge...
        # TO CHECK: mode hidden or unhidden is not correctly used in method Graph.expand() or get_neighbours(all...)
        #           get_db_edge_ids_from_node: I am not taking into account filters in this get...

        # TO DO!!! I have removed the part of expansion where I "complete" the network with edges of those nodes
        #          that have been added to the network as a result of an expansion (see section 3 expand_before_cleaning).
        #          This means that in the network we have nodes for which we haven't added their interactions at any depth
        #          It would be good to have a command in piana.py that "completes" the network at depth X, by searching
        #          for edges for each new added node in the network
        
        if not isinstance(expansion, NodeExpansion):
            raise TypeError ("expansion must be of type Expansion")

        if verbose_expansion_detailed:
            sys.stderr.write( "Starting to expand all nodes in Graph\n" )

        # -----------------------------------------------------------
        # Initialize variables that will be used during the expansion
        # -----------------------------------------------------------

        list_edges_to_propagate = []

        root_node_ids = self.get_root_node_ids_deep_copy()

        for root_node_id in root_node_ids:
            root_nodes[root_node_id] = None     # setting to None, the important thing here is to keep the root_node_id as a key
        
        expansion_name = expansion.get_expansion_name()

        if expansion_mode == "all":
            # obtain all nodes in original graph (before performing the expansion)
            list_node_id_to_expand = self.get_node_ids_list_deep_copy()   # deep copy needed: forces copy of vector instead of just having a reference
                                                                          #                   otherwise it gets confused in the loop: elements are added
                                                                          #                   to nodes_ids inside the loop and python doesn't like it.
        elif expansion_mode == "root":
            # since expansion_mode is root, obtain all root nodes in original graph (before performing the expansion)
            list_node_id_to_expand = root_node_ids
                                                    
        if verbose_expansion_shallow or verbose_expansion_detailed or verbose_expansion_minimal:
            number_of_nodes = len(list_node_id_to_expand)
            i = 0

        # -----------------------------------------------------------------------------------------------
        # For each node in the graph (or for root nodes in "root" expansion_mode), perform the expansion
        # -----------------------------------------------------------------------------------------------
        #
        # Algorithm description (process performed for each node we want to get expansions from):
        #
        # 1. create a list of graph edges where node_id_to_expand is involved
        #      --> getting the edges from the graph, not from the database (we asumme the network has all edges required by user)
        #      --> this is done at the beggining so we can keep the list of edges where node_id_to_expand is involved
        #
        # 2.  create a list of nodes that share 'something' with the node being processed (node_id_to_expand)
        #      --> these nodes will propagate their edges to node_id_to_expand
        #
        # 3   propagate edges of sharing_node_id to node_id_to_expand
        #     propagate edges of node_id_to_expand to sharing_node_id 
        #
        # 4. depending on the mode (print or add), print expansions to output or add them to the graph
        # -----------------------------------------------------------------------------------------------

        for node_id_to_expand in list_node_id_to_expand:

            if verbose_expansion_shallow or verbose_expansion_detailed or verbose_expansion_minimal:
                i +=1
                sys.stderr.write( "\nExpand: Processing node number %s <%s> of a total of %s with expansion %s\n" %(i, node_id_to_expand,
                                                                                                                    number_of_nodes,
                                                                                                                    expansion.name))

            # if expansion_mode is all, we need to get graph edges for node_id_to_expand to propagate them afterwards
            if expansion_mode == "all":

                ###############
                # 1
                ###############
                # FIRST of all, create a list of graph edges where node_id_to_expand is involved
                #               This list will then be used to propagate graph edges of node_id_to_expand to the
                #               new nodes added to the graph during the expansion that share a certain characteristic
                #               with node_id_to_expand
                #
                #               This step is done to avoid querying the database for edges of node_id_to_expand
                #               We asume that all edges of node_id_to_expand are already in the network
                #
                #               No need to do this for expansion_mode root, because in expansion_mode root we do not add graph edges of node_to_expand
                #               to new nodes being added to the network

                list_edge_db_ids_of_node_to_expand = []

                # loop through neighbour nodes of node_id_to_expand
                if verbose_expansion_shallow:
                        sys.stderr.write("Expand: obtaining graph edges for node_id_to_expand %s" %(node_id_to_expand) )

                for neighbour_id_of_node_to_expand in (self.get_node(identifier= node_id_to_expand)).get_neighbour_ids_dic():

                    edge_in_graph = self.get_edge(identifier1= node_id_to_expand,
                                                  identifier2= neighbour_id_of_node_to_expand,
                                                  get_mode="error")

                    edge_db_id_to_expand = edge_in_graph.get_db_edge_id()

                    # get the source nodes (None if edge coming from database) that originated this graph edge
                    list_edge_propagations = edge_in_graph.get_edge_attribute_object().get_propagations()
                    list_node_ids_propagated_from= []
                    for edge_propagation in list_edge_propagations:
                        list_node_ids_propagated_from.append( edge_propagation.get_source_node_id() )

                    # this is the list of edges of node_id_to_expand that will be "propagated" to its neighbours
                    #   it is a list of tuples [neighbour, edge db id, list node id propagated from]
                    list_edge_db_ids_of_node_to_expand.append([neighbour_id_of_node_to_expand, edge_db_id_to_expand, list_node_ids_propagated_from])

                # END OF for neighbour_id_of_node_to_expand in list_neighbour_ids_of_node_to_expand:

            # END OF  if expansion_mode == "all":

            
            ###############
            # 2
            ###############
            # expansion.get_sharing_nodes(node) returns the list of nodes that share 'something' (defined by the particular
            # expansion being used) with the node being processed. These nodes will propagate their edges to the original node
            #
            # these edges are obtained by expansion.get_sharing_nodes() by looking into an external database that
            # contains nodes, edges between the nodes and characteristics of nodes that may "expand" the interactions
            # from one node to another if they share a certain a characteristic (characteristic that is different for
            # each expansion type). 
            list_sharing_node_ids = expansion.get_sharing_nodes(node_id_to_expand)

            if expansion_threshold == 0 or len(list_sharing_node_ids) < expansion_threshold:
                # if threshold is respected, propagate edges of sharing nodes
                # if threshold not respected, the sharing nodes are ignored (ie. no edges are propagated to node_id_to_expand)
                
                if verbose_expansion_shallow or verbose_expansion_detailed or verbose_expansion_minimal:
                    number_of_sharing_node_ids = len(list_sharing_node_ids)
                    counter_number_of_sharing_node_ids = 0
                    sys.stderr.write( "Expand: Number of nodes that expand from node %s is: %s\n" %(node_id_to_expand, number_of_sharing_node_ids ))

                # For each node that shares characteristic X with node_object_to_expand, propagate edges between both
                for sharing_node_id in list_sharing_node_ids:

                    if verbose_expansion_shallow or verbose_expansion_detailed:
                        sys.stderr.write("3:%s out of %s -- " %(counter_number_of_sharing_node_ids, number_of_sharing_node_ids))
                        counter_number_of_sharing_node_ids +=1

                    ###############
                    # 3
                    ############### 
                    # ------------------------------------------------------------------
                    # PROPAGATION OF EDGES BETWEEN node_id_to_expand and sharing_node_id
                    # ------------------------------------------------------------------
                    #
                    # sharing_node_id is a node that shares a certain characteristic with node_id_to_expand
                    # we now need to:
                    #
                    # 3.1. Propagating edges of sharing_node_id to node_id_to_expand
                    # 3.1.1 add interacions from external database of sharing_node_id to node_id_to_expand
                    # 3.1.2 add interacions from graph of sharing_node_id to node_id_to_expand
                    #
                    # 3.2. Propagating edges of node_id_to_expand to sharing_node_id 
                    # 3.2.1 not needed: we assumme the graph already contains all edges for node_to_expand
                    # 3.2.2 add interacions from graph of node_id_to_expand to sharing_node_id
                    #
                    # 3.1.2 and 3.2.2 are needed because we also need to propagate those edges from previous
                    # expansions (those set as propagated), and these edges do not appear in the external database


                    # ------
                    # 3.1.1. add edges from external database of sharing_node_id to node_id_to_expand
                    # ------
                    if verbose_expansion_shallow:
                        sys.stderr.write( "Expand 3.1.1. add db edges of sharing_node_id (%s) to node_id_to_expand (%s)\n" %(sharing_node_id,
                                                                                                                             node_id_to_expand) )

                    # expansion.get_db_edge_ids_from_node is a method specific to the external database
                    # that is used to get database edge identifiers where a node participates
                    #   --> list_source_dbs and list_source_methods are fixed when creating the expansion object
                    list_db_edge_ids_of_sharing_node = expansion.get_db_edge_ids_from_node(sharing_node_id)

                    if hub_threshold ==0 or len(list_db_edge_ids_of_sharing_node) < hub_threshold:
                        # if hub_threshold respected, add db edges of sharing node to node_id_to_expand

                        
                        # for each edge of sharing_node_id, "propagate" it to node_id_to_expand
                        for db_edge_id_of_sharing_node in list_db_edge_ids_of_sharing_node:

                            # expansion.get_partner_node_id queries the external database to find the partner of sharing_node_id for
                            #                                                             the edge we are "propagating"
                            partner_id_of_sharing_node = expansion.get_partner_node_id(db_edge_id= db_edge_id_of_sharing_node,
                                                                                       node_id= sharing_node_id )

                            if verbose_expansion_detailed:
                                sys.stderr.write( "Expand 3.1.1 Creating edge from %s to %s derived from sharing_node %s\n" %(node_id_to_expand,
                                                                                                                              partner_id_of_sharing_node,
                                                                                                                              sharing_node_id))



                            # create an expanded edge between the node_id_to_expand and partner_id_of_sharing_node
                            #  -> appending EdgeToPropagate with information needed to propagate an edge

                            list_edges_to_propagate.append(EdgeToPropagate(node_id_edge_comes_from= sharing_node_id,
                                                                           node_id_edge_being_added= node_id_to_expand,
                                                                           node_id_new_partner= partner_id_of_sharing_node,
                                                                           db_edge_id_edge_comes_from= db_edge_id_of_sharing_node))

                        # END OF for db_edge_id_of_sharing_node in list_db_edge_ids_of_sharing_node
                    # END OF if hub_threshold ==0 or len(list_db_edge_ids_of_sharing_node) < hub_threshold:
                        
                    

                    # ------
                    # 3.1.2. add edges from graph of sharing_node_id to node_id_to_expand
                    # ------
                    # needed because we also need to propagate those edges from previous expansions (those set as propagated),
                    # and these edges do not appear in the external database
                    
                    if verbose_expansion_shallow:
                        sys.stderr.write( "Expand: 3.1.2. add edges from graph of sharing_node_id (%s) to node_id_to_expand (%s)\n" %(sharing_node_id,
                                                                                                                                      node_id_to_expand) )
                        
                    # lopp through neighbours of sharing_node 

                    for neighbour_id_of_sharing_node in self.get_node(identifier= sharing_node_id).get_neighbour_ids_dic():

                        edge_in_graph = self.get_edge(identifier1= sharing_node_id,
                                                      identifier2= neighbour_id_of_sharing_node,
                                                      get_mode="error")

                        edge_db_id_to_expand = edge_in_graph.get_db_edge_id()

                        propagate_edge = 1  # boolean used to decide if the graph edge has to be propagated or not
                                            #  it will be set to 0 if the graph edge was in the Graph from a previous propagation
                                            #  triggered by node_id_to_expand (i.e we don't want a node to have as "propagation"
                                            #  and edge that it originally had in the database)

                        for edge_propagation in edge_in_graph.get_edge_attribute_object().get_propagations():
                            if edge_propagation.get_source_node_id() == node_id_to_expand:
                                propagate_edge = 0
                                break


                        if propagate_edge == 1:
                            if verbose_expansion_detailed:
                                sys.stderr.write( "Expand 3.1.2 Creating edge from %s to %s derived from sharing_node %s\n" %(node_id_to_expand,
                                                                                                                              neighbour_id_of_sharing_node,
                                                                                                                              sharing_node_id))

                            # create an expanded edge between the node_id_to_expand and neighbour_id_of_sharing_node
                            #  -> appending EdgeToPropagate with information needed to propagate an edge

                            list_edges_to_propagate.append(EdgeToPropagate(node_id_edge_comes_from= sharing_node_id,
                                                                           node_id_edge_being_added= node_id_to_expand,
                                                                           node_id_new_partner= neighbour_id_of_sharing_node,
                                                                           db_edge_id_edge_comes_from= edge_db_id_to_expand))
                        # END OF if propagate_edge == 1:

                    # END OF for sharing_node_neighbour in sharing_node_neighbours:

                    if expansion_mode == "all":
                        # if expansion_mode is all, then add also edges from node_id_to_expand to sharing_node_id
                        #           -> since we assume that all edges of node_to_expand are already in the graph, no need to 
                        #              (ie no need of 3.2.1) query the database to get edges for node_to_expand: get edges only from graph
                        #
                        #           -> in expansion_mode root, no need to do this because we are only interested in propagating
                        #              to node_to_expand those edges of sharing node (from graph or from db) and we do not
                        #              want to propagate anything else
                        #
                        #           -> in expansion_mode all, we are propagating edges to nodes that were not necessarily previously
                        #              in the graph(sharing_node might have been in the graph or not): we have to do it,
                        #              to make sure the sharing_node has edges resulting from the expansion: this will
                        #              have the consequence that if we do another expansion afterwards, nodes that share
                        #              the characteristic with sharing_node_id will also receive expanded nodes (because they
                        #              will be in the graph)
                        #              TO CHECK: i am pretty sure I am doing this correctly, but maybe think about it once more...
                        #              TO CHECK: in fact, being extrict I think this step could be deleted, because we are
                        #                        adding an edge to a node that was not in the graph at the begining of the expansion process
                        #                        and expansion_mode==all refers to all nodes in the graph (not those added afterwards
                        #                        However, I have the feeling that if I don't do this step, when doing double
                        #                        expansions I am loosing these edges... I prefer to create more true expanded edges
                        #                        (although maybe useless) than forgetting a few of them


                        # 3.2.1 not needed: we assumme the graph already contains all edges for node_to_expand
                        
                        # ------
                        # 3.2.2. add edges from graph of node_id_to_expand to sharing_node_id
                        # ------

                        if verbose_expansion_shallow:
                            sys.stderr.write( "Expand: 3.2.2. add graph edges of node_id_to_expand %s to sharing_node_id %s\n" %(node_id_to_expand,
                                                                                                                            sharing_node_id)  )
                        # this list comes from 1.
                        #    -> it contains tuples [neighbour_id_of_node_to_expand, edge_db_id_to_expand, list_node_ids_propagated_from]
                        for edge_db_id_of_node_to_expand in list_edge_db_ids_of_node_to_expand:

                            propagate_edge = 1  # boolean used to decide if the graph edge has to be propagated or not
                                                #  it will be set to 0 if the graph edge was in the Graph from a previous propagation
                                                #  triggered by sharing_node_id (i.e we don't want a node to have as "propagation"
                                                #  and edge that it originally had in the database)

                            for node_id_propagated_from in edge_db_id_of_node_to_expand[2]:
                                if node_id_propagated_from == sharing_node_id:
                                    propagate_edge = 0
                                    break

                            if propagate_edge == 1:
                                if verbose_expansion_detailed:
                                    sys.stderr.write( "Expand: 3.2.2 Propagating interaction from %s to %s\n" %(sharing_node_id,
                                                                                                                edge_db_id_of_node_to_expand[0]) )

                                # create an expanded edge between the sharing_node_id and neighbour_id_of_node_to_expand
                                #  -> appending EdgeToPropagate with information needed to propagate an edge

                                list_edges_to_propagate.append(EdgeToPropagate(node_id_edge_comes_from= node_id_to_expand,
                                                                               node_id_edge_being_added= sharing_node_id,
                                                                               node_id_new_partner= edge_db_id_of_node_to_expand[0],
                                                                               db_edge_id_edge_comes_from= edge_db_id_of_node_to_expand[1]))
                            # END OF if propagate_edge == 1:

                        # END OF for edge_db_id_of_node_to_expand in list_edge_db_ids_of_node_to_expand:
                    # END OF if expansion_mode == "all":
                                    
                # END OF for sharing_node_id in list_sharing_node_ids:
                                
                if verbose_expansion_shallow or verbose_expansion_detailed:
                    sys.stderr.write("END sharing nodes\n")
            # END OF  if expansion_threshold == 0 or len(list_sharing_node_ids) < expansion_threshold:
            
            
            else:
                 if verbose_expansion_shallow or verbose_expansion_detailed:
                    sys.stderr.write("Expand: Number of nodes that expand from node %s is: Zero\n" %(node_id_to_expand) )
            # END OF else: (if list_sharing_node_ids:)


        # END OF for node_id_to_expand in node_id_to_expand:


        ###############
        # 4
        ############### 
        # Depending on expansion_mode, add or print propagated edges
        
        if  (verbose_expansion_minimal or verbose_expansion_shallow) and expansion_mode=="add":
            k = 0
            number_of_edges_to_propagate = len(list_edges_to_propagate)
            sys.stderr.write( "Expand: 4. Adding to graph %s edges created during the expansion\n" %number_of_edges_to_propagate)
       
        for edge_to_propagate in list_edges_to_propagate:
            
            if verbose_expansion_detailed and expansion_mode=="add":
                sys.stderr.write( "4: %s out of %s is" %(k, number_of_edges_to_propagate) )
                k += 1

            self._propagate_edge(node_source_id= edge_to_propagate.get_node_id_edge_comes_from(),
                                 node_target_id= edge_to_propagate.get_node_id_edge_being_added(),
                                 new_partner_id= edge_to_propagate.get_node_id_new_partner(),
                                 expansion = expansion,
                                 expansion_name = expansion_name,
                                 source_edge_db_id= edge_to_propagate.get_db_edge_id_edge_comes_from(),
                                 expansion_mode = expansion_mode,
                                 exp_output_mode= exp_output_mode,
                                 output_target= output_target,
                                 code_type= code_type,
                                 alternative_code_types= alternative_code_types,
                                 class_name=class_name)
            
            if verbose_expansion_detailed and expansion_mode=="add":
                sys.stderr.write("--")

        # END OF  for edge_to_propagate in list_edges_to_propagate:
        
        if verbose_expansion_detailed and expansion_mode=="add":
            sys.stderr.write( "4:END \n")
        
        self.applied_expansions.append(expansion)
    
    

    #------------------------------------------
    # Graph Methods: Filtering
    #------------------------------------------
    
    def filter(self, filters, behaviour="fresh"):
        """
        Hides node and edges in the graph based on filters.

        Behaviour can either be "fresh" or "inc".

        "fresh" (the default) unhides all nodes/edges previously hidden before applying
        the new filters.

        "inc" incrementally applies the new filters on the current visible graph.

        not working at the minute!
        """

        if verbose:
            sys.stderr.write( "Starting filtering with filters %s\n" %filters)
            
        if behaviour == "fresh":
            self.unfilter()
            
        elif behaviour == "inc":
            pass
        
        elif isinstance(behaviour,str):
            raise ValueError("behaviour must be \"fresh\" or \"inc\"")
        
        else:
            raise TypeError("behaviour must be \"fresh\" or \"inc\"")


        # TO DO!!! Create independent "filter" methods for Nodes and Edges???? Maybe useful...

        for node in self.get_node_object_list():
            for filter in filters:
                if isinstance(filter,NodeFilter):

                    # filter.test returns 1 if node passes the test, 0 if not
                    # passing the test means node remains unhidden. 0 means node must be hidden
                    
                    if filter.test(node= node):
                        pass
                    else:
                        node.hide()
                        self.has_hidden = 1

        for edge in self.get_edge_object_list():
            for filter in filters:
                #sys.stderr.write( "trying edge %s with filter %s\n" %(edge,filter))
                if isinstance(filter,EdgeFilter):

                    # filter.test returns 1 if edge passes the test, 0 if not
                    # passing the test means edge remains unhidden. 0 means node must be hidden
                    
                    if filter.test(edge= edge):
                        pass
                    else:
                        edge.hide()
                        self.has_hidden = 1

        if self.applied_filters is None:
            self.applied_filters = filters
        else:
            for filter in filters:
                self.applied_filters.append(filter)

        if verbose:   
            sys.stderr.write( "All nodes are:\n", self.print_nodes(filter_mode= "all"))
            sys.stderr.write( "Hidden nodes are:\n", self.print_nodes(filter_mode= "hidden"))
            sys.stderr.write( "Unhidden nodes are:\n", self.print_nodes(filter_mode= "unhidden"))
            
            sys.stderr.write( "All edges are:\n", self.print_edges(filter_mode= "all"))
            sys.stderr.write( "Hidden edges are:\n", self.print_edges(filter_mode= "hidden"))
            sys.stderr.write( "Unhidden edges are:\n", self.print_edges(filter_mode= "unhidden"))
            


    def unfilter(self):
        """
        Un-hides all nodes and edges of the graph.

        not working at the minute!
        """
        if self.has_hidden:
            
            for node in self.hidden_nodes:
                node.unhide()
                
            for edge in self.hidden_edges:
                edge.unhide()
                
            self.has_hidden = 0
            #self._changed()
            self.applied_filters = None


    def replace_filters(self, new_filters):
        """
        Replaces current set of filters on the graph with some new
        filters. Saves the old filters for restoration through
        restore_filters()

        not working at the minute!
        """
        self.saved_filters = self.applied_filters[:] # force copy
        self.unfilter()
        self.filter(filters= new_filters, behaviour="fresh")

    def restore_filters(self):
        """
        Restores an old set of filters replaced by replace_filters.

        not working at the minute!
        """
        self.unfilter()
        self.filter(filters= self.saved_filters, behaviour="fresh")
        self.saved_filters = None
    
    #------------------------------------------
    #  Graph Methods: Output
    #------------------------------------------


    def print_nodes(self, filter_mode="all"):
        """
        Prints nodes in Graph
        
        filter_mode can be:
         - "all": prints all nodes of the graph
         - "hidden": prints hidden nodes of the graph
         - "unhidden": prints unhidden nodes of the graph

        
        """
        string_nodes = "nodes="
        for node in self.get_node_object_list():

            if (not node.is_hidden() and filter_mode=="unhidden") or filter_mode=="all":
                string_nodes += "%s, " %(node.nodeID)
                
            elif (node.is_hidden() and filter_mode=="hidden"):
                string_nodes += "%s, " %(node.nodeID)
                
        return string_nodes


    def print_edges(self, filter_mode="all"):
        """

        prints edges in the graph
        
        filter_mode can be:
         - "all": prints all edges of the graph
         - "hidden": prints hidden edges of the graph
         - "unhidden": prints unhidden edges of the graph

        
        """
        string_edges = "edges="
        
        for edge in self.get_edge_object_list():

            if (not edge.is_hidden() and filter_mode=="unhidden") or filter_mode=="all":
                string_edges += "%s, " %(edge.edgeID)
                
            elif (edge.is_hidden() and filter_mode=="hidden"):
                string_edges += "%s, " %(edge.edgeID)
                
        return string_edges


    def output_edges_table(self, filter_mode="all", output_target= None):
        """

        prints  pairs of interacting nodes to output_target
        
        "output_target" is a file object (sys.stdout to print to screen)
        
        filter_mode can be:
         - "all": prints all edges of the graph
         - "hidden": prints hidden edges of the graph
         - "unhidden": prints unhidden edges of the graph

        The output will look like this:
        
            node1 id<TAB>node2 id
            node3 id<TAB>node4 id
            ..............
        
        """
        for edge in self.get_edge_object_list():

            print_edge = "no"
            
            if (not edge.is_hidden() and filter_mode=="unhidden") or filter_mode=="all":
                print_edge = "yes"
                
            elif (edge.is_hidden() and filter_mode=="hidden"):
                print_edge = "yes"

            if print_edge == "yes":
                output_target.write("%s\t%s\n" %(edge.get_start_node_id(), edge.get_end_node_id()))


    def output_dot_file(self, filter_mode="all", output_target= None, use_alternative_id="no"):
        """
        Generates .dot file output that can be fed directly into the
        GraphViz program neato. This produces very nice network images...


        "output_target" is a file object where the dot file will be written
        
        filter_mode can be:
        - "all": prints all edges of the graph
        - "hidden": prints hidden edges of the graph
        - "unhidden": prints unhidden edges of the graph

        
        "use_alternative_id" can be
          - "yes" --> uses alternative id for printing graph
          - "no"  --> uses internal id for printing graph

           use_alternative_id is used to codify in the node which is the label to be used for that node. This is useful when
           we want to label the node with something different than the node id.



        to produce a GIF file from the .dot file returned by this method, do the following:

        cat output_of_this_method | neato -Tgif -o output_network.gif
        
        """
        
        # print graph headers
        #output_target.write("graph G { graph [orientation=portrait]\n") # uncomment this line for more flexible generation of image
        output_target.write("graph G { graph [orientation=portrait, pack=true, overlap=scale]\n") # uncomment this line for best presentation
        output_target.write(" node [shape=box,fontsize=7,width=0.15,height=0.15,style=filled,fillcolor=lightblue];\n")
        
        for edge in self.get_edge_object_list():
            
            if (not edge.is_hidden() and filter_mode=="unhidden") or (filter_mode=="all") or (edge.is_hidden() and filter_mode=="hidden"):

                start_internal_node_id = edge.get_start_node_id()
                end_internal_node_id = edge.get_end_node_id()
                
                start_node_object = self.get_node(identifier= start_internal_node_id, get_mode="error" )
                end_node_object = self.get_node(identifier= end_internal_node_id , get_mode="error"   )
                
                if use_alternative_id == "yes":
                    start_node_id_to_print = start_node_object.get_node_alternative_id()
                    end_node_id_to_print = end_node_object.get_node_alternative_id()
                else:
                    start_node_id_to_print = start_internal_node_id
                    end_node_id_to_print = end_internal_node_id
                    
                if start_node_object.is_root():
                    output_target.write(""" "%s" [fillcolor = %s]\n""" %(start_node_id_to_print, "yellow"))
                    
                if end_node_object.is_root():
                    output_target.write(""" "%s" [fillcolor = %s]\n""" %(end_node_id_to_print, "yellow"))

                output_target.write( """ "%s" -- "%s";\n""" %(start_node_id_to_print, end_node_id_to_print))
        # END OF for edge in self.edges:
        
        # print graph termination    
        output_target.write( "}\n")

    
    def print_route(self, distance, route):
        """
        prints the route codified in "route" (which is just a list of node objects)
        """
        sys.stderr.write( "======================================================\n")
        sys.stderr.write( "Printing route with distance: %s\n" %distance)
        sys.stderr.write( "======================================================\n")

        sys.stdout.write(" START \n")
        for node in route:
            sys.stdout.write(" %s \n" %(node.nodeID) )

        sys.stderr.write( " END\n")
        sys.stderr.write( "======================================================\n")


    
    # ------------------------------------------
    #  Graph Methods: Internal
    # ------------------------------------------

    
    def _gen_hidden_lists(self):
        """
        generates the lists of hidden and unhidden node objects and edges
        """
        self.has_hidden = 0
        self.hidden_nodes = []
        self.unhidden_nodes = []
        self.hidden_edges = []
        self.unhidden_edges = []

        
        for node in self.get_node_object_list():

            if verbose:
                sys.stderr.write("Checking if node %s  is hidden: %s\n" %(node.get_node_id(), node.is_hidden()))
            
            if node.is_hidden():
                self.hidden_nodes.append(node)
                self.has_hidden = 1
            else:
                self.unhidden_nodes.append(node)

                
        for edge in self.get_edge_object_list():
            if edge.is_hidden():
                self.hidden_edges.append(edge)
                self.has_hidden = 1
            else:
                self.unhidden_edges.append(edge)

    def _changed(self):
        """
        Called whenever an edge or node has been added. Sets appropriate
        flags so that graph global values get recalculated as necessary

        Not being used at the minute!
        """
        
        self.clus_coef = None
        self.adjacency = None
        self.connectivity = None
        self.degree = None
        self.ave_degree = None
        self.laplacian = None

        # TO CHECK!!!! Why should I unhide the nodes?
        self.hidden_nodes = None
        self.unhidden_nodes = None

    def _build_adjacency(self):
        """
        Builds adjacency matrix of graph

        """
        
        k_node = len(self.get_unhidden_nodes())
        self.adjacency = numarray.zeros((k_node,k_node))
        for i in range(k_node):
            node = self.unhidden_nodes[i]
            for neighbour_id in node.get_neighbour_ids():
                neigh = self.get_node(identifier=neighbour_id,
                                      get_mode="error")
                j = self.unhidden_nodes.index(neigh)
                self.adjacency[i,j] = 1.
                self.adjacency[j,i] = 1.

    def _build_connectivity(self):
        """
        builds the connectivity matrix
        """

        if piana_configuration_parameters.piana_mode == "developer" or piana_configuration_parameters.piana_mode == "advanced_user":
        
            if self.adjacency is None:
                self._build_adjacency()
            self.connectivity = fgraph.connect(self.adjacency)
            
        elif piana_configuration_parameters.piana_mode == "simple_user":
            raise TypeError("This method not available in piana_mode 'simple_user': \
            check your configuration details in piana/code/utilities/piana_configuration_parameters.py")

    def _gen_degree(self):
        """
        Calculates the degree vector of all nodes of the graph and
        the average degree of the graph
        """
        if piana_configuration_parameters.piana_mode == "developer" or piana_configuration_parameters.piana_mode == "advanced_user":


            if self.adjacency is None:
                self._build_adjacency()
            self.degree = fgraph.gen_degree(self.adjacency)
            self.ave_degree = fgraph.ave_degree(self.degree)
            
        elif piana_configuration_parameters.piana_mode == "simple_user":
            raise TypeError("This method not available in piana_mode 'simple_user': \
            check your configuration details in piana/code/utilities/piana_configuration_parameters.py")
            

    def _build_laplacian(self):
        """
        Builds the laplacian matrix as the basis of spectral methods.
        """
        if self.adjacency is None:
            self._build_adjacency()
        if self.degree is None:
            self._gen_degree()
        # laplacian is degree down the diagonal with -adjacency off-diagonal
        k_node = len(self.get_unhidden_nodes())
        self.laplacian = numarray.zeros((k_node,k_node))
        for i in range(k_node):
            for j in range(k_node):
                self.laplacian[i,j] = - float(self.adjacency[i,j])
            self.laplacian[i,i] = float(self.degree[i])


    def _calc_clus_coef(self):
        """
        Calculate the graph clustering coefficient.
        """

        if piana_configuration_parameters.piana_mode == "developer" or piana_configuration_parameters.piana_mode == "advanced_user":


            if self.adjacency is None:
                self._build_adjacency()
            (clus_coefs,self.clus_coef) = fgraph.calc_clus_coef(self.adjacency)
            for i in range(len(self.get_unhidden_nodes())):
                self.unhidden_nodes[i].c = clus_coefs[i]
            
        elif piana_configuration_parameters.piana_mode == "simple_user":
            raise TypeError("This method not available in piana_mode 'simple_user': \
            check your configuration details in piana/code/utilities/piana_configuration_parameters.py")

    def get_clus_coef(self):
        """
        Returns the clustering coefficient of the graph. This is an
        intelligent interface - if the graph has been modified since the
        last calculation of the clustering coefficient (via a call to
        Graph.calc_clus_coef()), its value will be re-calculated before
        being returned.
        """
        if self.clus_coef is None:
            self._calc_clus_coef()
        return self.clus_coef

    def get_ave_degree(self):
        """
        Returns the average degree of the graph.
        This variable may be directly accessed as Graph.ave_degree but
        interface through this method is preferable as it ensures that the
        value is current.
        """
        if self.ave_degree is None:
            self._gen_degree()
        return self.ave_degree


    def _dij_init(self, query_node):
        """
        provides the common initialisation for Dijkstra algo's

        returns a tuple (distance dictionary with keys node_id and value distance to query node, list of nodes in graph)
        """
        global distance
        
        # if filters is not None:
        #    self.hide(filters)

        infinity = 1000000 # or just a large number! Max distance per edge i
        node_list = self.get_node_object_list()[:] # force a copy rather than a reference

        # set initial distance for all nodes withs respect to query node
        distance = {}
        for node in node_list:
            if node.get_node_id() == query_node.get_node_id():
                distance[node.get_node_id()] = 0
            else:
                distance[node.get_node_id()] = infinity
        # END OF for node in node_list:

        return (distance, node_list)



    #------------------------------------------
    #  Graph Methods: Graph theory methods
    #------------------------------------------

    def old_connected(self):
        """
        Returns 1 if graph is fully connected, 0 otherwise.

        Not very efficient! Use connected() instead!
        """
        if self.laplacian is None:
            self._build_laplacian()
        try:
            ev = LinearAlgebra.eigenvalues(self.laplacian)
            eigenvalues = []
            k_node = len(self.get_unhidden_nodes())
            for i in range(k_node):
                eigenvalues.append(ev[i])
                eigenvalues.sort()
            # print eigenvalues
            # secont smallest eigenvalue gives info about connectivity
            # check smallest is zero first though
            if not ( abs(eigenvalues[0]) < 0.000001 ):
                raise Exception("Internal error: Eigenvalue problem")
            if eigenvalues[1] > 0.0:
                return 1
            else:
                return 0
        except:
            # having problems here - need to sort this out
            return 1

    def connected(self):
        """
        Returns 1 if graph is fully connected, 0 otherwise.
        Uses fortran subroutine.
        """

        if piana_configuration_parameters.piana_mode == "developer" or piana_configuration_parameters.piana_mode == "advanced_user":

            if self.connectivity is None:
                self._build_connectivity()
            return fgraph.is_fully_connected(self.connectivity)
            
        elif piana_configuration_parameters.piana_mode == "simple_user":
            raise TypeError("This method not available in piana_mode 'simple_user': \
            check your configuration details in piana/code/utilities/piana_configuration_parameters.py")
                   
    def nodes_connected(self, node1, node2):
        """
        Returns 1 is node1 is connected to node2 (at any network depth), 0 otherwise.
        """
        if self.connectivity is None:
            self._build_connectivity()

        if verbose:
            sys.stderr.write( "connectivity: \n", self.connectivity)
            
        indx1 = self.unhidden_nodes.index(node1)
        indx2 = self.unhidden_nodes.index(node2)
        if self.connectivity[indx1,indx2]:
            return 1
        else:
            return 0


    def get_all_distances(self):
        """
        Uses Dijkstra's algorithm to find every pairwise distance in the
        graph. This is more efficient than calling get_distances() for
        each node.

        NOT IMPLEMENTED YET!!!
        """
        pass

    def get_distances(self, query_node_id):
        """
        Implements Dijkstra's algorithm for finding the distance to all
        nodes of the graph from one query node. This is _much_ more efficient
        than calling find_route() for all node combinations.

        Returns a dictionary of distances, the keys being the node ids and the contents the distance at which that node can be found
        """

        query_node = self.get_node(identifier= query_node_id, get_mode="error") # make sure we have the correct node

        # Commenting this 2 lines from Daniel code... I don't see why he imposed the graph to be fully connected...
        # First check that graph is fully connected
        #if not self.connected():
        #    raise Exception("Graph not fully connected - cannot determine all distances")

        # set this global otherwise we run into problems in the sort function
        global distance
        (distance, node_list) = self._dij_init(query_node) 
        
        # define the sort function
        def dij_sort_function(node_a, node_b):
            dist_a = distance[node_a.get_node_id()]
            dist_b = distance[node_b.get_node_id()]
            if dist_a > dist_b:
                return -1 # NB We want a reverse sort!
            elif dist_a < dist_b:
                return 1  # NB We want a reverse sort!
            else:
                return 0

        # start the main loop
        while node_list:
            # first reverse.sort the distance list
            node_list.sort(dij_sort_function)
            
            # pop the top of the list
            pop_node = node_list.pop()
            cur_dist = distance[pop_node.get_node_id()]

            # for all adjacent nodes
            # calculate a new distance via the pop'ed node
            # if this is shorter then update the distance hash
            # because the pop'ed node is always the node closest to
            # the query node. Neat, hey?!
            for neighbour_id in pop_node.get_neighbour_ids():

                current_edge = self.get_edge(identifier1=neighbour_id, identifier2= pop_node.get_node_id(), get_mode="error")
                
                new_dist = cur_dist + current_edge.distance()
                
                if distance[neighbour_id] > new_dist :
                    distance[neighbour_id] = new_dist
            # END OF for neighbour_id in pop_node.get_neighbour_ids():

        # all done - the distances hash holds the shortest distances
        return distance

    def find_shortest_route(self, start_node_id, end_node_id):
        """
        Uses Dijkstra's algorithm to find the shortest route between start_node_id and end_node_id

        returns a tuple ( distance, route)
        """
        start_node_object = self.get_node(identifier=start_node_id, get_mode="error")
        end_node_object = self.get_node(identifier=end_node_id, get_mode="error")


        if verbose:
            sys.stderr.write( "printing distances: \n" )

        if not self.nodes_connected(node1= start_node_object, node2= end_node_object):
            raise Exception("Nodes aren't connected")

        
        global distance
        (distance, node_list) = self._dij_init(start_node_object)

        
        # define the sort function
        def dij_sort_function(a, b):
            dist_a = distance[a.nodeID]
            dist_b = distance[b.nodeID]
            if dist_a > dist_b:
                return -1 # NB We want a reverse sort!
            elif dist_a < dist_b:
                return 1  # NB We want a reverse sort!
            else:
                return 0

        routes = {}
        # set the route hash
        for node in node_list:
            routes[node.nodeID] = []

        # start the main loop
        while node_list:
            # first sort
            node_list.sort(dij_sort_function)
            # pop
            pop_node = node_list.pop()
            if pop_node == end_node_object:
                break
            cur_dist = distance[pop_node.nodeID]

            for neighbour_id in pop_node.get_neighbour_ids():
                # for each neighbour of current node, add neighbour and distance to routes
                #   -> neigh is the neighbour of pop_node
                neigh= self.get_node(identifier=neighbour_id, get_mode="error")
                # cur_edge is the edge between the current node and its neighbour
                cur_edge = self.get_edge(identifier1= pop_node.nodeID, identifier2= neighbour_id, get_mode = "error" )

                # cur_edge = pop_node.get_edge_to(other_node_id= neigh)  this was implemented this way before I removed get_edge_to

                
                new_dist = cur_dist + cur_edge.distance()
                if distance[neigh.nodeID] > new_dist:
                    distance[neigh.nodeID] = new_dist
                    # now add the route as well
                    routes[neigh.nodeID] = []
                    for r_node in routes[pop_node.nodeID]:
                        routes[neigh.nodeID].append(r_node)
                    routes[neigh.nodeID].append(pop_node)

        routes[end_node_object.nodeID].append(end_node_object)
        
        return ( distance[end_node_object.nodeID], routes[end_node_object.nodeID])

    def find_routes(self,node1_id ,node2_id):
        """
        When passed two node objects, finds routes between them.
        Make sure nodes are retreived from the graph by extracting them with
        g.get_node(node_nodeID,get_mode="error") for example.
        THIS IS BROKEN AT THE MINUTE - IT'S NOT RETURNING ALL THE PATHS
        """
        # nice little recursive algorithm blatently hacked from somewhere...
        # first check a and z are what we need
        a = self.get_node(identifier= node1_id, get_mode="error")
        z = self.get_node(identifier= node2_id, get_mode="error")
        
        # define recursive subroutine
        def get_next_step(b, y, route = []):
            sys.stderr.write("In get_next_step with start node %s and end node %s\n" %(b,y))
            sys.stderr.write( "Route is %s\n" %route)
            route.append(b)
            if b == y:
                return route
            routes = []
            for node_id in b.get_neighbour_ids():
                node = self.get_node(identifier=node_id, get_mode="error")
                if node not in route:
                    newroutes = get_next_step(node,z,route)
                    for newroute in newroutes:
                        routes.append(newroute)
                else:
                    pass
            return routes
        # and call it
        return get_next_step(a,z)
                
    def clean_nodes(self):
        """
        Looks for and removes nodes with no edges.
        This is applied to the UNFILTERED graph - filtering has no effect so
        a node which has no edges when filtered will NOT be removed.
        """
        # use the GraphNode method
        # this may suffer from a performance hit - if so, building a degree
        # vector and looking at this may be faster.
        nodes_to_remove = []
        for node in self.get_node_object_list():
            if node.get_degree(all=1) == 0:
                nodes_to_remove.append(node)
        for node in nodes_to_remove:
            self.rm_node(node_object= node)


    # ----------------------------------------------------------
    # Static Methods that must be moved to class Graph Utilities
    # ----------------------------------------------------------

    # TO DO!!! Move these methods to graph utilities class

      
    def build_cluster_graph(node_list=None, old_edges_dictionary=None, old_node_2_new_node_dictionary=None,
                             new_node_to_old_node_list_dictionary= None,newGraph=None):
        """
        Method that builds a new cluster_graph from:

        "node_list": a list of new nodes to be added to the new cluster_graph

        "old_edge_dictionary": a dictionary that in each position there is a list of nodes that had edge with this node
        
        "old_node_2_new_node_dictionary": a dictionary that references old nodes_id to new_node_id e.g: old_node_2_new_node[old_node_id]=new_node_id

        "node_to_old_node_list_dictionary": a dictionary that references new nodes_id to a list of old nodes_id that contents inside
                                            eg: node_to_old_node_list_dictionary[new_node_id]=[old_node_id1,old_node_id2...]

        """
        
        if node_list is None or old_edges_dictionary is None or old_node_2_new_node_dictionary is None:
            raise ValueError("Error: bad use of function Clustering.build_cluster_graph")

        for node in node_list:
            newGraph.add_node(node)
            # According with Graph object all nodes should be inserted before the edge insertion

        for node in node_list:

            associated_old_nodes = new_node_to_old_node_list_dictionary[node.get_node_id()]

            for associated_old_node in associated_old_nodes:

                for old_node_to_link in old_edges_dictionary[associated_old_node]:
                    """
                    Here we search each old node edges to make new edges
                    old_node_2_new_node_dictionary conteGGnt the references between new nodes created and old nodes in old graph
                    """
                    # add the edge
                    if verbose:
                        sys.stderr.write("creating new edge betwee node %s and node %s\n" %(node.get_node_id(), old_node_2_new_node_dictionary[old_node_to_link]) )
                    if node.get_node_id()!=old_node_2_new_node_dictionary[old_node_to_link]:        
                        edge=GraphEdge(node1_id=node.get_node_id(),node2_id= old_node_2_new_node_dictionary[old_node_to_link],graph=newGraph)
                        newGraph.add_edge(edge)

        # END OF for node in node_list:

        return newGraph
        
    build_cluster_graph = staticmethod(build_cluster_graph)


    def build_expanded_graph(node_list=None, old_edges_dictionary=None, old_node2new_node_dic=None,
                             new_node2old_node_dic= None,newGraph=None):

        
        """
        Method that builds a new cluster_graph from:

        "node_list": a list of new nodes to be added to the new cluster_graph

        "old_edge_dictionary": a dictionary that in each position there is a list of nodes that had edge with this node
        
        "old_node2new_node_dic": a dictionary that references old nodes_id to new_node_id
                                 e.g:old_node_2_new_node[old_node_id]=[new_node_id,new_node_id2...]
        
        "new_node2old_node_dic": a dictionary that references new nodes_id to a list of old nodes_id that contents inside
                                 eg: node_to_old_node_list_dictionary[new_node_id]=[old_node_id1,old_node_id2...]
        """
        
        for node in node_list:
            newGraph.add_node(node)
            # According with Graph object all nodes should be inserted before the edge insertion
        #End of node in  node_list

        for node in node_list:
            old_node=new_node2old_node_dic[node.get_node_id()]
            #getting old_node id associated to the current node
            old_edge_list=old_edges_dictionary[old_node]
            
            for node_in_edge in old_edge_list:
            #node_in_edge are old nodes having edge with the old node associated to the current node
                if old_node2new_node_dic.has_key(node_in_edge):
                    # creating and adding the edge to the graph
                    for associated_node in old_node2new_node_dic[node_in_edge]:
                        edge=GraphEdge(node1_id=node.get_node_id(),node2_id= associated_node,graph=newGraph)
                        newGraph.add_edge(edge)
                    
               
                   #END of associated_node..

            #END of node_in_edge...
        #END of node in node_list...

        return newGraph

    build_expanded_graph = staticmethod(build_expanded_graph)


                
