"""
File     : GraphNode.py
Author   : Daniel Jaeggi and Ramon Aragues
Creation : 2003
Contents : main class providing graph processing methods
Comments :

 
=======================================================================================================
A basic class defining a node in a graph.

The basic node class on which the graph is built.
Must be given unique nodeID.
Optionally given an attribute object GraphNodeAttribute.

"""
# GraphNode.py: implements a class for nodes contained in a Graph object
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

import sys
from sets import *

from GraphNodeAttribute import GraphNodeAttribute
#from Filter import NodeFilter, EdgeFilter

verbose = 0
verbose_has_edge = 0

class GraphNode:
    """
    A basic class defining a node in a graph.
    
    The basic node class on which the graph is built.
    Must be given unique nodeID.
    Optionally given an attribute object GraphNodeAttribute.
    
    """
    def __init__(self, nodeID, attribute = None, isRoot =0, graph = None, expanded_from=None, expansion_type=None, ishidden=0, alternative_id= None,
                 original= 0):
        
        self.nodeID = int(nodeID)
        self.alternative_id = alternative_id # can be used by user to keep an alternative id (eg, string describing node)
                                             # for all the internal operations, it is faster to use an integer

        self.graph = graph # the graph to which this node belongs to

        self.c = None   # connectivity value

        self.edge_ids_dic = {} # storing edges ids (as dic keys) which contain this node. Content is always None. Dic is faster than list.
        self.neighbour_ids_dic = {} # storing node ids (as dic keys) which are neighbours. Content is always None. Dic is faster than list.
        
        self.attribute = None  # for storing values specific to the node type being used in the graph
        self.hidden = ishidden
        self.isRoot = isRoot   # it will state whether the node was used as a seed for the graph (root node)
        self.expanded = [expanded_from, expansion_type]   # node_id and expansion type that generated this new node from an expansion
                                                          # if not a result of an expansion: [None, None]
        
        self.original = original # used to know whether this was of of the nodes initially added to the graph (used in clustering)
        
        if attribute is not None:
            if isinstance(attribute, GraphNodeAttribute):
                self.set_attribute(attribute_object= attribute)
            else:
                raise TypeError("Must be passed Attribute object")

    def __eq__(self,other):
        return isinstance(other, GraphNode) and (self.nodeID == other.nodeID)
    
    def __ne__(self,other):
        return not self.__eq__(other)
    
    def __lt__(self,other):
        if not isinstance(other,GraphNode):
            return NotImplemented
        else:
            return (self.nodeID < other.nodeID)
        
    def __gt__(self,other):
        if not isinstance(other,GraphNode):
            return NotImplemented
        else:
            return(self.nodeID > other.nodeID)

    def __le__(self,other):
        if not isinstance(other,GraphNode):
            return NotImplemented
        else:
            return (self < other) or (self == other)

    def __ge__(self,other):
        if not isinstance(other,GraphNode):
            return NotImplemented
        else:
            return (self > other) or (self == other)

    def __str__(self):
        string = "<GraphNode: NodeID " + str(self.nodeID)
        if self.c is not None:
            string += ", Connectivity " + str(self.c)
            
        if self.attribute is not None:
            string += ", Attribute " + str(self.attribute)
        string += ">"
        
        return string
    
    def __repr__(self):
        return self.nodeID


    def _has_edge(self, edge_id= None, partner_id= None):

        if not self.edge_ids_dic.has_key(edge_id):
            # edge not in node edges: add it
            self.edge_ids_dic[edge_id] = None     # setting to None, since the ids are kept as keys
                                                  # (ie if there is a key, then there is that edge)
                                                  # None is set as any other value could have been set
        # END OF if not self.edge_ids_dic.has_key(edge_id):
        
        if not self.neighbour_ids_dic.has_key(partner_id):
            # neighbour not in node neighbours: add it
            
            self.neighbour_ids_dic[partner_id] = None  # setting to None, since the ids are kept as keys
                                                       # (ie if there is a key, then there is that edge)
                                                       # None is set as any other value could have been set
        # END OF if not self.neighbour_ids_dic.has_key(partner_id):

    def set_node_id(self, new_node_id= None):
        """
        changes current nodeID for new_node_id
        """
        if not new_node_id is None:
            self.nodeID = new_node_id
            
    def get_node_id(self):
        """
        returns the node id of this node
        """
        return self.nodeID
    
    def get_node_alternative_id(self):
        """
        returns the alternative node id of this node
        """
        return self.alternative_id
    
    def get_number_neighbours(self):
        """
        returns the number of neighbours of this node in the graph where it belongs
        """
        return len(self.neighbour_ids_dic)

    def get_number_shared_neighbours(self, other_node_neighbours_ids_list):
        """
        returns the number of neighbours of self node that are as well in other_node_neighbours_ids_list (which is a list with neighbours of another node)
        """
        # this has been tested to be faster than checking dict vs list, or list vs list
        this_set= Set( self.neighbour_ids_dic.keys() )
        other_set= Set(other_node_neighbours_ids_list)
        return len(this_set.intersection(other_set))

    def get_neighbour_ids_dic(self, all=0):
        """
        Return a list of neighbouring nodes.
        
        Attention! argument 'all' not working.
            Normally: Will not return nodes if the node is hidden or the edge connecting them is hidden.
               -> If "all" is set true, will return all nodes irrespective of hidden status
        """
        # TO DO!!!! take into account hidden nodes/edges
        return self.neighbour_ids_dic

    def get_neighbour_ids(self, all=0):
        """
        Return a list of neighbouring nodes.
        
        
        Attention! argument 'all' not working.
            Normally: Will not return nodes if the node is hidden or the edge connecting them is hidden.
               -> If "all" is set true, will return all nodes irrespective of hidden status
        """
        # TO DO!!!! take into account hidden nodes/edges
        return self.neighbour_ids_dic.keys()

        
    def get_edge_ids(self, filter_mode = "all"):
        """
        Return a list of edge ids where node participates

        filter_mode not working at the minute. Future usage will be:
            -> will not return nodes if the node is hidden or the edge connecting them is hidden.
            -> if filter_mode=="all" is set true, will return all nodes irrespective of hidden status
        """
        # TO DO!!!! take into account hidden nodes/edges
        return self.edge_ids_dic.keys()

# commented because the edge can be obtained from the Graph using get_edge()
# as it is shown in get_distances...
# Daniel was getting the edge from the node, but in the new implementation the node doesn have that info
#    def get_edge_to(self, other_node_id):
#        """
#        Returns an edge linking self node to another node. Raises an
#        ValueError if no edge can be found.
#        """
        # returns the edge linking self to other
#        ret_edge = None
#        self.graph. get_nodes_connectivities(self, distance= 1):
#        for neigh in self.get_edge_ids():
#            if neigh == other_node_id:
#                print "encontrada arista entre %s y %s" %(self.nodeID, neigh)
#                ret_edge =self.graph.get_edge(self, identifier1 = self.nodeID, identifier2= neigh, get_mode="new" )
#                break

#        return ret_edge

    def merge_nodes(self, node_object, ignore_ids= 0):
        """
        merges current node with node_object passed as argument, as well as their attributes
        
        ignore_ids=1 merges attributes regardless of their ids 
        
        """
        if verbose:
            sys.stderr.write("Merging nodes %s and %s\n" %(self.get_node_id(), node_object.get_node_id()) )

        if not ignore_ids:
            if self.get_node_id() != node_object.get_node_id():
                raise TypeError("Merging nodes with different node id\n")

        # merging values of GraphNode
        if node_object.is_root():
            self.set_as_root()
            
        if node_object.is_hidden():
            self.hide()

        if verbose:
            sys.stderr.write("Adding edges of node %s (%s) to node %s (%s)\n" %(self.get_node_id(),self.get_edge_ids(),
                                                                                node_object.get_node_id(), node_object.get_edge_ids() ) )
            
        self.add_edges(list_edge_ids= node_object.get_edge_ids())

        if verbose:
            sys.stderr.write("Finished adding edges of node %s to node %s" %(self.get_node_id(), node_object.get_node_id()) )

        node_object_attribute = node_object.get_node_attribute_object() 

        if node_object_attribute is not None:
            self.merge_attribute(attribute_object= node_object_attribute, ignore_ids= ignore_ids)

    def merge_attribute(self, attribute_object, ignore_ids=0):
        """
        Merges current attribute with attribute_object passed as argument

        Since GraphNode does not know the specific details of its attributes, this method just calls
        the method merge_attribute of the current attribute

        ignore_ids=1 merges attributes regardless of their ids 
        
        """
        # TO CHECK!!! I don think it is very clean to change values of GraphNode when the method is suposed to
        #             change only the attribute... although it is closely related
        
        if isinstance(attribute_object, GraphNodeAttribute):
            
            if self.attribute is not None:
                self.attribute.merge_attribute(attribute_object= attribute_object, ignore_ids=ignore_ids)
            else:
                self.set_attribute(attribute_object= attribute_object)
        else:
            raise TypeError("Must be passed a GraphNodeAttribute object")

    def add_edges(self, list_edge_ids):
        """
        Adds edges from list_edges to node.edges, after checking they do not exist already

        This is used to update in the node which neighbours it currently has in the graph
        """
        if verbose:
            sys.stderr.write("Number of edges being added= %s" %(len(list_edges) ) )
      
        for edge_id in list_edge_ids:
                
            if  not self.edge_ids_dic.has_key(edge_id):
                self.edge_ids_dic[edge_id] = None # setting to None, since the ids are kept as keys
        

        
    def set_attribute(self, attribute_object):
        """
        Sets the GraphNodeAttribute "attribute_object" object to current node.
        Attribute must be an GraphNodeAttribute object.
        """
        if isinstance(attribute_object, GraphNodeAttribute):
            self.attribute = attribute_object
        else:
            raise TypeError("Must be passed a GraphNodeAttribute object")

    def get_node_attribute_object(self):
        """
        returns list of GraphNodeAttribute objects
        """
        return self.attribute

    def is_hidden(self):
        """
        if node is hidden, returns 1
        if node is not hidden returns 0
        """
        if self.hidden:
            return 1
        else:
            return 0

    def hide(self):
        """
        Hides the node and all edges associated with it
        """
        if not self.is_hidden():
            self.hidden = 1
            for edge_id in self.get_edge_ids():
                edge_object = self.graph.get_edge(identifier1= edge_id)
                if not edge_object.is_hidden():
                    edge_object.hide()
                

    def unhide(self):
        """
        Unhides the node.
        """
        self.hidden = 0

    def get_degree(self, all=0):
        """
        Returns the nodal degree taking into account hidden edges which
        are not included. This is the default behaviour.
        If all is set to true, hidden edges will also be included.
        """
        degree = 0
        if all:
            degree = len(self.edge_ids_dic.keys())
        else:
            for edge_id in self.get_edge_ids():

                edge_object = self.graph.get_edge(identifier1= edge_id, get_mode= "error")
                
                if not edge_object.is_hidden():
                    degree += 1
        return degree
  
    def set_as_root(self):
        """
        Sets the node as one of the root nodes. 
        """
        
        self.isRoot = 1
  
    def is_root(self):
        """
        Returns 1 if it is a root node (node that was used to build graph
        """
        return self.isRoot
  
    def set_as_expanded(self, expanded_from, expansion_type):
        """
        Sets the node as a result from an expansion

        expanded_from is the node_id that originated this expanded node
        expansion_type is the expansion name that originated this expanded node
        """
        
        self.expanded = [expanded_from, expansion_type]
  
    def is_expanded(self):
        """
        returns [None, None] if it is not a node coming from an expansion

        otherwise returns [node id that generated this node, type of expansion that generated this node]
        """
         
        return self.expanded

    def is_original(self):
        """
        returns 1 if this is a original node (added in the first round of the graph creation)
        0 otherwise

        this method is only used by the clustering...
        """
        return self.original
