"""
 File       : PianaGraphEdgeAttribute.py
 Author     : R. Aragues & D. Jaeggi
 Creation   : 2003
 Contents   : class PianaGraphEdgeAttribute (attribute specific to Piana for an Edge)
 Called from: 

=======================================================================================================

This file implements class PianaGraphEdgeAttribute

It describes an interaction of the protein-protein interaction network


"""

# PianaGraphEdgeAttribute.py: class PianaGraphEdgeAttribute (attribute specific to Piana for an Edge)
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

import copy

from GraphEdgeAttribute import *
from PianaDB import *
from PianaGraphEdgeTypes import *


verbose = 0
# --------------------------------
# Class PianaGraphEdgeAttribute
# --------------------------------   
    
class PianaGraphEdgeAttribute(GraphEdgeAttribute):
    """

    Class that implements attributes for Piana Edges: accessing edge values, setting edge values, ...
    
    """
    def __init__(self, interactionPiana_value= None, mem_mode = "onDemand", piana_access = None,
                 confidence_interaction_value= 0, propagated = None, extended = None):

        """
        "interactionPiana_value" is the database identifier for this edge

        "piana_access" is a PianaDBaccess object, used to retrieve information from the database
       
        "mem_mode" can be:
        --> "onDemand": values are retrieved from DB only when they are needed
        --> "inMemory": values are retrieved from DB at initilisation

  
        "propagated" is an object of class PropagatedEdge
        
        "extended" is an object of class ExtendedEdge

        Attention!!! Right now, not using extended for anything: I do not add edges to the graph by extension (extension meant that
        edges from the database where added for nodes that were added to network during an expansion.

              --> There will just be propagated edges

        Attention!!! Currently, not using "confidence_interaction_value"
 
        """

        self.db_access = piana_access   # setting the connection to the piana DB

        if interactionPiana_value is not None:
            self.interactionPiana = int(interactionPiana_value) # interactionPiana will be None if edge only comes from a propagation
        else: 
            self.interactionPiana = None
            
        self.sourceDB_methodID = [] # this will be a list of tuples [sourceDBID, methodID]
        self.keys_sourceDB_methodID = {} # to avoid redundancy in self.sourceDB_methodID
        self.list_sourceDB = [] # to speed up things when only interested in sourceDB
        self.list_methodID = [] # to speed up things when only interested in methodID
            
        self.confidenceInteraction = confidence_interaction_value

        self.has_changed = 1 # used to control when to update information and when information was already updated
                             # (this only affects the get_* methods)
                             # we start with a 1 to make sure the lists are updated at the beginning


        # propagated is a list of objects containing interactions of another node are added to another node
        # because both nodes share a certain characteristic
        self.propagated = []
        if propagated is not None:
            self.set_propagation(propagated_edge_object= propagated)  # will contain objects PropagatedEdge
                              
        # extended is a list of objects containing interactions of a node that has been added to the graph as a result
        # of an expansion
        self.extended = []
        if extended is not None:
            self.set_extension(extended_edge_object= extended)  # will contain objects ExtendedEdge


        if mem_mode == "onDemand":
            pass
            
        elif mem_mode == "inMemory":

            #self._set_confidenceInteraction()
            self._set_sourceDB_methodID()
            
        else:
            raise ValueError(""" mem_mode set to incorrect value (possible values are "onDemand" and "inMemory" """)

    def __deepcopy__(self, memo):

        clone = copy.copy(self) # Make a shallow copy
        for name, value in vars(self).iteritems():
            if name != "db_access":
                setattr(clone, name, copy.deepcopy(value, memo))
            else:
                setattr(clone, name, value)
                

        return clone


            
    def __str__(self):
        """
        implementation of conversion to string for PianaGraphEdgeAttribute
        """

        string = "PianaGraphEdgeAttribute consisting of <interactionPiana=%s, confidenceInteraction=%s, sourceDB_methodID=%s, extended=%s, propagated=%s> \
                  " %(self.interactionPiana, self.confidenceInteraction, self.sourceDB_methodID, self.extended, self.propagated)
        
        return string

    # --------------------------------
    # PianaGraphEdgeAttribute "set" methods
    # --------------------------------
    #
    # Methods to set values of PianaGraphEdgeAttributes
    #
    # 

    #def _set_confidenceInteraction(self):
        
    #     self.confidenceInteraction = self.db.get_db_content(PianaGlobals.ConfidenceInteraction, self.interactionPiana, answer_mode="single")


        
    def _get_sourceDB_methodID_key(self, sourceDBID=None, methodID=None):
        """
        returns a key for the pair sourceDB methodID
        """
        return sourceDBID + "." + methodID
    

    def _set_sourceDB_methodID(self):
        """
        Updates by quering the database the value sourceDB_methodID with a list of tuples [sourceDB, methodID]
        
        """

        if self.interactionPiana is not None:
            
            # this edge comes from a database interaction: get sourceDB and methodID for it

            # First of all, we obtain all sourceDBs for this interaction
            # then, we obtain the methods used for each one of those sourceDBs
            # we then add tuples [sourceDB, methodID] to sourceDB_methodID

            self.has_changed = 1
            all_sourceDB = []
            self.sourceDB_methodID = []
            self.keys_sourceDB_methodID = {} # to avoid redundancy in self.sourceDB_methodID
            
            for interaction_sourceDB in self.db_access.get_interaction_sourceDB_list(self.interactionPiana):
                all_sourceDB.append(interaction_sourceDB)
            # END OF for interaction_sourceDB in self.db_access.get_interaction_sourceDB_list(self.interactionPiana):

            if verbose:
                sys.stderr.write( "sourceDB list is: %s\n" %all_sourceDB)

            for sourceDBID in all_sourceDB:
                method_list_for_this_sourceDB = self.db_access.get_interaction_methodID_list_for_sourceDB(interactionPiana_value= self.interactionPiana,
                                                                                                          sourceDBID_value= sourceDBID)

                if verbose:
                    sys.stderr.write( "method list for interactionPiana %s with sourceDB %s is: %s" %(self.interactionPiana, sourceDBID,
                                                                                          method_list_for_this_sourceDB))

                if not method_list_for_this_sourceDB:
                    raise ValueError("How can it be that a sourceDB doesn't have any method? \
                    method list for interactionPiana %s with sourceDB %s is: %s" %(self.interactionPiana, sourceDBID,
                                                                                   method_list_for_this_sourceDB))
                else:
                    for method_for_this_sourceDB in method_list_for_this_sourceDB:

                        key_to_use= self._get_sourceDB_methodID_key(sourceDBID= sourceDBID, methodID= method_for_this_sourceDB)

                        if self.keys_sourceDB_methodID.has_key(key_to_use):
                            continue

                        self.keys_sourceDB_methodID[key_to_use] = None

                        self.sourceDB_methodID.append([sourceDBID, method_for_this_sourceDB])

                    # END OF for method_for_this_sourceDB in method_list_for_this_sourceDB:

            # END OF for sourceDB in all_sourceDB:
            
        else:
            # this edge does not come from a database interaction: therefore, it has no sourceDB or methodID
            self.sourceDB_methodID = []


    def add_sourceDB_methodID(self, sourceDBID =None, methodID= None):
        """
        adds (checking that there is no redundancy) the sourceDB and methodID to the current edge (used for merging edges)
        """
        key_to_use= self._get_sourceDB_methodID_key(sourceDBID= sourceDBID, methodID=methodID )

        if not self.keys_sourceDB_methodID.has_key(key_to_use):
            self.sourceDB_methodID.append([sourceDBID, methodID ])
            self.keys_sourceDB_methodID[key_to_use] = None
            self.has_changed = 1
        
    # --------------------------------
    # PianaGraphEdgeAttribute  methods to handle origin of the edge
    # --------------------------------
    #
    # Methods to handle the origin of the edge
    #
    # And edge can be original, extended and propagated. It can be the three at the same time. It has to be at least one of them
    #
    # 
 
    def set_propagation(self, expansion_type= None, propagated_from_node= None, propagated_from_db_edge_id= None, propagated_from_edge_id= None,
                        propagated_edge_object=None):
        """

        Adds to the edge an object describing a propagation (interactions of another node are added to another node because both nodes share
                                                             a certain characteristic)

        if propagated_edge_object is not None, adds that object (avoiding adding an object that already exists)
        if not, creates a new object using expansion_type, propagated_from_node, propagated_from_db_edge_id and propagated_from_edge_id

        The propagation is described in the object propagated_edge_object of class PianaNodeExpansion PropagatedEdge
        """

        propagated_edge_object = PropagatedEdge(expansion_type=expansion_type,
                                                propagated_from_node= propagated_from_node,
                                                propagated_from_db_edge_id= propagated_from_db_edge_id, 
                                                propagated_from_edge_id= propagated_from_edge_id )

        if propagated_edge_object not in self.propagated:
            self.propagated.append(propagated_edge_object)

  
    def get_propagations(self):
        """

        returns [] if it is not a edge coming from an propagation

        otherwise returns a list of objects PropagatedEdge

        """
         
        return self.propagated

   
    def set_extension(self, expansion_type=None, extended_from_node= None, extended_edge_object= None):
        """

        Adds to the edge an object describing an extension ( database interactions added to the graph because one of the nodes involved
                                                             in the interaction was added to the graph as a result of an expansion)


        if extended_edge_object is not None, adds that object (avoiding adding an object that already exists)
        if not, creates a new object using expansion_type and extended_from_node

        edge_extended_object is of class ExtendedEdge

        """
        if extended_edge_object is None:
            extended_edge_object = ExtendedEdge(expansion_type=expansion_type, extended_from_node= extended_from_node)
        
        if extended_edge_object not in self.extended:
            self.extended.append(extended_edge_object)
  
    def get_extensions(self):
        """

        returns [] if it is not an edge coming from an extension

        otherwise returns a list of objects ExtendedEdge

        """
         
        return self.extended

   
    # PianaGraphEdgeAttribute "get" methods
    # --------------------------------
    #
    # Methods to get values from PianaGraphEdgeAttributes
    #
    # If the value requested is not already in memory (i.e. mem_mode is onDemand), it is retrieved from the database and returned
    

    def get_db_edge_id(self):
        """
        Overrides method get_db_edge_id() of superclass GraphEdgeAttribute
        
        returns the interactionPiana identifier for this interaction

        It is the same as get_interactionPiana, but in this case it is overridden a method that all XXGraphEdgeAttribute must have
        This is the method that is called from Graph, since Graph doesn't know which kind of EdgeAttribute it is processing
        
        """
        
        return self.interactionPiana    

    def get_interactionPiana(self):
        """
        returns the interactionPiana identifier for this interaction
        """
        
        return self.interactionPiana    

    def _update_lists_values(self):
        """
        updates lists with methods and source dbs
        """
        if not self.sourceDB_methodID:
            self._set_sourceDB_methodID()

        self.list_sourceDB = []
        self.list_methodID = []
        
        for sourceDB_methodID in self.sourceDB_methodID:
            self.list_sourceDB.append(sourceDB_methodID[0])
            self.list_methodID.append(sourceDB_methodID[1])
            
        self.has_changed = 0
        
    def get_sourceDB_methodID(self):
        """
        returns sourceDB_methodID attribute, which is a list of tuples [sourceDB, methodID]
        """
        if not self.sourceDB_methodID:
            self._set_sourceDB_methodID()

        return self.sourceDB_methodID 
 
 
    def get_list_source_db(self):
        """
        returns a list with source databases for this interaction
        """
        if self.has_changed:
            self._update_lists_values()

        return  self.list_sourceDB  
 
    def get_list_method_id(self):
        """
        returns a list with methods for this interaction
        """
        if self.has_changed:
            self._update_lists_values()
      
        return self.list_methodID

    def merge_attribute(self, new_edge_attribute, ignore_ids= 0):
        """
        merges the attribute of the object with the new edge attribute passed as argument


        ignore_ids is used to avoid conflicts when unifying the network in PianaGraph
        """

        if not isinstance(new_edge_attribute, PianaGraphEdgeAttribute):
            raise TypeError("attribute to be merged must be PianaGraphEdgeAttribute")

        if not ignore_ids:
            # 1. merging interactionPiana values
            new_edge_interactionPiana = new_edge_attribute.get_interactionPiana()


            if self.interactionPiana is not None:
                if new_edge_interactionPiana is not None:
                    # if both interactionPiana are not None, make sure we are dealing with the same interaction. It wouldn't make sense
                    # to merge two attributes which dot not describe the same interaction.
                    if self.interactionPiana != new_edge_interactionPiana:
                        raise ValueError("Merge attribute of two different interactionPiana (%s and %s) interactions" %(self.interactionPiana,
                                                                                                                        new_edge_interactionPiana ))
                    # END OF if self.interactionPiana != new_edge_interactionPiana:

                    # if new_edge_interactionPiana is None, then the merging of the attributes will keep the one in self.interactionPiana

                # END OF if new_edge_interactionPiana is not None: ... 
            else:
                self.interactionPiana = new_edge_interactionPiana
        # END OF if not ignore_ids:


        # 2. merging sourceDB_methodID values
        list_sourceDB_methodID = new_edge_attribute.get_sourceDB_methodID()

        for pair_sourceDBID_methodID in list_sourceDB_methodID:
            self.add_sourceDB_methodID(sourceDBID=pair_sourceDBID_methodID[0], methodID=pair_sourceDBID_methodID[1])

        # 3. merging propagated
        for propagation in new_edge_attribute.get_propagations():
            self.set_propagation(propagated_edge_object= propagation)

        # 3. merging extended
        for extension in new_edge_attribute.get_extensions():
            self.set_extension(extended_edge_object= extension)
        

   #def get_confidenceInteraction(self):
   #
   #    if self.confidenceInteraction is None:
   #        
   #       self._setconfidenceInteraction()
   #       
   #    return self.confidenceInteraction
        
    def distance(self):
        """
        To  do!!! Get distance that makes biological sense

        Returns a distance like value.

        Defaults to return 1
        """
        return 1.0
