"""
 File       : PianaDB.py
 Author     : R. Aragues & D. Jaeggi
 Creation   : 2003
 Contents   : class for establishing conexions to pianaDB and handling inserts and selects
 Called from: PianaDBaccess.py

=======================================================================================================

This a generalization of mysql commands

To see how this class is used, look at any method in PianaDBaccess.py

"""

# PianaDB.py: class for establishing conexions to pianaDB and handling inserts and selects
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

from GraphNodeAttribute import *
from GraphEdgeAttribute import *
import PianaGlobals
import MySQLdb

import sys


verbose = 0


class PianaDB(object):
    """
    Class for establishing conexions to pianaDB and handling inserts and selects
    """

    def __init__(self, dbname=None, dbhost=None, dbuser=None, dbpassword=None ):

        if verbose:
            sys.stderr.write("Arguments received in PianaDB() are: %s, %s, %s, %s\n" %( dbname, dbhost, dbuser, dbpassword))

        self.dbname = dbname
        self.dbhost = dbhost
        self.dbuser = dbuser
        self.dbpassword = dbpassword

        
        # init database connection (different connection parameters depending on user preferences...)
        if not dbuser is None and not dbpassword is None:
            self.db = MySQLdb.connect(db= dbname, host= dbhost, user=dbuser, passwd=dbpassword)
        elif not dbuser is None:
            self.db = MySQLdb.connect(db= dbname, host= dbhost, user=dbuser)
        else:
            self.db = MySQLdb.connect(db= dbname, host= dbhost)
        # END OF else: (elif not dbuser is None:)
        
        self.cursor = self.db.cursor()


    # --
    # methods needed for using pickle with piana objects
    # -- 

    def __getstate__(self):

        odict = self.__dict__.copy() # copy the dict since we are going to change it
        del odict['db']              # remove conexion to MySQL: attribute self.db cannot be pickled
        return odict

    def __setstate__(self, dict):

        self.__dict__ = dict          # recover previous dictionary 
        dict.__class__.__init__(dict) # recover conexion to MySQL by calling init
        
    def __getnewargs__(self):

        return (self.dbname, self.dbhost, self.dbuser, self.dbpassword)  # returns arguments that init will get
                                                                         # when it is called after the pickle.load



    # --
    # handling inserts and selects through a generalized class
    # --

    def insert_db_content(self, sql_query, answer_mode = None):
        """
        Inserts values into a piana database (connection was established in self.db)

        Depending on argument "answer_mode", different things are returned.

        This method is called by PianaDBaccess to then process the results and return them to the user

        "slq_query" is normally obtained through classes implemented in PianaInsertSQL.py, which have a method get_sqlquery
        that creates the sql query needed to retrieve the searched value.

        "answer_mode" can be one of the following:

        - None: nothing is returned
        - 'last_id' : last id inserted is returned
        - 'num_updated' : number of rows that were updated (used by UPDATE statements...)

          --> 'last_id' mode only works for those tables that have an auto_increment ID!!!!!!! Will not work with primary keys that are not
              auto_increment. Currently, following tables have auto_increment ids: protein (ID=proteinPiana) and interaction (ID=interactionPiana)

        """
        if sql_query is not None:
            try:
                self.cursor.execute(sql_query)
            except Exception, inst:
                sys.stderr.write("Attention: this query was not executed due to a mysql exception: <<%s>>\n" %(sql_query))
                sys.stderr.write("           Error Reported: %s\n" %(inst))
                answer_mode = None
        else:
            raise ValueError("Trying to execute an empty insertion sqlquery")

        if answer_mode == "last_id":
            # in case mode "last_id" was chosen, do a select to the database to obtain the last autoincrement id inserted
            
            self.cursor.execute("""Select LAST_INSERT_ID()""")
            answer = self.cursor.fetchall()

            if answer:
                return answer[0][0]
            else:
                return None
        
        elif answer_mode == "num_updated":
            # in case mode "num_updated" was chosen, the number of columns that were updated will be in the answer of the sql query
            answer = self.cursor.fetchall()
            if not answer:
                return None
            else:
                return answer[0]
        
        else:
            return None
        # END OF else: (elif answer_mode == "num_updated":)

        
    def select_db_content(self, sql_query= None, answer_mode="single", remove_duplicates="yes", number_of_selected_elems= 1):
        """
        Returns content from a piana database (connection was established in self.db)
        
        "slq_query" is normally obtained through classes implemented in PianaSelectSQL.py, which have a method get_sqlquery
        that creates the sql query needed to retrieve the searched value
        
        "answer_mode" is used to let the user choose what kind of answer he is expecting from the sql query
        answer_mode can take values (default is "single"):
        - "single": just one element (if nothing is found, returns None)
        - "list": a list of single elements (if nothing is found, returns empty list [])
        - "raw":  the raw result from the sql query (a matrix)
        
        "remove_duplicates" is used to let the user decide if the returning list can contain duplicates or not
        (only used when answer_mode="list")
        (this is useful to remove similar entries from one query, for example same uniprot accession numbers returned
        that are actually the same under different swissAccession source DB )
        
        - "yes" will return a list where all elements are unique
        - "no" will return the complete list returned by the sql query
        
        "number_of_selected_elems" sets the number of elements being selected in the sql query.
        - If 1, only the string of the element is returned. 
        - If >1, each group of elements is represented by a tuple (elem1, elem2, ...)
        - if you want to use a number_of_selected_elems higher than 3, you have to modify the code below
        to create keys of the number of elements you wish to select
        """
        if sql_query is not None:
            try:
                self.cursor.execute(sql_query)
                answer = self.cursor.fetchall()
            except Exception, inst:
                sys.stderr.write("Attention: this query was not executed due to a mysql exception: <<%s>>\n" %(sql_query))
                sys.stderr.write("           Error Reported: %s\n" %(inst))
                answer = None
                
            
             
        else:
            raise ValueError("Trying to execute an empty selection sqlquery")
            
        # when fetchall doesn't find anything, it returns an empty object.
        # if something was found, transform to answer_mode requested by user
        
        # TO DO!!!! Speed up things by creating a dictionary with those values that have been already inserted...
        #           that will be much faster than checking on a list!!!
        
        if answer:
            # something was found: each answer_mode returns a different "something"
            if answer_mode == "single":
                if number_of_selected_elems == 1:
                    return answer[0][0]
                else:
                    return answer[0]
            
            elif answer_mode == "list":
                auxList = []
                added_elements = {}   # used to decide whether the element was already added or not
                for element in answer:
                        if number_of_selected_elems == 1:

                            if (remove_duplicates=="no") or ( remove_duplicates=="yes" and (not added_elements.has_key(element[0]) ) ):
                                added_elements[element[0]] = None
                                auxList.append(element[0])
                        else:
                            # if there are more than one element, we have to create a key for the dictionary of added_elements
                            # the key depends on the lenght of the elements

                            if number_of_selected_elems == 2:

                                in_key_a = min(element[0],element[1]) 
                                in_key_b = max(element[0],element[1])

                                key = "%s.%s" %(in_key_a, in_key_b)
                                
                            elif number_of_selected_elems == 3:

                                if element[0] <= element[1] and element[0] <= element[2] :
                                    in_key_a = element[0]
                                    
                                    in_key_b = min(element[1], element[2])
                                    in_key_c = max(element[1], element[2])
                                    
                                elif element[1] <= element[0] and element[1] <= element[2]:
                                    in_key_a = element[1]
                                    
                                    in_key_b = min(element[0],element[2]) 
                                    in_key_c = max(element[0],element[2])
                                    
                                elif element[2] <= element[0] and element[2] <= element[1]:
                                    in_key_a = element[2]
                                    
                                    in_key_b = min(element[0],element[1]) 
                                    in_key_c = max(element[0],element[1])
                                
                                key = "%s.%s.%s" %(in_key_a, in_key_b, in_key_c )
                                
                            else:
                                raise ValueError("Currently, PianaDB doesn't admit more than 3 selected elements. You can modify the code if you wish")
                            
                            if (remove_duplicates=="no") or ( remove_duplicates=="yes" and (not added_elements.has_key(key) ) ):
                                added_elements[key] = None
                                auxList.append(element)
                        # END OF else: (if number_of_selected_elems == 1:)
                # END OF for element in answer:
                            
                return auxList
            # END OF elif answer_mode == "list":
            
            elif answer_mode == "raw":
                return answer
            else:
                raise ValueError("answer_mode value is not correct")
        # END OF if answer:
        
        else:
            # nothing was found: each answer_mode returns a different "nothing"
            if answer_mode == "single":
                return None
            elif answer_mode == "list":
                return []
            elif answer_mode == "raw":
                return answer
            else:
                raise ValueError("answer_mode value is not correct")
        # END OF else: (if answer:)
