"""
 File       : PianaInsertSQL.py
 Author     : R. Aragues & D. Jaeggi
 Creation   : 2003
 Contents   : subclasses for generating insert sql statements
 Called from: PianaDBaccess

=======================================================================================================

This class is used in the same way as PianaSelectSQL

Each of the subclasses creates an sql query string that can be used to insert information into Piana

This is done this way to generalize as much as possible the interaction of users with the database pianaDB.

A user that needs to insert information into pianaDB will not create the SQL statement himself, but will
use one of these classes, together with method insert_db_content() of class PianaDB

class PianaDBaccess integrates into a single and clear interface all the access to piana databases
"""

# PianaInsertSQL.py: subclasses for generating insert sql statements
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues


import PianaGlobals

verbose = 1


class PianaInsertSQL:
    """
    General class for generating Piana SQL INSERT queries. All SQL queries inserting data into piana should be extensions of this class
    """
    def get_sqlquery():
        """
        Override this with the sql query that inserts the desired values
        """
        return None
    
    get_sqlquery = staticmethod(get_sqlquery) 

#---------------------------------------------------------------------
#
# PROTEIN RELATED CLASSES
#
#---------------------------------------------------------------------

class UpdateTableColumn(PianaInsertSQL):
    """
    Generates sql statement to insert information on table protein
    """
    def get_sqlquery(proteinPiana= None, table= None, column= None, new_value= None):
                
        sqlquery = """UPDATE %s SET %s=%s where %s=%s """ \
        %(table, column, new_value, PianaGlobals.proteinPiana_col, proteinPiana )
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProtein(PianaInsertSQL):
    """
    Generates sql statement to insert information on table protein
    """
    def get_sqlquery(proteinPiana_value, tax_id_value, proteinSequence_value, proteinMD5_value,
                     proteinSequenceLength_value, proteinMW_value, proteinIP_value):
                
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s, %s, %s) \
        VALUES (%s, %s, "%s", "%s", %s, %s, %s) """ \
        %(PianaGlobals.protein_table,
          PianaGlobals.proteinPiana_col, PianaGlobals.speciesNCBI_col, PianaGlobals.proteinSequence_col, PianaGlobals.proteinMD5_col,
          PianaGlobals.proteinSequenceLength_col, PianaGlobals.proteinMW_col, PianaGlobals.proteinIP_col,
          proteinPiana_value, tax_id_value, proteinSequence_value, proteinMD5_value, proteinSequenceLength_value, proteinMW_value, proteinIP_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class InsertProteinPianaCorrespondence(PianaInsertSQL):
    """
    Generates sql statement to generate the SQL statement that obtains the proteinPiana associated to a proteinMD5 in table proteinCorrespondence

    This class is used to keep consistency of proteinPiana <--> proteinSequence between different versions of piana databases

    Attention: not using it! Well... information is inserted, but coherence between different versions of piana databases is not maintained
    """
    def get_sqlquery(proteinPiana_value, proteinMD5_value, tax_id_value):
        
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s) VALUES (%s, "%s", %s)""" %(PianaGlobals.proteinCorrespondence_table,
                                                                                    PianaGlobals.proteinPiana_col,
                                                                                    PianaGlobals.proteinMD5_col,
                                                                                    PianaGlobals.speciesNCBI_col,
                                                                                    proteinPiana_value,
                                                                                    proteinMD5_value,
                                                                                    tax_id_value)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class InsertProteinPianaCounter(PianaInsertSQL):
    """
    Generates sql statement to generate the SQL statement that obtains a new proteinPiana identifier from proteinPianaCounter table

    After getting the new proteinPiana identifier, the method calling this class must make sure that the counter in increased by 1
    """
    def get_sqlquery(proteinPiana_value):
        
        sqlquery = """UPDATE %s set %s=%s""" %(PianaGlobals.proteinPianaCounter_table, PianaGlobals.proteinPianaCounter_col,
                                               proteinPiana_value)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinSimilarity(PianaInsertSQL):
    """
    Generates sql statement to insert  protein similarity information into table proteinSimilarity
    """
    def get_sqlquery(proteinPianaA_value, proteinPianaB_value):

        # make sure proteinPianaA < proteinPianaB
        if proteinPianaA_value > proteinPianaB_value:
            temp = proteinPianaA_value
            proteinPianaA_value = proteinPianaB_value
            proteinPianaB_value = temp
                
        sqlquery = """INSERT IGNORE INTO %s (%s, %s) \
        VALUES (%s, %s) """ \
        %(PianaGlobals.proteinSimilarity_table,
          PianaGlobals.proteinPianaA_col, PianaGlobals.proteinPianaB_col,
          proteinPianaA_value, proteinPianaB_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinExternalCode(PianaInsertSQL):
    """
    Generates sql statement to insert protein information on tables that follow the structure:

    codeID  (external code, for example swissprotID) (string)
    proteinPiana  (internal piana protein code)
    codeSource  (name of origin of the data being inserted)

    Instead of writing one class for each table, this class is designed to handle all these similar tables. This is somehow
    dangerous, since we are hardcoding the order of the fields of the tables. If the order is changed, this class
    will not work... and we would need to de-generalize it to insert information separately in each table.

    current tables managed through this class are: swissProt,  geneName, gi, protein_id_intDB_table
    """
    def get_sqlquery(table_name, code_value,  proteinPiana_value, codeSource_value):
        
        sqlquery = """INSERT IGNORE INTO %s \
        VALUES ("%s", %s, "%s") """ \
        %(table_name, code_value,  proteinPiana_value, codeSource_value)

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinSwissAccessionCode(PianaInsertSQL):
    """
    Generates sql statement to insert protein information on tables that follow the structure:

    codeID  (external code, for example swissprotAccession) (string)
    proteinPiana  (internal piana protein code)
    isPrimary (boolean field 1 (primary id) or 0 (secondary id)
    codeSource  (name of origin of the data being inserted)

    Instead of writing one class for each table, this class is designed to handle all these similar tables

    current tables managed through this class are: swissAccession 
    """
    def get_sqlquery(table_name, code_value,  proteinPiana_value, isPrimary_value, codeSource_value):

        sqlquery = """INSERT IGNORE INTO %s VALUES ("%s", %s, %s, "%s") """ \
        %(table_name, code_value,  proteinPiana_value, isPrimary_value, codeSource_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)


class InsertProteinEMBLCode(PianaInsertSQL):
    """
    Generates sql statement to insert protein information on tables that follow the structure:

    code ID (external code, for example emblPID)
    proteinPiana  (internal piana protein code)
    version (number of version for the external code)
    codeSource  (name of origin of the data being inserted)

    Instead of writing one class for each table, this class is designed to handle all these similar tables

    current tables managed through this class are: emblAcc, emblPID
    """
    def get_sqlquery(table_name, code_value, proteinPiana_value, version_value, codeSource_value):
        
        sqlquery = """INSERT IGNORE INTO %s VALUES ("%s", %s, %s, "%s") """ \
        %(table_name, code_value,  proteinPiana_value, version_value, codeSource_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinInterProCode(PianaInsertSQL):
    """
    Generates sql statement to insert protein information on tables that follow the structure:

    code ID (external code, for example interProID) (string)
    proteinPiana  (internal piana protein code)
    description (description of codeID) (string)
    codeSource  (name of origin of the data being inserted)

    Instead of writing one class for each table, this class is designed to handle all these similar tables

    current tables managed through this class are: interPro
    """
    def get_sqlquery(table_name, code_value, proteinPiana_value, description_value, codeSource_value):

        
        sqlquery = """INSERT IGNORE INTO %s VALUES ("%s", %s, "%s", "%s") """ \
        %(table_name, code_value,  proteinPiana_value, description_value, codeSource_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinPIRCode(PianaInsertSQL):
    """
    Generates sql statement to insert protein information on tables that follow the structure:

    code ID (external code, for example pirEntry) (string)
    proteinPiana  (internal piana protein code)
    isComplete boolean information about if the entry is either a complete protein (1) or a fragment(0). NULL when not known
    codeSource  (name of origin of the data being inserted)

    Instead of writing one class for each table, this class is designed to handle all these similar tables

    current tables managed through this class are: pirEntry, pirAcc
    """
    def get_sqlquery(table_name, code_value,  proteinPiana_value, isComplete_value, codeSource_value):

        sqlquery = """INSERT IGNORE INTO %s VALUES ("%s", %s, %s, "%s") """ \
        %(table_name, code_value,  proteinPiana_value, isComplete_value, codeSource_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinPdbCode(PianaInsertSQL):
    """
    Generates sql statement to insert protein information on tables that follow the structure:

    code ID (external code, for example PDB ID) (string)
    proteinPiana  (internal piana protein code)
    chain ID (specific to codeID) (string)
    codeSource  (name of origin of the data being inserted)

    Instead of writing one class for each table, this class is designed to handle all these similar tables

    current tables managed through this class are: pdb
    """
    def get_sqlquery(table_name, code_value, proteinPiana_value, chain_value, codeSource_value):

        # column pdb_chain will always follow format pdb_code.chain, where chain will never be None
        #   if the input chain is None, we will write an A instead (convention)
        combined_pdb_chain = code_value

        if chain_value != "_" and chain_value != "" and chain_value is not None:
            combined_pdb_chain = combined_pdb_chain + "." + chain_value
        else:
            combined_pdb_chain = combined_pdb_chain + ".A"
        
        sqlquery = """INSERT IGNORE INTO %s VALUES ("%s", %s, "%s", "%s", "%s") """ \
        %(table_name, code_value,  proteinPiana_value, chain_value, combined_pdb_chain, codeSource_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

#---------------------------------------------------------------------
#
# PROTEIN ATTRIBUTES RELATIONSHIP RELATED CLASSES
#
#---------------------------------------------------------------------

class InsertProteinAttributeRelationship(PianaInsertSQL):
    """
    Generates sql statement to insert information on protein attributes relationship tables
    
    the tables must follow the structure:
    
    (currently for tables:
    
    proteinCog_table
    proteinGo_table
    proteinEC_table 
    proteinSpecies_table 
    proteinDescription_table 
    proteinKeyword_table 
    proteinFunction_table 
    proteinSubcellularLocation_table
    )
    """
    def get_sqlquery( attribute_relationship_table, attribute_value, proteinPiana_value, attributeSource_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s   \
        VALUES ("%s",%s,"%s") """ \
        %(attribute_relationship_table,
          attribute_value, proteinPiana_value, attributeSource_value)

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinDBAliRelationship(PianaInsertSQL):
    """
    Generates sql statement to insert information of DBAli clustering of pibase patches to table proteinDBAliCluster
    """

    def get_sqlquery( dbali_cluster_id_value, proteinPiana_value, clustering_method_value, patch_residues_value, protein_dbali_cluster_source):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s )  \
        VALUES ("%s", %s, "%s", "%s", "%s") """ \
        %(PianaGlobals.proteinDBAliCluster_table,
          PianaGlobals.DBAliclusterID_col, PianaGlobals.proteinPiana_col, PianaGlobals.DBAliclusteringMethod_col, PianaGlobals.patchResidues_col,
          PianaGlobals.sourceDBID_col,
          dbali_cluster_id_value , proteinPiana_value, clustering_method_value, patch_residues_value, protein_dbali_cluster_source )
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinScopRelationship(PianaInsertSQL):
    """
    Generates sql statement to insert information of SCOP to table proteinScop
    """
    def get_sqlquery( cf_value, sf_value, fa_value, proteinPiana_value, source_value):
        
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s )  \
        VALUES (%s, %s, %s, %s, "%s") """ \
        %(PianaGlobals.proteinScop_table,
          PianaGlobals.cf_col, PianaGlobals.sf_col, PianaGlobals.fa_col, PianaGlobals.proteinPiana_col, PianaGlobals.proteinScopSource_col,
          cf_value, sf_value, fa_value, proteinPiana_value, source_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertProteinCathRelationship(PianaInsertSQL):
    """
    Generates sql statement to insert information of CATH to table proteinCath
    """
    def get_sqlquery( cathID_value, res_start_value, res_end_value, segmentID_value, proteinPiana_value, source_value):
        
        cath_fields = cathID_value.strip().split(".")

        if len(cath_fields) == 4:
            c_value = cath_fields[0]
            a_value = cath_fields[1]
            t_value = cath_fields[2]
            h_value = cath_fields[3]
            
        elif len(cath_fields) == 3:
            c_value = cath_fields[0]
            a_value = cath_fields[1]
            t_value = cath_fields[2]
            h_value = NULL
            
        elif len(cath_fields) == 2:
            c_value = cath_fields[0]
            a_value = cath_fields[1]
            t_value = NULL
            h_value = NULL
            
        elif len(cath_fields) == 1:
            c_value = cath_fields[0]
            a_value = NULL
            t_value = NULL
            h_value = NULL

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s  )  \
        VALUES (%s, %s, %s, %s, "%s","%s",%s, %s, %s, "%s") """ \
        %(PianaGlobals.proteinCath_table,
          PianaGlobals.c_col, PianaGlobals.a_col, PianaGlobals.t_col, PianaGlobals.h_col, PianaGlobals.cathID_col,
          PianaGlobals.segmentID_col, PianaGlobals.res_start_col,  PianaGlobals.res_end_col,
          PianaGlobals.proteinPiana_col, PianaGlobals.proteinCathSource_col,
          c_value, a_value, t_value, h_value, cathID_value,
          segmentID_value, res_start_value, res_end_value,
          proteinPiana_value, source_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

#---------------------------------------------------------------------
#
# PROTEIN ATTRIBUTES RELATED CLASSES
#
#---------------------------------------------------------------------

class InsertSpecies(PianaInsertSQL):
    """
    Generates sql statement to insert information on table species
    """
    def get_sqlquery(speciesNCBIid_value, speciesName_value, speciesDescription_value,  speciesSource_value, speciesKingdom_value=None):

        if speciesKingdom_value is None:
            speciesKingdom_value = "unknown"
            
        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s)  \
        VALUES (%s, "%s", "%s", "%s", "%s") """ \
        %(PianaGlobals.species_table, PianaGlobals.speciesNCBI_col, PianaGlobals.speciesName_col, PianaGlobals.speciesDescription_col, 
          PianaGlobals.speciesSource_col, PianaGlobals.speciesKingdom_col,
          speciesNCBIid_value, speciesName_value, speciesDescription_value, speciesSource_value, speciesKingdom_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class UpdateSpeciesKingdom(PianaInsertSQL):
    """
    Generates sql statement to update the kingdom field in species table
    """
    def get_sqlquery(speciesNCBIid_value, speciesKingdom_value=None):

        if speciesKingdom_value is None:
            speciesKingdom_value = "unknown"
        
        sqlquery = """UPDATE IGNORE %s SET %s="%s" where %s=%s """ \
        %(PianaGlobals.species_table,
          PianaGlobals.speciesKingdom_col, speciesKingdom_value,
          PianaGlobals.speciesNCBI_col, speciesNCBIid_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class InsertGo(PianaInsertSQL):
    """
    Generates sql statement to insert information on table go
    """
    def get_sqlquery(goID_value, go_name_value, go_acc_value, go_term_type_value, go_distance2root_value, goSource_value):


        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s, %s)  \
        VALUES ("%s", "%s", "%s", "%s", "%s", "%s") """ \
        %(PianaGlobals.go_table,
          PianaGlobals.goID_col, PianaGlobals.go_name_col, PianaGlobals.go_acc_col, PianaGlobals.go_term_type_col, PianaGlobals.go_distance2root,
          PianaGlobals.goSource_col,goID_value,    go_name_value,    go_acc_value,    go_term_type_value, go_distance2root_value,    goSource_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertGo_term2term_distance(PianaInsertSQL):
    """
    Generates sql statement to insert distance between two terms
    """
    def get_sqlquery(term1_id_value, term2_id_value, distance_value):

        if term1_id_value > term2_id_value:
            # make sure order term1 < term2 is respected
            temp = term1_id_value
            term1_id_value = term2_id_value
            term2_id_value = temp
            
        sqlquery= """ INSERT IGNORE INTO %s (%s,%s,%s)  VALUES ("%s","%s","%s")"""\
                  %(PianaGlobals.go_term2term_distance_table,
                    PianaGlobals.go_term2term_distance_term1_col,
                    PianaGlobals.go_term2term_distance_term2_col,
                    PianaGlobals.go_term2term_distance_col,
                    term1_id_value,
                    term2_id_value,
                    distance_value)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertCog(PianaInsertSQL):
    """
    Generates sql statement to insert information on table cog
    """
    def get_sqlquery(cogID_value, cogDescription_value, cogFunction_value, cogSource_value):


        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s)  \
        VALUES ("%s", "%s", "%s", "%s") """ \
        %(PianaGlobals.cog_table,
          PianaGlobals.cogID_col, PianaGlobals.cogDescription_col, PianaGlobals.cogFunction_col, PianaGlobals.cogSource_col,
            cogID_value,             cogDescription_value,                cogFunction_value,            cogSource_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

#---------------------------------------------------------------------
#
# PROTEIN EXTERNAL DATABASES RELATED CLASSES
#
#---------------------------------------------------------------------

class InsertUniprotInfo(PianaInsertSQL):
    """
    Generates sql statement to insert information on table uniprotInfo
    """
    def get_sqlquery(proteinPiana_value, swissProtID_value, swissAccessionID_value, data_class_value, description_value, geneName_value,
                     organism_value, organelle_value, proteinSequenceLength_value, proteinMW_value):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s)  \
        VALUES (%s, "%s","%s","%s","%s","%s","%s","%s", %s, %s) """ \
        %(PianaGlobals.uniprotInfo_table,
          PianaGlobals.proteinPiana_col, PianaGlobals.swissProtID_col, PianaGlobals.swissAccessionID_col, PianaGlobals.data_class_col,
          PianaGlobals.description_col, PianaGlobals.geneName_col, 
          PianaGlobals.organism_col, PianaGlobals.organelle_col, PianaGlobals.proteinSequenceLength_col, PianaGlobals.proteinMW_col, 
          proteinPiana_value, swissProtID_value, swissAccessionID_value, data_class_value, description_value, geneName_value,
          organism_value, organelle_value, proteinSequenceLength_value, proteinMW_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

    
class InsertCellFitness(PianaInsertSQL):
    """
    Generates sql statement to insert information on table cell_fitness
    """
    def get_sqlquery(fitnessScore_value, reaction_value, conditions_value, cellFitnessSource_value, proteinPiana_value):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s)  \
        VALUES (%s,"%s","%s", "%s", %s) """ \
        %(PianaGlobals.cellFitness_table, PianaGlobals.fitnessScore_col, PianaGlobals.reaction_col, PianaGlobals.conditions_col,
          PianaGlobals.cellFitnessSource_col, PianaGlobals.proteinPiana_col, \
          fitnessScore_value, reaction_value, conditions_value, cellFitnessSource_value, proteinPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

#---------------------------------------------------------------------
#
# INTERACTION RELATED CLASSES
#
#---------------------------------------------------------------------

class DeleteInteraction(PianaInsertSQL):
    """
    Generates sql statement to delete an interaction entry from table interaction
    """
    def get_sqlquery(interactionPiana_value):

        sqlquery = "DELETE FROM %s where %s = %s" %(PianaGlobals.interaction_table, PianaGlobals.interactionPiana_col, interactionPiana_value)
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
        
class DeleteInteractionSourceDB(PianaInsertSQL):
    """
    Generates sql statement to delete an interaction entry from table interactionSourceDB

    It can be used to delete a particular interactionPiana or just those entries that belong to a specific source db

       - if interactionPiana_value 'not None' --> delete entries where interactionPiana = interactionPiana_value

       - if sourceDBID_value 'not None' -->  delete entries where sourceDBID = sourceDBID_value

    Only one argument can be 'not None'
    """
    def get_sqlquery(interactionPiana_value=None, sourceDBID_value=None):

        if interactionPiana_value is not None and sourceDBID_value is not None:
            raise ValueError("in DeleteInteractionSourceDB, both arguments cannot be 'not None'")
        
        if interactionPiana_value is not None:
            sqlquery = """DELETE FROM %s where %s=%s """ %(PianaGlobals.interactionSourceDB_table, PianaGlobals.interactionPiana_col, interactionPiana_value)

        elif sourceDBID_value is not None:
            sqlquery = """DELETE FROM %s where %s="%s" """ %(PianaGlobals.interactionSourceDB_table, PianaGlobals.sourceDBID_col, sourceDBID_value)
            
        else:
            raise ValueError("in DeleteInteractionSourceDB, both arguments cannot be 'None'")

        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
        
class DeleteInteractionMethod(PianaInsertSQL):
    """
    Generates sql statement to delete an interaction entry from table interactionMethod

    It can be used to delete a particular interactionPiana or just those entries that belong to a specific source db or those with a method ID

      - if interactionPiana_value 'not None' --> delete entries where interactionPiana = interactionPiana_value

      - if sourceDBID_value 'not None' -->  delete entries where sourceDBID = sourceDBID_value

      - if methodID_value 'not None' -->  delete entries where methodID = methodID_value

    Only one argument can be 'not None'
    """
    def get_sqlquery(interactionPiana_value=None, sourceDBID_value=None, methodID_value=None):

        if interactionPiana_value is not None and sourceDBID_value is not None and methodID_value is not None:
            raise ValueError("in DeleteInteractionMethod, all arguments cannot be 'not None'")
        
        elif interactionPiana_value is not None:
            sqlquery = """DELETE FROM %s where %s=%s """ %(PianaGlobals.interactionMethod_table, PianaGlobals.interactionPiana_col, interactionPiana_value)
            
        elif sourceDBID_value is not None:
            sqlquery = """DELETE FROM %s where %s="%s" """ %(PianaGlobals.interactionMethod_table, PianaGlobals.sourceDBID_col, sourceDBID_value)
            
        elif methodID_value is not None:
            sqlquery = """DELETE FROM %s where %s="%s" """ %(PianaGlobals.interactionMethod_table, PianaGlobals.methodID_col, methodID_value)
 
        else:
            raise ValueError("in DeleteInteractionMethod, all arguments cannot be 'None'")
            
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class DeleteInteractionScores(PianaInsertSQL):
    """
    Generates sql statement to delete an interaction scores entry from table interactionScores

    It can be used to delete a particular interactionPiana or just those entries that belong to a specific source db

      - if interactionPiana_value 'not None' --> delete entries where interactionPiana = interactionPiana_value

      - if sourceDBID_value 'not None' -->  delete entries where sourceDBID = sourceDBID_value


    Only one argument can be 'not None'
    """
    def get_sqlquery(interactionPiana_value=None, sourceDBID_value=None):

        if interactionPiana_value is not None and sourceDBID_value is not None and methodID_value is not None:
            raise ValueError("in DeleteInteractionScores, all arguments cannot be 'not None'")
        
        elif interactionPiana_value is not None:
            sqlquery = """DELETE FROM %s where %s=%s """ %(PianaGlobals.interactionScores_table, PianaGlobals.interactionPiana_col, interactionPiana_value)
            
        elif sourceDBID_value is not None:
            sqlquery = """DELETE FROM %s where %s="%s" """ %(PianaGlobals.interactionScores_table, PianaGlobals.sourceDBID_col, sourceDBID_value)
            
        else:
            raise ValueError("in DeleteInteractionScores, all arguments cannot be 'None'")
            
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
        
class DeleteInteractionProteinSource(PianaInsertSQL):
    """
    Generates sql statement to delete an interaction protein source entry from table interactionProteinSource

    It can be used to delete a particular interactionPiana or just those entries that belong to a specific source db

      - if interactionPiana_value 'not None' --> delete entries where interactionPiana = interactionPiana_value

      - if sourceDBID_value 'not None' -->  delete entries where sourceDBID = sourceDBID_value

    Only one argument can be 'not None'
    """
    def get_sqlquery(interactionPiana_value=None, sourceDBID_value=None):

        if interactionPiana_value is not None and sourceDBID_value is not None and methodID_value is not None:
            raise ValueError("in DeleteInteractionProteinSource, all arguments cannot be 'not None'")
        
        elif interactionPiana_value is not None:
            sqlquery = """DELETE FROM %s where %s=%s """ %(PianaGlobals.interactionProteinSource_table, PianaGlobals.interactionPiana_col, interactionPiana_value)
            
        elif sourceDBID_value is not None:
            sqlquery = """DELETE FROM %s where %s="%s" """ %(PianaGlobals.interactionProteinSource_table, PianaGlobals.sourceDBID_col, sourceDBID_value)
            
        else:
            raise ValueError("in DeleteInteractionProteinsource, all arguments cannot be 'None'")
            
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertInteraction(PianaInsertSQL):
    """
    Generates sql statement to insert information on table interaction
    """
    def get_sqlquery(proteinPianaA_value, isSourceA_value, proteinPianaB_value, isSourceB_value, interactionConfidence_value ):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
       
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s )\
        VALUES (%s,%s,%s,%s,%s) """ %(PianaGlobals.interaction_table, PianaGlobals.proteinPianaA_col, PianaGlobals.isSourceA_col,
                                      PianaGlobals.proteinPianaB_col, PianaGlobals.isSourceB_col, PianaGlobals.interactionConfidence_col,
                                      proteinPianaA_value, isSourceA_value, proteinPianaB_value, isSourceB_value, interactionConfidence_value )
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertInteractionSourceDB(PianaInsertSQL):
    """
    Generates sql statement to insert information on table interactionSourceDB
    """

    def get_sqlquery(sourceDBID_value, confidenceAssignedSourceDB_value, interactionPiana_value ):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s)\
        VALUES ("%s",%s,%s) """ %(PianaGlobals.interactionSourceDB_table, PianaGlobals.sourceDBID_col,
                                  PianaGlobals.confidenceAssigned_col,  PianaGlobals.interactionPiana_col,
                                  sourceDBID_value, confidenceAssignedSourceDB_value, interactionPiana_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertInteractionMethod(PianaInsertSQL):
    """
    Generates sql statement to insert information on table interactionMethod
    """
    def get_sqlquery(methodID_value, interactionPiana_value, sourceDBID_value):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s)\
        VALUES ("%s",%s, "%s") """ %(PianaGlobals.interactionMethod_table,
                               PianaGlobals.methodID_col, PianaGlobals.interactionPiana_col, PianaGlobals.sourceDBID_col,
                               methodID_value, interactionPiana_value, sourceDBID_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

class InsertInteractionFeatures(PianaInsertSQL):
    """
    Generates sql statement to insert information on table interactionMethod
    """
    def get_sqlquery(pubmedID_value, interactionPiana_value, sourceDBID_value):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s)\
        VALUES ("%s",%s, "%s") """ %(PianaGlobals.interactionFeatures_table,
                               PianaGlobals.pubmedID_col, PianaGlobals.interactionPiana_col, PianaGlobals.sourceDBID_col,
                               pubmedID_value, interactionPiana_value, sourceDBID_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
    
class InsertInteractionScores(PianaInsertSQL):
    """
    Generates sql statement to insert information on table interactionScores
    """
    def get_sqlquery(interactionPiana_value, sourceDBID_value, equiv_nscore_value, equiv_nscore_transferred_value, equiv_fscore_value,
                     equiv_pscore_value, equiv_hscore_value, array_score_value, array_score_transferred_value,
                     experimental_score_value, experimental_score_transferred_value, database_score_value, database_score_transferred_value,
                     textmining_score_value, textmining_score_transferred_value, combined_score_value):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
       
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)\
                                     VALUES (%s, "%s", %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s) """ %(
            PianaGlobals.interactionScores_table,
            PianaGlobals.interactionPiana_col,
            PianaGlobals.sourceDBID_col,
            PianaGlobals.equiv_nscore_col,
            PianaGlobals.equiv_nscore_transferred_col,
            PianaGlobals.equiv_fscore_col,
            PianaGlobals.equiv_pscore_col,
            PianaGlobals.equiv_hscore_col,
            PianaGlobals.array_score_col,
            PianaGlobals.array_score_transferred_col,
            PianaGlobals.experimental_score_col,
            PianaGlobals.experimental_score_transferred_col,
            PianaGlobals.database_score_col,
            PianaGlobals.database_score_transferred_col,
            PianaGlobals.textmining_score_col,
            PianaGlobals.textmining_score_transferred_col,
            PianaGlobals.combined_score_col,
            interactionPiana_value,
            sourceDBID_value,
            equiv_nscore_value,
            equiv_nscore_transferred_value,
            equiv_fscore_value,
            equiv_pscore_value,
            equiv_hscore_value,
            array_score_value,
            array_score_transferred_value,
            experimental_score_value,
            experimental_score_transferred_value,
            database_score_value,
            database_score_transferred_value,
            textmining_score_value,
            textmining_score_transferred_value,
            combined_score_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)

    
class InsertInteractionProteinSource(PianaInsertSQL):
    """
    Generates sql statement to insert information on table interactionProteinSource
    """
    def get_sqlquery(interactionPiana_value = None, proteinPianaSource_value = None, sourceDBID_value=None):

        # Those %s referring to character types should be surrounded by quotes. Otherwise, mysql gives error
        sqlquery = """INSERT IGNORE INTO %s (%s, %s, %s) VALUES (%s, %s, "%s") """ %(
            PianaGlobals.interactionProteinSource_table,
            PianaGlobals.interactionPiana_col,
            PianaGlobals.proteinPiana_col,
            PianaGlobals.sourceDBID_col,
            interactionPiana_value,
            proteinPianaSource_value,
            sourceDBID_value)
        
        return sqlquery
    
    get_sqlquery = staticmethod(get_sqlquery)
 
