"""
File        : ScoringFunctionQuotient.py
Author      : Ramon Aragues
Creation    : 09.2004
Contents    : implementation of ScoringFunctionQuotient.py
Called from : ScoringFunction
Subclass    : 

=======================================================================================================

This file implements class ScoringFunctionQuotient

ScoringFunctionQuotient is an implementation of ScoringFunction that simply applys a quotient scoring function using the input weights (see __init__)

matrixes are implemented using numarray


These scores will then be used to find which are the patchgroups to be clustered together
"""

from ScoringFunction import *
from numarray import *


verbose_scoring_function = 0

#----------------------------------------------------------------------------------------------
class ScoringFunctionQuotient(ScoringFunction):
#----------------------------------------------------------------------------------------------
    """
    
    """
    def __init__(self, w_int_patchgroups_shared = 1, w_int_proteinPianas_shared = 1, w_belong_to_same_protein = 1 ):
        
        """

        This scoring function calculates scores with the following formula applied to each cell of the matrix:

                (int_patchgroups_shared * w_int_patchgroups_shared) + (int_proteinPianas_shared * w_int_proteinPianas_shared) 
        score = --------------------------------------------------------------------------------------------------------------
                                     1 + belong_to_same_protein * w_belong_to_same_protein

        
        w_* are used to tune the formula. Default values are all 1

        the matrixes array_interacting_patchgroups_shared, array_interacting_proteinPianas_shared, array_belong_to_same_protein
        are passed as arguments to method get_scores_array()

        
        """
        
        
        self.w_int_patchgroups_shared = w_int_patchgroups_shared
        self.w_int_proteinPianas_shared = w_int_proteinPianas_shared
        self.w_belong_to_same_protein = w_belong_to_same_protein
        

        ScoringFunction.__init__(self, scoring_function_name= "quotient")

    def get_w_patches(self):
        """
        returns the weight assigned to w_int_patchgroups_shared
        """
        return self.w_int_patchgroups_shared
    
    def get_w_prots(self):
        """
        returns the weight assigned to w_int_proteinPianas_shared
        """
        return self.w_int_proteinPianas_shared
    
    def get_w_belong(self):
        """
        returns the weight assigned to w_belong_to_same_protein
        """
        return self.w_belong_to_same_protein


    def get_scores_array(self, array_interacting_patchgroups_shared, array_interacting_proteinPianas_shared, array_belong_to_same_protein):
        """
        returns an array with the result of applying this scoring function for each position of the matrix
        
        """

        number_of_patchgroups = shape(array_interacting_patchgroups_shared)[0]
        
        scores_array = zeros( (number_of_patchgroups, number_of_patchgroups), Float32)

        scores_array = ( (array_interacting_patchgroups_shared*self.w_int_patchgroups_shared) +
                         (array_interacting_proteinPianas_shared*self.w_int_proteinPianas_shared)) / ((array_belong_to_same_protein*self.w_belong_to_same_protein) + 1)
        
        return scores_array
        
