"""
File     : PianaGraphExpansion.py
Author   : Ramon Aragues
Creation : 04.03.2004
Contents : class that implements expansions that can be applied to PianaGraph
Comments :

=======================================================================================================

These classes specify how Expansions are carried out with piana graphs

An expansion can either be a node or and edge expansion and must contain methods specific to the PianaGraph

"""

# PianaGraphExpansion.py: implements a classes that specify how Expansions are carried out with piana graphs
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

from Expansion import *
from PianaDBaccess import *
from PianaGraph import *
from PianaGraphEdgeAttribute import *
from PianaGraphNodeAttribute import *


verbose = 0

class PianaNodeExpansion(NodeExpansion):
    """

    Class that contains methods used for all kinds of Piana Node expansions

    Specific piana expansions only need to implement the methods that return a list of nodes that share a node characteristic (method get_expanded_nodes() )

    This is the class used for Node Expansions: propagating interactions of one node to another
    """
    def __init__(self, piana_access_object=None, name="", use_self_ints="yes",
                 list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):
        
        if piana_access_object is None:
            raise ValueError("Cannot create a PianaNodeExpansion without providing PianaDBaccess object")

        self.piana_access = piana_access_object
        self.name = name
        
        self.use_self_ints = use_self_ints
        self.list_source_dbs = list_source_dbs
        self.inverse_dbs = inverse_dbs
        self.list_source_methods =list_source_methods
        self.inverse_methods = inverse_methods




    def get_expansion_name(self):
        """
        returns the name of the expansion
        """
        return self.name


    def create_new_node_attribute(self, node_id= None):
        """
        returns an attribute of type PianaGraphNodeAttribute with given characteristics
        """

        if node_id is None:
            raise ValueError("impossible to create a new node attribute for node_id= None")


        new_node_attribute = PianaGraphNodeAttribute(proteinPiana_value=node_id, mem_mode = "onDemand", piana_access = self.piana_access)

        return new_node_attribute

    def create_new_edge_attribute(self, edge_db_id= None, expansion_type= None, propagated_from_node= None, propagated_from_edge_id=None,
                              propagated_from_db_edge_id=None, extended_from_node= None):
        """
        returns attribute of type PianaGraphEdgeAttribute with given characteristics

        expansion_type, propagated_from_node, propagated_from_edge are mandatory when defining a propagation

        expansion_type, extended_from_node are mandatory when defining an extension

        if edge_db_id is not None, it means we are creating an attribute for a given interactionPiana. In propagations edge_db_id will be None
        because propagated edges do not exist as such in the database. Extensions will have an edge_db_id
        
        """
        # checking coherence of arguments
        # commented to make it faster...
        #if expansion_type is not None:
        #    if (propagated_from_node is not None and propagated_from_edge_id is None) or  (propagated_from_node is None and propagated_from_edge_id is not None):
        #        raise ValueError("in create_new_attribute, arguments not coherent (1)")
        #    elif extended_from_node is not None and (propagated_from_node is not None or propagated_from_edge_id is not None):
        #        raise ValueError("in create_new_attribute, arguments not coherent (2)")
        #    elif extended_from_node is None and propagated_from_node is None:
        #        raise ValueError("in create_new_attribute, arguments not coherent (3)")
        #        
        #else:
        #    raise ValueError("in create_new_attribute, expansion_type cannot be None")
        

        new_edge_attribute = PianaGraphEdgeAttribute(interactionPiana_value= edge_db_id, mem_mode = "onDemand", piana_access = self.piana_access)

        if extended_from_node is not None:
            new_edge_attribute.set_extension(expansion_type= expansion_type,
                                        extended_from_node= extended_from_node)
        
        elif propagated_from_node is not None:
            new_edge_attribute.set_propagation(expansion_type= expansion_type,
                                          propagated_from_node= propagated_from_node,
                                          propagated_from_edge_id= propagated_from_edge_id,
                                          propagated_from_db_edge_id= propagated_from_db_edge_id)
            
        else:
            raise ValueError("Missing parameters to create_new_attribute")
        
        return new_edge_attribute
  
    def get_partner_ids(self, node_id):
        """
        Returns a list of partners (proteinPiana) of node_id

        returns empty list if nothing is found

        """
        # TO CHECK!!! It looks as if I am not using this method anywhere... why?
        
        return self.piana_access.get_all_partners(proteinPiana_value = node_id,
                                                  use_self_ints=self.use_self_ints,
                                                  list_source_dbs= self.list_source_dbs,
                                                  inverse_dbs= self.inverse_dbs,
                                                  list_source_methods= self.list_source_methods,
                                                  inverse_methods= self.inverse_methods)   
       
    def get_db_edge_ids_from_node(self, node_id):
        """
        Returns a list of interactionPiana (which is the db_edge_id describing interactions between node_id and all its partners

        returns empty list if nothing is found
        
        """
        list_partners= self.piana_access.get_all_partners(proteinPiana_value = node_id,
                                                          use_self_ints=self.use_self_ints,
                                                          list_source_dbs= self.list_source_dbs,
                                                          inverse_dbs= self.inverse_dbs,
                                                          list_source_methods= self.list_source_methods,
                                                          inverse_methods= self.inverse_methods)

        list_edge_ids = []
        for one_partner in list_partners:
            list_edge_ids.append( self.piana_access.get_interactionPiana(proteinPianaA_value= node_id,
                                                                         proteinPianaB_value= one_partner,
                                                                         list_source_dbs= "all", inverse_dbs="no",
                                                                         list_source_methods= "all", inverse_methods="no")) # no need to place restrictions
                                                                                                                            # because they were placed before
                                                                                                                            # when retrieving the partners

        
        return list_edge_ids

    def get_edges_attributes_from_node(self, node_id):
        """
        Returns a list of PianaGraphEdgeAttribute objects describing interactions between node_id and all its partners

        returns empty list if nothing was found
        
        """
        list_interactions_attributes = []
        all_interactionPiana = self.get_db_edge_ids_from_node(node_id=node_id)

        for interactionPiana in all_interactionPiana:
            
            attribute = PianaGraphEdgeAttribute(interactionPiana_value= interactionPiana, mem_mode="onDemand", piana_access = self.piana_access)
            
            list_interactions_attributes.append(attribute)
        # END OF for interactionPiana in all_interactionPiana:
                                             
        
        return list_interactions_attributes
    
     
    def get_partner_node_id(self, db_edge_id= None , node_id= None):
        """
        Returns the partner node (a protein) of "node_id" in pianaDB interaction described by PianaGraphEdgeAttribute edge_attribute
        
        """
        if db_edge_id is None or node_id is None:
            raise ValueError("Parameters to get_partner_node_id cannot be None")
        
        return self.piana_access.get_partner(interaction_id= db_edge_id,
                                             proteinPiana_value = node_id)

    
    def get_external_code(self, node_id = None, code_type= None, alternative_code_types= []):
        """
        returns the external code for node_id, first trying with code_type and then with alternative_code_types

        if no code of code_type is found, string returned with alternative code type will follow the format "type_of_code_found:protein_code_value"
        """
        list_ext_codes = self.piana_access.get_list_protein_external_codes(proteinPiana= node_id,
                                                                           protein_type_name= code_type,
                                                                           alternative_type_names=alternative_code_types)
        
        if list_ext_codes:
            ext_code = list_ext_codes[0] # use only one code for clarity
        else:
            ext_code = "no_code_found"
            
        return ext_code

     
    def class_test(self, node_id = None, class_name = None):
        """
        tests whether the node_id belongs to class_name or not

        returns 1 if class test is passed, 0 otherwise

        In Piana this means, testing if proteinPiana node_id is of tax_id 'class_name'

        'class_name' must be an integer (unless it is set to 'all')
        """

        passes_species_test = 0

        
        if class_name == "all":
            passes_species_test = 1
        else:
            if class_name == 0:
                # when tax_id fixed is 0, it means there is no tax id to be fixed
                passes_species_test = 1
            else:
                list_protein_taxonomies = self.piana_access.get_protein_taxonomy_ids(proteinPiana_value = node_id)

                if int(class_name) in list_protein_taxonomies:
                    passes_species_test = 1
                
            # END OF else (if class_name == 0:):
        # END OF else: (if class_name == "all":)
        
        return passes_species_test


        
# ---------------------------------------------------------------------------------------------
# classes that implement specific expansions of piana
# ---------------------------------------------------------------------------------------------
# ATTENTION: if you add a new Expansion, you must add it as well to PianaApi.expand_piana_graph_interactions
#               --> add the creation of your object based on the expansion_type chosen by user
#               --> add a key to PianaGlobals.expansion_types
# ---------------------------------------------------------------------------------------------


class ExpansionSameEC(PianaNodeExpansion):
    """

    This class is used to propagate interactions from protein A to protein B if protein A and protein B share the same EC code

    methods have the same name (to make trasparent kind of expansion) but are different from other expansions to use EC as expansion characteristic


    """
    
    def __init__(self, piana_access_object, use_self_ints="yes", list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):
        
        self.piana_access = piana_access_object

        PianaNodeExpansion.__init__(self, piana_access_object = self.piana_access, name = "ExpansionSameEC",
                                    use_self_ints=use_self_ints, list_source_dbs=list_source_dbs, inverse_dbs=inverse_dbs,
                                    list_source_methods=  list_source_methods, inverse_methods=inverse_methods)
      

    def get_sharing_nodes(self, node_id):
        """
        returns a list of nodes that share ec with node_id (node_id == proteinPiana in PianaGraph networks)

        returns empty list if nothing found
        """

        proteins_sharing_ec = self.piana_access.get_proteins_sharing_ec(proteinPiana_value= node_id)
                                                                                          
        return  proteins_sharing_ec



class ExpansionSameCog(PianaNodeExpansion):
    """

    This class is used to propagate interactions from protein A to protein B if protein A and protein B share the same COG code

    methods have the same name (to make trasparent kind of expansion) but are different from other expansions to use COG as expansion characteristic

    """
    
    def __init__(self, piana_access_object, use_self_ints="yes", list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):

        self.piana_access = piana_access_object
        
        PianaNodeExpansion.__init__(self, piana_access_object = self.piana_access, name = "ExpansionSameCog",
                                    use_self_ints=use_self_ints, list_source_dbs=list_source_dbs, inverse_dbs=inverse_dbs,
                                    list_source_methods=  list_source_methods, inverse_methods=inverse_methods)

    def get_sharing_nodes(self, node_id):
        """
        returns a list of nodes that share cog with node_id (node_id == proteinPiana in PianaGraph networks)

        returns empty list if nothing found
        """
                
        proteins_sharing_cog = self.piana_access.get_proteins_sharing_cog(proteinPiana_value= node_id)
                                                                                          
        return  proteins_sharing_cog

    def get_partners_of_nodes_sharing_characteristic(self, node_id):
        """
        Returns a list of partner proteins of proteins that share cog with proteinPiana node_id
        
        returns empty list if nothing found

        ATTENTION: currently not being used!!!
        
        """
        return self.piana_access.get_partners_of_proteins_sharing_cog(proteinPiana_value = node_id,
                                                                      list_source_dbs= self.list_source_dbs,
                                                                      inverse_dbs= self.inverse_dbs,
                                                                      list_source_methods= self.list_source_methods,
                                                                      inverse_methods= self.inverse_methods)      

class ExpansionSameScop(PianaNodeExpansion):
    """

    This class is used to propagate interactions from protein A to protein B if protein A and protein B share the same SCOP family code

    methods have the same name (to make trasparent kind of expansion) but are different from other expansions to use SCOP as expansion characteristic

    """
    
    def __init__(self, piana_access_object, use_self_ints="yes", list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):
        
        self.piana_access = piana_access_object
        
        PianaNodeExpansion.__init__(self, piana_access_object = self.piana_access, name = "ExpansionSameScop",
                                    use_self_ints=use_self_ints, list_source_dbs=list_source_dbs, inverse_dbs=inverse_dbs,
                                    list_source_methods=  list_source_methods, inverse_methods=inverse_methods)

    def get_sharing_nodes(self, node_id):
        """
        returns a list of nodes that share scop family code with node_id (node_id == proteinPiana in PianaGraph networks)

        returns empty list if nothing found
        """
                
        proteins_sharing_scop = self.piana_access.get_proteins_sharing_scop(proteinPiana_value= node_id)
                                                                                          
        return  proteins_sharing_scop

    
class ExpansionSameInterpro(PianaNodeExpansion):
    """

    This class is used to propagate interactions from protein A to protein B if protein A and protein B share the same interpro code

    methods have the same name (to make trasparent kind of expansion) but are different from other expansions to use interPro as expansion characteristic

    """
    
    def __init__(self, piana_access_object, use_self_ints="yes", list_source_dbs= "all", inverse_dbs="no", list_source_methods= "all", inverse_methods="no"):
        
        self.piana_access = piana_access_object
        
        PianaNodeExpansion.__init__(self, piana_access_object = self.piana_access, name = "ExpansionSameInterpro",
                                    use_self_ints=use_self_ints, list_source_dbs=list_source_dbs, inverse_dbs=inverse_dbs,
                                    list_source_methods=  list_source_methods, inverse_methods=inverse_methods  )

    def get_sharing_nodes(self, node_id):
        """
        returns a list of nodes that share interpro code with node_id (node_id == proteinPiana in PianaGraph networks)

        returns empty list if nothing found
        """
                
        proteins_sharing_interpro = self.piana_access.get_proteins_sharing_interpro(proteinPiana_value= node_id)
                                                                                          
        return  proteins_sharing_interpro


class PianaEdgeExpansion(EdgeExpansion):
    """
    Not implemented: no edge expansions being carried out for Piana
    """
    pass

