"""
 File       : PianaGraphNodeAttribute.py
 Author     : R. Aragues & D. Jaeggi
 Creation   : 2003
 Contents   : class PianaGraphNodeAttribute (attribute specific to Piana for a Node)
 Called from: 

=======================================================================================================

This file implements class PianaGraphNodeAttribute, describing the attribute of a GraphNode, specific to PianaGraph 

"""

# PianaGraphNodeAttribute.py: class PianaGraphNodeAttribute (attribute specific to Piana for a Node)
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues
import copy

from GraphNodeAttribute import *
from PianaDB import *

# --------------------------------
# Class PianaGraphNodeAttribute
# --------------------------------
class PianaGraphNodeAttribute(GraphNodeAttribute):
    
    """
    Class that implements attributes for Piana Nodes: accessing node values, setting values, ...
    
    """
    def __init__(self, proteinPiana_value,  piana_access = None, mem_mode = "onDemand"):
        
        """
        "proteinPiana_value" is the proteinPiana of this node

        "piana_access" is a PianaDBaccess object used to access the database
        
        "mem_mode" can be:
        --> onDemand: values are retrieved from DB only when they are needed
        --> inMemory: values are retrieved from DB at initilisation

        
        """

        self.db_access = piana_access    # setting the connection to the piana DB through class PianaDB

        # features of the node. All information about the characteristics particular to this type of node should be contained here
        #   -> characteristics typical of all nodes (root, expanded, ...) are kept at the GraphNode level
        self.proteinPiana = int(proteinPiana_value)
        self.molecularWeight = None 
        self.isoelectricPoint = None


        # Attention: the following variables will only be used when 'unifiying' nodes!
        #            these variables are not used when building PianaGraph networks, only when printing results
        #            to fuse into a single node attribute all the information associated to proteinPianas that have
        #            the same external code
        self.associated_proteinPianas = {} 
        self.dic_functions = {}
        self.dic_descriptions = {}
        self.dic_taxonomies = {}
        self.dic_cell_location = {}
        self.expression = None
        self.keywords_appearing = []
        self.fitness_scores = []
        self.union_neighbours = {}
        

        if mem_mode == "onDemand":
            pass
            
        elif mem_mode == "inMemory":
            self._set_molecular_weight() 
            self._set_isoelectric_point() 
            
        else:
            raise ValueError(""" mem_mode set to incorrect value  %s(possible values are "onDemand" and "inMemory" """ %(mem_mode))

      
    def __deepcopy__(self, memo):
        
        clone = copy.copy(self) # Make a shallow copy
        for name, value in vars(self).iteritems():
            if name != "db_access":
                setattr(clone, name, copy.deepcopy(value, memo))
            else:
                setattr(clone, name, value)

        return clone
  
    
    def __getstate__(self):

        odict = self.__dict__.copy() # copy the dict since we change it
        return odict

 
    def __setstate__(self, dict):

        self.__dict__ = dict
        dict.__class__.__init__(dict)
        
    def __getnewargs__(self):

        return (self.proteinPiana,  self.db_access)

    # --------------------------------
    # PianaGraphNodeAttribute "set" methods
    # --------------------------------
    #
    # Methods to set values of PianaGraphNodeAttributes
    #
    #
    
    def merge_attribute(self, attribute_object, ignore_ids=0):
        """
        merges existing attribute with atribute_object passed as argument

        certain characteristics of the node will only be merged in ignore_ids is 1,
        since we are interested in having those characteristics filled
        only when we are unifying the nodes (ie. the operation by which PIANA
        decides which name will it use for the protein)  
        """
        if not ignore_ids:
            # 'ignore_ids' is used to avoid conflicts when unifying the network in PianaGraph, because we want to merge nodes with different proteinPianas
            if self.get_proteinPiana() != attribute_object.get_proteinPiana():
                raise ValueError("Trying to merge two node attributes with different proteinPiana: current is %s (%s) and merging is %s (%s)" %(
                    self.get_proteinPiana(),
                    type(self.get_proteinPiana()),
                    attribute_object.get_proteinPiana(),
                    type(attribute_object.get_proteinPiana()))  )

        else:
            # this is only executed when unifying a node

            # merge the list of proteinPianas associated to the unified node
            for proteinPiana in attribute_object.get_associated_proteinPianas():
                self.associated_proteinPianas[proteinPiana] = None
            
            # merge the list of descriptions
            for description in attribute_object.get_descriptions():
                self.dic_descriptions[description] = None
                
            # merge the list of functions
            for function in attribute_object.get_functions():
                self.dic_functions[function] = None
                
            # merge the list of taxonomies
            for taxonomy in attribute_object.get_dic_taxonomies():
                self.dic_taxonomies[taxonomy] = None

            # merge the list of cell locations
            for cell_location in attribute_object.get_dic_cell_location():
                self.dic_cell_location[cell_location] = None

            # merge the expressions
            if self.expression == attribute_object.expression:
                pass
            elif self.expression is None:
                self.expression = attribute_object.expression
            elif attribute_object.expression is None:
                pass
            else:
                sys.stderr.write("Weird... different expressions for nodes with the same ext_code: %s and %s\n" %(self.expression, attribute_object.expression))

            for keyword in attribute_object.get_keywords_appearing():
                if keyword not in self.keywords_appearing:
                    self.keywords_appearing.append(keyword)

            for fitness_score in attribute_object.get_fitness_scores():
                if fitness_score not in self.fitness_scores:
                    self.fitness_scores.append(fitness_score)

            self.union_neighbours.update(attribute_object.union_neighbours)
        # END OF else: (if not ignore_ids:)
    
    
    def _set_molecular_weight(self):
        
        self.molecularWeight = self.db_access.get_protein_mw(proteinPiana= self.proteinPiana)
    
    def _set_isoelectric_point(self):
        
        self.isoelectricPoint =  self.db_access.get_protein_ip(proteinPiana= self.proteinPiana)

    # 
    # the following methods are used for unifying nodes, not for normal PianaGraph construction
    # 

    def _set_associated_proteinPianas(self, proteinPiana= None):
        self.associated_proteinPianas[proteinPiana] = None
        
    def _set_dic_taxonomies(self):
        
        list_taxonomies =  self.db_access.get_protein_taxonomy_ids(proteinPiana_value = self.proteinPiana )
        for taxonomy in list_taxonomies:
            self.dic_taxonomies[taxonomy] = None
            
    def _set_dic_cell_location(self):
        
        list_cell_locations =  self.db_access.get_protein_subcellularLocation(proteinPiana_value = self.proteinPiana )
        for cell_location in list_cell_locations:
            self.dic_cell_location[cell_location] = None
            
    def _set_expression(self, expression = None):
        self.expression = expression
        
    def _set_keywords_appearing(self, user_keywords = []):
        self.keywords_appearing = self.db_access.check_keywords_in_protein(list_proteinPiana=[self.proteinPiana], keywords=user_keywords)
        
    def _set_fitness_scores(self):
        self.fitness_scores = self.db_access.get_fitness_score_reaction_conditions(proteinPiana_value= self.proteinPiana)
        
    def _set_dic_descriptions(self):

        list_descriptions =  self.db_access.get_protein_description(proteinPiana_value = self.proteinPiana)

        for description in list_descriptions:
            self.dic_descriptions[description] = None
            
    def _set_dic_functions(self):

        list_functions =  self.db_access.get_protein_function(proteinPiana_value = self.proteinPiana)

        for function in list_functions:
            self.dic_functions[function] = None

    def _set_union_neighbours(self, neighbours_dic={}):

        self.union_neighbours.update(neighbours_dic)

    # --------------------------------
    # PianaGraphNodeAttribute "get" methods
    # --------------------------------
    #
    # Methods to get values from PianaGraphNodeAttributes
    #
    # If the value requested is not already in memory (i.e. mem_mode is onDemand), it is retrieved from the database


    def get_node_id(self):
        """
        overwrittes the method in GraphNodeAttribute, identifying this attribute with the proteinPiana
        """
        return self.proteinPiana 
    
    def get_proteinPiana(self):
                  
        return self.proteinPiana  # proteinPiana cannot be None


    def get_molecular_weight(self):
        """
        Returns the molecular weight of the protein from the database
    
        """
        if self.molecularWeight is None:
            self._set_molecular_weight()
            
        return self.molecularWeight
    
    def get_isoelectric_point(self):
        """
        Returns the molecular weight of the protein 

        """
        if self.isoelectricPoint is None:
            self._set_isoelectric_point()
            
        return self.isoelectricPoint

    # 
    # the following methods are used for unifying nodes, not for normal PianaGraph construction
    # 

    def get_associated_proteinPianas(self):
                  
        return self.associated_proteinPianas 
    
    def get_dic_taxonomies(self):
        """
        Returns a dictionary {tax:None, tax:None, ...} with the taxonomies for this node
        """
        if not self.dic_taxonomies:
            self._set_dic_taxonomies()
            
        return self.dic_taxonomies

    def get_dic_cell_location(self):
        """
        Returns a dictionary {location:None, location:None, ...} with the cellular locations for this node
        """
        if not self.dic_cell_location:
            self._set_dic_cell_location()
            
        return self.dic_cell_location
    
    def get_keywords_appearing(self):
        """
        returns the list of user keywords that appear in the name, description or function of this protein
        """
        return self.keywords_appearing
    
    def get_fitness_scores(self):
        """
        returns the list of user keywords that appear in the name, description or function of this protein
        """
        return self.fitness_scores
    
    def get_expression(self):
        """
        returns the list of descriptions for this node
        """
        return self.expression
    
    def get_descriptions(self):
        """
        returns the list of descriptions for this node
        """
        return self.dic_descriptions.keys()
    
    def get_functions(self):
        """
        returns the list of functions for this node
        """
        return self.dic_functions.keys()

    def get_union_neighbours(self):
        """
        returns the list of neighbours for the unified node
         (ie. list of neighbours for all nodes that have been 'unified' into this node attribute)
        """
        return self.union_neighbours.keys()
