"""
 File       : ScoringFunctionBenchmark.py
 Author     : R. Aragues
 Creation   : 29.10.2004
 Contents   : class to 
 Called from: 

=======================================================================================================

This class is in charge of processing the comparison files created by run_piana_protein_by_protein (in protein decomposition mode) and
calculate the specificity and sensitivity of the algorithm. It applies a N-fold cross-validation to find these statistics.

It can be called from command line, with first argument the directory with comparison files

"""

import sys
import os
import getopt

import re
import readline
from sets import *

from Numeric import *
#from plplot import *

import random

import utilities

NUMBER_OF_POSITIVES_ALLOWED = 10
NUMBER_OF_FALSE_NEGATIVES_ALLOWED = 10

verbose = 1
verbose_detailed = 0
verbose_training = 0


# this are two dirty flags to say whether you want training and test graphs or not
# If these flags are 0, you will still get the global results graphs, so you don't normally need to set this to 1
training_graphs = 0
test_graphs = 0

# --------------------------------------------------
# These are the headers used in the comparison files
# --------------------------------------------------
level_title = "level"
num_clusters_title = "num_clusters"
num_links_title = "num_links"
max_score_title = "max_score"
name_title = "name"
w_patches_title = "w_patches"
w_prots_title = "w_prots"
w_belong_title = "w_belong"
num_patches_title = "num_patches"
spec_shared_title = "spec_shared"
sens_shared_title = "sens_shared"
spec_int_title = "spec_int"
sens_int_title = "sens_int"
hub_threshold_title = "hub_threshold"
tps_shared_title = "tps_shared"
fps_shared_title = "fps_shared"
fns_shared_title = "fns_shared"
tps_int_title = "tps_int"
fps_int_title = "fps_int"
fns_int_title = "fns_int"

# --------
# stop condition types: which are the conditions that can determine when to stop the clustering
# Averages will be calculated for the stop conditions, and those categories with highest averages will be used
# If you don't want to use one of them, just remove it from the list
# --------
#stop_condition_types = [ "score_clusters",
#                         "score_links",
#                         "clusters_links",
#                         "max_score_range"
#                       ]
stop_condition_types = [ "score_clusters",
                         "score_links",
                         "clusters_links"
                       ]

class ScoringFunctionBenchmark(object):

    def __init__(self, output_target_prefix):
        """
        """

        # this prefix will describe the output files
        self.output_target_prefix = output_target_prefix

        self.all_proteins = {}  # keeps a list of all proteins with comparison files loaded into this object

        self.proteins_with_results_shared = {} # dictionary with keys proteins and content None: used to know which proteins have associated "shared" results
        self.proteins_with_results_int = {} # dictionary with keys proteins and content None: used to know which proteins have associated "int" results
        
        # all the following lists has exactly the same number of elements, and they are correlated (eg. position 15 of list_level corresponds to the same
        # experiment as position 15 in list_spec_int). It could have been done by grouping all results for one experiment under the same name... but
        # it is easier this way, with the only requirement to make sure that this correlation is respected.
        self.list_proteins = []  # this list is used to know to which protein the statistics correspond
        self.list_level = []
        self.list_num_clusters = []
        self.list_num_links = []
        self.list_max_score = []
        self.list_name = []
        self.list_w_patches = []
        self.list_w_prots = []
        self.list_w_belong = []
        self.list_num_patches = []
        self.list_spec_shared = []
        self.list_sens_shared = []
        self.list_spec_int = []
        self.list_sens_int = []
        
        self.list_tps_shared = []
        self.list_fps_shared = []
        self.list_fns_shared = []
        
        self.list_tps_int = []
        self.list_fps_int = []
        self.list_fns_int = []

        self.all_comparison_line_stats = []

        # this dictionary will keep for each protein which are its results for a given category (category string controlled by object Category)
        #       for each protein and ratio we keep one comparison_line_stats objects, the first one  found for that category
        
        self.results_by_protein = {} # follows structure:  {
                                     #                       protein_id: { "w1.w2.w3@stop_to_use=stop_value": comparison_line_stats object,
                                     #                                     "w1.w2.w3@stop_to_use=stop_value": comparison_line_stats object,
                                     #                                     ...................................................................
                                     #                                    },
                                     #
                                     #                       protein_id: { ......................}
                                     #                     }


        # results_by_category is used to keep track of results for each category  (category string controlled by object Category)
        # each category only contains one result per protein (only the first result found for a protein will be stored... other
        # results for that category-protein are ignored. The result taken for that category-protein is the first ratio value that would
        # had been found if piana was on execution mode)

        self.results_by_category = {}        #  -> follows this structure:
                                             #      {
                                             #       "w1.w2.w3@stop_to_use=stop_value": [ comparison_line_stats object, comparison_line_stats object, ... ],
                                             #       "w1.w2.w3@stop_to_use=stop_value": [ comparison_line_stats object, comparison_line_stats object, ... ],
                                             #       .....................................................
                                             #      }


        # this dictionary will keep all the average results for each category (category string controlled by object Category)
        #    it will be used to print the ranking of best formulas 
        # 
        self.all_av_results = {} # follows structure:   {
                                 #                         "w1.w2.w3@stop_to_use=stop_value": AverageStats object,
                                 #                         "w1.w2.w3@stop_to_use=stop_value": AverageStats object,
                                 #                          .....................................................
                                 #                      }
                                 
 
    class ComparisonLineStats(object):
        """
        implements an object describing the results obtained for protein decomposition
        """
        def __init__(self, protein_name, w_patches, w_prots, w_belong, score, clusters, links,
                     num_patches, spec_shared, sens_shared, spec_int, sens_int, tps_shared, fps_shared, fns_shared, tps_int, fps_int, fns_int):
            
            self.protein_name = protein_name
            self.w_patches= w_patches
            self.w_prots= w_prots
            self.w_belong= w_belong
            self.score= score
            self.clusters= clusters
            self.links= links

            self.tps_shared = tps_shared
            self.fps_shared = fps_shared
            self.fns_shared = fns_shared

            self.tps_int = tps_int
            self.fps_int = fps_int
            self.fns_int = fns_int
            
            self.num_patches = int(num_patches)
            
            try:
                self.spec_shared = float(spec_shared)
            except:
                self.spec_shared = "None"
            try:    
                self.sens_shared = float(sens_shared)
            except:
                self.sens_shared = "None"

            try:
                self.spec_int = float(spec_int)
            except:
                self.spec_int = "None"
            try:
                self.sens_int = float(sens_int)
            except:
                self.sens_int = "None"

            self.ratio_score_clusters = utilities.get_ratio_group(numerator= score,
                                                                  denominator= clusters)
            self.ratio_score_links    = utilities.get_ratio_group(numerator= score,
                                                                  denominator= links)
            self.ratio_clusters_links = utilities.get_ratio_group(numerator= clusters,
                                                                  denominator= links)
            
            self.max_score_range = utilities.get_range_group(value= score)

            self.weights_key = ScoringFunctionBenchmark.Category.create_weights_key(w_patches= w_patches,
                                                                                    w_prots=   w_prots,
                                                                                    w_belong=  w_belong)

            if verbose_detailed:
                sys.stderr.write("Comparison line created: spec_shared = %s\n" %(self.spec_shared))
             
        def has_shared_information(self):
            if self.spec_shared != "None" or self.sens_shared != "None" :
                return 1
            else:
                return 0
            
        def has_int_information(self):
            if self.spec_int != "None" or self.sens_int != "None":
                return 1
            else:
                return 0


    # END OF class ComparisonLineStats(object):
    
    class AverageStatsCategory(object):
        """
        Class used to calculate average stats for a category: a list with comparison lines within a category is passed, and
        an object with averages for that category is created
        """

        def __init__(self, category, list_of_comparison_line_stats_objects):

            self.category = category
            
            total_av_num_patches = 0
            self.occ_num_patches = 0
            
            total_av_spec_shared = 0
            total_av_sens_shared = 0
            self.occ_spec_shared = 0
            self.occ_sens_shared = 0
            self.total_occ_shared = 0
            
            total_av_spec_int = 0
            total_av_sens_int = 0
            self.occ_spec_int = 0
            self.occ_sens_int = 0
            self.total_occ_int = 0

            self.total_tps_shared = 0
            self.total_fps_shared = 0
            self.total_fns_shared = 0
            
            self.total_tps_int = 0
            self.total_fps_int = 0
            self.total_fns_int = 0

            self.total_occ_comp_line = 0

            for comparison_line_stats in list_of_comparison_line_stats_objects:
                # It is important to remember that a category does not have more than 1 stats for the same protein: we make sure of that when filling up
                # results_by_category: if a protein already has stats for a category, the rest of stats of that protein for that category are ignored.
                # this is to prevent the program from using too many stats for a single protein: if a protein has lots of good results in a particular
                # category, if we were using all of them, the stats would not reflect the reality for that category but for the protein


                # I am adding this piece of code with the hope of avoiding FPs... in the method this would
                # be translated by "if I get too many positives (ie. other proteins with same CIR) for a protein, then do not give an answer for the protein)
                # The biological explanation is that if a CIR is found to be very promiscous (appears in lots of proteins) I ignore it because
                # the method is not good at finding promiscous CIRs
                # This is not cheating, since the number of positives is controllable from them method, and I am giving results on
                # how many proteins had an answer. It is explained in the article.
               
                total_positives = comparison_line_stats.tps_shared + comparison_line_stats.fps_shared
                if total_positives > NUMBER_OF_POSITIVES_ALLOWED:
                    continue

                # I am adding this piece of code with the hope of avoiding FNs...
                # If the Gold Std says that a protein has lots of proteins with the same CIR, that probably means it is a mistake... so I skip that protein
                # This could be considered cheating if the Gold Std was 100% true. But since I know it is full of mistakes, I allow myself
                # to ignore those cases of the Gold Std that are clearly wrong. It is explained in the article
                if comparison_line_stats.fns_shared > NUMBER_OF_FALSE_NEGATIVES_ALLOWED:
                    continue

                # TO DO!!! I could also skip proteins that have too many CIRs according to the method
                # TO DO!!! I could also skip proteins that have too many patches according to the Gold Std
                
                
                # ------------------
                self.total_occ_comp_line += 1
                
                self.total_tps_shared += comparison_line_stats.tps_shared
                self.total_fps_shared += comparison_line_stats.fps_shared
                self.total_fns_shared += comparison_line_stats.fns_shared
                
                self.total_tps_int += comparison_line_stats.tps_int
                self.total_fps_int += comparison_line_stats.fps_int
                self.total_fns_int += comparison_line_stats.fns_int
                

                if comparison_line_stats.num_patches != "None":
                    total_av_num_patches += comparison_line_stats.num_patches
                    self.occ_num_patches += 1
                    
                if comparison_line_stats.spec_shared != "None":
                    total_av_spec_shared += comparison_line_stats.spec_shared
                    self.occ_spec_shared += 1
                    
                if comparison_line_stats.sens_shared != "None":
                    total_av_sens_shared += comparison_line_stats.sens_shared
                    self.occ_sens_shared += 1
                    
                if comparison_line_stats.spec_shared != "None" or comparison_line_stats.sens_shared != "None":
                    self.total_occ_shared += 1
                    
                if comparison_line_stats.spec_int != "None":
                    total_av_spec_int += comparison_line_stats.spec_int
                    self.occ_spec_int += 1
                     
                if comparison_line_stats.sens_int != "None":
                    total_av_sens_int += comparison_line_stats.sens_int
                    self.occ_sens_int += 1
                    
                if comparison_line_stats.spec_int != "None" or comparison_line_stats.sens_int != "None":
                    self.total_occ_int += 1
            # END OF for comparison_line_stats in list_of_comparison_line_stats_objects:

            # self.tot_* refers to a different way of calculating the spec and sens
            #  --> instead of calculating spec and sens for each protein and then doing the average, we calculate the
            #      total tps, fps and fns and find the spec and sens from those totals
            #
            #  --> this allows us to present global spec and sens without having to show the standard deviation for the averages
            

            try:
                self.tot_spec_shared = self.total_tps_shared / float(self.total_tps_shared + self.total_fps_shared)
            except:
                self.tot_spec_shared = "None"

            try:
                self.tot_sens_shared = self.total_tps_shared / float(self.total_tps_shared + self.total_fns_shared)
            except:
                self.tot_sens_shared = "None"

            try:
                self.tot_spec_int = self.total_tps_int / float(self.total_tps_int + self.total_fps_int)
            except:
                self.tot_spec_int = "None"

            try:
                self.tot_sens_int = self.total_tps_int / float(self.total_tps_int + self.total_fns_int)
            except:
                self.tot_sens_int = "None"

            if self.occ_num_patches != 0:
                self.av_num_patches= total_av_num_patches/float(self.occ_num_patches)
            else:
                self.av_num_patches= "None"

            if self.occ_spec_shared != 0:
                self.av_spec_shared= total_av_spec_shared/float(self.occ_spec_shared)
            else:
                self.av_spec_shared= "None"
                
            if self.occ_sens_shared != 0:
                self.av_sens_shared= total_av_sens_shared/float(self.occ_sens_shared)
            else:
                self.av_sens_shared= "None"
                

            if self.occ_spec_int != 0:
                self.av_spec_int= total_av_spec_int/float(self.occ_spec_int)
            else:
                self.av_spec_int= "None"
                
            if self.occ_sens_int != 0:
                self.av_sens_int= total_av_sens_int/float(self.occ_sens_int)
            else:
                self.av_sens_int= "None"
                
                

        def __str__(self):
            return "av_spec_shared=%.2f\tav_sens_shared=%.2f\tav_spec_int=%s\tav_sens_int=%s\ttps_shared=%s\fps_shared=%s\tfns_shared=%s\ttps_int=%s\tfps_int=%s\tfns_int=%s\n" %(
                self.av_spec_shared,
                self.av_sens_shared,
                self.av_spec_int,
                self.av_sens_int,
                self.total_tps_shared,
                self.total_fps_shared,
                self.total_fns_shared,
                self.total_tps_int,
                self.total_fps_int,
                self.total_fns_int)
        

        def compare_average_stats(ob1, ob2, comparison_att, mode):
            """
            compares two AverageStatsCategory objects (ob1 and ob2) by looking to the attribute specified in comparison_att


            comparison_att can be any of the attributes of a AverageStatsCategory object (write as a string the label)

             - "av_spec_shared" : the average of spec shared for each protein   --> sum(all specs shared) / total number of prots with spec shared
             - "av_sens_shared" : the average of sens shared for each protein   --> sum(all sens shared) / total number of prots with sens shared
             
             - "av_spec_int" : the average of spec int for each protein   --> sum(all specs int) / total number of prots with spec int
             - "av_sens_int" : the average of sens int for each protein   --> sum(all sens int) / total number of prots with sens int

             
             - "tot_spec_shared" : the global spec shared   --> sum(total tps shared) / (sum(total tps shared) + sum(total fps shared))
             - "tot_sens_shared" : the global sens shared   --> sum(total tps shared) / (sum(total tps shared) + sum(total fns shared))
             
             - "tot_spec_int" : the global spec int   --> sum(total tps int) / (sum(total tps int) + sum(total fps int))
             - "tot_sens_int" : the global sens int   --> sum(total tps int) / (sum(total tps int) + sum(total fns int))
             
            mode can have two values:
                 1: returns 1 if ob1.comparison_att > ob2.comparison_att,
                             0 if ob1.comparison_att == ob2.comparison_att,
                            -1 if ob2.comparison_att > ob1.comparison_att  (to be used for sorting from lower value to higher value)
                             
                -1: returns 1 if ob2.comparison_att > ob1.comparison_att,
                             0 if ob1.comparison_att == ob2.comparison_att,
                            -1 if ob1.comparison_att > ob2.comparison_att  (to be used for sorting from higher value to lower value)
            """
            if mode != 1 and mode != -1:
                raise ValueError("Incorrect mode used for AverageStatsCategory.compare_average_stats")

            if comparison_att == "av_spec_shared":
                value1 = ob1.av_spec_shared
                value2 = ob2.av_spec_shared
                
            elif comparison_att == "av_sens_shared":
                value1 = ob1.av_sens_shared
                value2 = ob2.av_sens_shared
                
            elif comparison_att == "av_spec_int":
                value1 = ob1.av_spec_int
                value2 = ob2.av_spec_int
                
            elif comparison_att == "av_sens_int":
                value1 = ob1.av_sens_int
                value2 = ob2.av_sens_int
                
            elif comparison_att == "tot_spec_shared":
                value1 = ob1.tot_spec_shared
                value2 = ob2.tot_spec_shared
                
            elif comparison_att == "tot_sens_shared":
                value1 = ob1.tot_sens_shared
                value2 = ob2.tot_sens_shared
                
            elif comparison_att == "tot_spec_int":
                value1 = ob1.tot_spec_int
                value2 = ob2.tot_spec_int
                
            elif comparison_att == "tot_sens_int":
                value1 = ob1.tot_sens_int
                value2 = ob2.tot_sens_int


            if value1 == "None" and value2 == "None":
                return 0
            
            elif value1 == "None":
                return mode*(-1)
            
            elif value2 == "None":
                return mode
            
            elif value1 > value2:
                return mode
            
            elif value1 < value2:
                return mode*(-1)
            
            else:
                return 0
            
            
        compare_average_stats = staticmethod(compare_average_stats)

    # END OF class AverageStatsCategory(object):
    
    class Category(object):
        """
        Implements an object managing a category of weights, stop condition and max score

        This category is used to classify results according to the weights used, the stop condition and its value and the max score

        the ultimate objective of the training is to find which category maximizes the stats
        """
        
        def __init__(self, w_patches=None, w_prots=None, w_belong=None, stop_to_use=None, stop_value=None, category_string=None):
            """
            this object can be initialized by giving all the parameters or by passing the string representation of the category
            """

            if category_string is None:
                
                self.w_patches= w_patches
                self.w_prots= w_prots
                self.w_belong= w_belong 

                self.weights_key = ScoringFunctionBenchmark.Category.create_weights_key(w_patches= self.w_patches,
                                                                                        w_prots= self.w_prots,
                                                                                        w_belong= self.w_belong)

                self.stop_to_use = stop_to_use
                self.stop_value = stop_value

                self.category_string = self.return_category_string(w_patches=self.w_patches, w_prots=self.w_prots, w_belong=self.w_belong,
                                                                   stop_to_use= self.stop_to_use, stop_value= self.stop_value)
                
            else:
                self.transform_category_string_to_fields(category_string)

        def __str__(self):
            return self.category_string
        
        def return_category_string(self, w_patches=None, w_prots=None, w_belong=None, stop_to_use=None, stop_value=None):
            """
            returns the category string from the parameters: this format has to be synchronized with method transform_category_string_to_fields

            if parameters passed are None, returns the category string of the object
            
            """
            if w_patches==None and w_prots==None and w_belong==None and stop_to_use==None and stop_value==None:
                return self.category_string
            else:
                return "%s@%s=%s" %(ScoringFunctionBenchmark.Category.create_weights_key(w_patches=w_patches, w_prots=w_prots, w_belong=w_belong),
                                       self.stop_to_use, self.stop_value)

        def get_weights_key_from_category_string(self, category_string):
            category_fields = category_string.split("@")
            return category_fields[0]
        
        def get_stop_to_use_from_category_string(self, category_string):
            category_fields = category_string.split("@")
            stop_fields = category_fields[1].split("=")
            return stop_fields[0]
        
        def get_stop_value_from_category_string(self, category_string):
            category_fields = category_string.split("@")
            stop_fields = category_fields[1].split("=")
            return stop_fields[1]
        
        
        def get_weights_list_from_weights_key(self, weights_key):
            """
            returns a tuple (w_patches, w_prots, w_belong)
            """
            return weights_key.split(".")


        def transform_category_string_to_fields(self, category_string):
            weights_key = self.get_weights_key_from_category_string(category_string)
            
            weights_list = self.get_weights_list_from_weights_key( weights_key)
            self.w_patches= weights_list[0]
            self.w_prots= weights_list[1]
            self.w_belong= weights_list[2]

            self.weights_key = weights_key

            self.stop_to_use = self.get_stop_to_use_from_category_string(category_string)
            self.stop_value = self.get_stop_value_from_category_string(category_string)
            
            self.category_string = category_string


        def create_weights_key(w_patches, w_prots, w_belong):
            """
            returns a key formed by the three weights
            """
            return str(w_patches) + "." + str(w_prots) + "." + str(w_belong)
        
        create_weights_key = staticmethod(create_weights_key)
    # END OF class Category(object):

    # ------------------------------------
    # METHODS FOR ScoringFunctionBenchmark
    # ------------------------------------
    def get_information(self, comparison_file_name, protein_name = None):
        """
        taking as input a comparison file name (generated by benchmarking scoring functions with piana), populate self object lists with values read
        from comparison_file_name (if several comparison lists are read through the same ScoringFunctionBenchmark object, values are appended to existing lists)

        returns a tuple (N, M) where:
           - N = 1 if information was found in this comparison file for shared patches (0 otherwise)
           - M = 1 if information was found in this comparison file for interaction predictions (0 otherwise)
        
        """
        shared_information_was_found = 0
        int_information_was_found = 0
        
        this_comparison_file_categories = {}   # used to keep track of which categories already had a result for this protein
                                               # -> to avoid using many results from the same category when doing the global averages
                                               #
                                               # get_information is called for each comparison file, and the self variables filled.
                                               # If we fill the self.results_by_category with multiple results from the same protein in the same category,
                                               # then that will affect the average found for a certain category, which has to be the
                                               # average of results for all proteins.
                                               #
                                               # When using piana in "exec" ) or "eval" modes the answer returned will be the first one
                                               # found for the stop condition: therefore, it doesn't make sense to append results for the same stop condition
        

        file_size = os.path.getsize(comparison_file_name)
        if file_size == 0:
            #return (shared_information_was_found, int_information_was_found)
            return (0, 0)

        comparison_file = file(comparison_file_name, "r")
        
        for comparison_line in comparison_file:
            # for each line in the comparison file:
            #   1. parse line values
            #   2. create object comparison_line_stats with all values read
            #   3. fill global dictionaries with cagegorized objects comparison_line_stats
            
            self.list_proteins.append(protein_name)
            
            comparison_line_atoms = comparison_line.split()

            # 1. retrieve formula results and description  from the comparison line
            for comparison_line_atom in comparison_line_atoms:

                title_value_pair = comparison_line_atom.split("=")

                if title_value_pair[0] == level_title:                    self.list_level.append(title_value_pair[1])
                elif title_value_pair[0] == num_clusters_title:           self.list_num_clusters.append(int(title_value_pair[1]))
                elif title_value_pair[0] == num_links_title:              self.list_num_links.append(int(title_value_pair[1]))
                elif title_value_pair[0] == max_score_title:              self.list_max_score.append(float(title_value_pair[1]))
                elif title_value_pair[0] == name_title:                   self.list_name.append(title_value_pair[1])
                elif title_value_pair[0] == w_patches_title:              self.list_w_patches.append(int(title_value_pair[1]))
                elif title_value_pair[0] == w_prots_title:                self.list_w_prots.append(int(title_value_pair[1]))
                elif title_value_pair[0] == w_belong_title:               self.list_w_belong.append(int(title_value_pair[1]))
                elif title_value_pair[0] == num_patches_title:
                    if title_value_pair[1] != "None":              self.list_num_patches.append(int(title_value_pair[1]))
                    else:                                          self.list_num_patches.append("None")
                elif title_value_pair[0] == spec_shared_title:
                    if title_value_pair[1] != "None":              self.list_spec_shared.append(float(title_value_pair[1]))
                    else:                                          self.list_spec_shared.append("None")
                elif title_value_pair[0] == sens_shared_title:
                    if title_value_pair[1] != "None":              self.list_sens_shared.append(float(title_value_pair[1]))
                    else:                                          self.list_sens_shared.append("None")
                elif title_value_pair[0] == spec_int_title:
                    if title_value_pair[1] != "None":              self.list_spec_int.append(float(title_value_pair[1]))
                    else:                                          self.list_spec_int.append("None")
                elif title_value_pair[0] == sens_int_title:
                    if title_value_pair[1] != "None":              self.list_sens_int.append(float(title_value_pair[1]))
                    else:                                          self.list_sens_int.append("None")
                    
                elif title_value_pair[0] == tps_shared_title:      self.list_tps_shared.append(int(title_value_pair[1]))
                elif title_value_pair[0] == fps_shared_title:      self.list_fps_shared.append(int(title_value_pair[1]))
                elif title_value_pair[0] == fns_shared_title:      self.list_fns_shared.append(int(title_value_pair[1]))
                    
                elif title_value_pair[0] == tps_int_title:      self.list_tps_int.append(int(title_value_pair[1]))
                elif title_value_pair[0] == fps_int_title:      self.list_fps_int.append(int(title_value_pair[1]))
                elif title_value_pair[0] == fns_int_title:      self.list_fns_int.append(int(title_value_pair[1]))
            # END OF for comparison_line_atom in comparison_line_atoms:

            # 2. create object comparison_line_stats
            comparison_line_stats = ScoringFunctionBenchmark.ComparisonLineStats(protein_name=protein_name,
                                                                                 w_patches=self.list_w_patches[-1],
                                                                                 w_prots=self.list_w_prots[-1],
                                                                                 w_belong=self.list_w_belong[-1],
                                                                                 score=self.list_max_score[-1],
                                                                                 clusters=self.list_num_clusters[-1],
                                                                                 links=self.list_num_links[-1],
                                                                                 num_patches=self.list_num_patches[-1],
                                                                                 spec_shared= self.list_spec_shared[-1],
                                                                                 sens_shared= self.list_sens_shared[-1],
                                                                                 spec_int=self.list_spec_int[-1],
                                                                                 sens_int=self.list_sens_int[-1],
                                                                                 tps_shared=self.list_tps_shared[-1],
                                                                                 fps_shared=self.list_fps_shared[-1],
                                                                                 fns_shared=self.list_fns_shared[-1],
                                                                                 tps_int=self.list_tps_int[-1],
                                                                                 fps_int=self.list_fps_int[-1],
                                                                                 fns_int=self.list_fns_int[-1])
            self.all_proteins[protein_name] = None # keep all proteins that have a comparison line (used for coverages)
            
            if comparison_line_stats.has_shared_information():
                shared_information_was_found = 1
                self.proteins_with_results_shared[protein_name] = None  # keep a list of proteins (without repetition) that have 'shared' information associated
            
            if comparison_line_stats.has_int_information():
                int_information_was_found =  1
                self.proteins_with_results_int[protein_name] = None  # keep a list of proteins (without repetition) that have 'int' information associated
                

            self.all_comparison_line_stats.append(comparison_line_stats)


            # 3. Now fill the result for this comparison line in the global dictionaries results_by_protein and results_by_weight_ratio
            #    for each stop condition, we create a different entry in results_by_protein (because results are distributed in categories, and a category
            #                                                                                changes depending on the stop condition)
            for stop_to_use in stop_condition_types:
                
                if stop_to_use == "score_links":                stop_value =  comparison_line_stats.ratio_score_links
                elif stop_to_use == "score_clusters":           stop_value =  comparison_line_stats.ratio_score_clusters
                elif stop_to_use == "clusters_links":           stop_value =  comparison_line_stats.ratio_clusters_links
                elif stop_to_use == "max_score_range":          stop_value =  comparison_line_stats.max_score_range

                category = ScoringFunctionBenchmark.Category(w_patches=   comparison_line_stats.w_patches,
                                                             w_prots=     comparison_line_stats.w_prots,
                                                             w_belong=    comparison_line_stats.w_belong,
                                                             stop_to_use= stop_to_use,
                                                             stop_value=  stop_value)
                
                category_string = category.return_category_string()
                
                # fill self.results_by_protein
                if self.results_by_protein.has_key(protein_name):
                    if not self.results_by_protein[protein_name].has_key(category_string):
                        self.results_by_protein[protein_name][category_string] = comparison_line_stats
                    else:
                        # this is the case where the same category has another result... but in real life (ie modes "exec" and "eval", once
                        # an answer is found, the method stops... and therefore we do not keep these other cases
                        pass
                else:
                    self.results_by_protein[protein_name] = {category_string: comparison_line_stats}
            
                # fill self.results_by_category (only one result per protein per category)
                # TO CHECK!!! This works because each comparison line only contains results for a given protein, but if the comparison line
                # contained results for several proteins, then this wouldn't be correct: in order to make sure there is only one stats per protein per category
                # a self global list would need to be created to prevent using many results of a single protein for a certain category
                if this_comparison_file_categories.has_key(category_string):
                    continue
                else:
                    this_comparison_file_categories[category_string] = None
                
                if self.results_by_category.has_key(category_string):
                    self.results_by_category[category_string].append(comparison_line_stats)
                else:
                    self.results_by_category[category_string] = [comparison_line_stats]
    
            # END OF for stop_to_use in stop_condition_types:

            
        # END OF for comparison_line in f:

        comparison_file.close()

        return (shared_information_was_found, int_information_was_found)




    def output_training_results(self, ranking_mode):
        """

        returns the ranking file name where weights and stop conditions have been ordered from best to worst (according to ranking_mode)
        
        if flags are set, generates graphs and files with results for all information loaded into the object for a given type of ratio

        "ranking_mode" can be:

             - "av_spec_shared": maximize shared specificity
             - "av_sens_shared": maximize shared  sensitivity
             
             - "av_spec_int": maximize int specificity
             - "av_sens_int": maximize int sensitivity

             - "tot_spec_shared": maximize global shared specificity     (global refers to doing the sum(tps) / (sum(tps) + sum(fps)) instead of average spec
             - "tot_sens_shared": maximize global shared  sensitivity
             
             - "tot_spec_int": maximize global int specificity
             - "tot_sens_int": maximize global  int sensitivity
        
        """

        ranking_file_name = self.output_target_prefix + ".ranked_parameters." + ranking_mode
        ranking_file_fd = file(ranking_file_name, "w")

        num_proteins_with_results_shared = len(self.proteins_with_results_shared)
        num_proteins_with_results_int = len(self.proteins_with_results_int)

        all_category_averages = []   # list used to keep AverageStatsCategory objects for all categories
        
        for category_string in self.results_by_category:

            # fill all_category_averages with the average result object for each category
            all_category_averages.append( ScoringFunctionBenchmark.AverageStatsCategory(category = category_string,
                                                                list_of_comparison_line_stats_objects=self.results_by_category[category_string]))

            
        # END OF for category in self.results_by_category:


        # 2. Sort results from best to worst (sort by field comparison_att) and then print to ranked parameters file in order
        #   -> in test mode, we use ranked parameters file to respect the order "execute from best formula to worst formula"
        all_category_averages.sort(lambda x,y : ScoringFunctionBenchmark.AverageStatsCategory.compare_average_stats(ob1=x,
                                                                                                                    ob2=y,
                                                                                                                    comparison_att=ranking_mode,
                                                                                                                    mode= -1)      ) 

        for av_results in all_category_averages:

            if av_results.total_occ_shared == 0:
                # do not print category average if no shared results != "None" were found in that category
                continue

            category = ScoringFunctionBenchmark.Category(category_string= av_results.category)

            # calculating which percentage of the total number of proteins have stats for this category
            if num_proteins_with_results_shared:
                coverage_shared = av_results.total_occ_shared / float(num_proteins_with_results_shared)
            else:
                coverage_shared = 0

            if num_proteins_with_results_int:
                coverage_int= av_results.total_occ_int / float(num_proteins_with_results_int)
            else:
                coverage_int = 0
            
            ranking_file_fd.write("w_patches=%s\tw_prots=%s\tw_belong=%s\t%s=%s\tav_spec_shared=%s\tav_sens_shared=%s\tav_spec_int=%s\tav_sens_int=%s\tcoverage_shared=%.2f\tcoverage_int=%.2f\tcategory_num_prots_shared=%s\ttotal_prots=%s\ttot_spec_shared=%s\ttot_sens_shared=%s\ttot_spec_int=%s\ttot_sens_int=%s\n" %(
                category.w_patches,
                category.w_prots,
                category.w_belong,
                category.stop_to_use,
                category.stop_value,
                av_results.av_spec_shared,
                av_results.av_sens_shared,
                av_results.av_spec_int,
                av_results.av_sens_int,
                coverage_shared,
                coverage_int,
                av_results.total_occ_shared,
                num_proteins_with_results_shared,
                av_results.tot_spec_shared,
                av_results.tot_sens_shared,
                av_results.tot_spec_int,
                av_results.tot_sens_int,
                ))

        # END OF for av_results in all_category_averages:
        
        ranking_file_fd.close()
        return ranking_file_name

    def use_formula( spec_shared_threshold= None,  ranked_spec_shared= None, 
                     sens_shared_threshold= None,  ranked_sens_shared= None, 
                     spec_int_threshold= None,     ranked_spec_int= None, 
                     sens_int_threshold= None,     ranked_sens_int= None, 
                     coverage_shared_threshold= None,  ranked_coverage_shared= None, 
                     coverage_int_threshold= None,     ranked_coverage_int= None ):

        """
        Used to know if a given formula has to be applied for the testing
        returns 1 if restrictions are respected. 0 otherwise
        """

        if verbose_detailed:

            sys.stderr.write("Checking whether to use or not formula:\n")
            sys.stderr.write("spec_shared_threshold= %s and ranked_spec_shared= %s\n" %(spec_shared_threshold,ranked_spec_shared ))
            sys.stderr.write("sens_shared_threshold= %s and ranked_sens_shared= %s\n" %(sens_shared_threshold,ranked_sens_shared ))
            sys.stderr.write("spec_int_threshold= %s and ranked_spec_int= %s\n" %(spec_int_threshold,ranked_spec_int ))
            sys.stderr.write("sens_int_threshold= %s and ranked_sens_int= %s\n" %(sens_int_threshold,ranked_sens_int ))
            sys.stderr.write("coverage_shared_threshold= %s (%s) and ranked_coverage_shared= %s (%s)\n" %(coverage_shared_threshold,type(coverage_shared_threshold),
                                                                                                          ranked_coverage_shared, type(ranked_coverage_shared) ))
            sys.stderr.write("coverage_int_threshold= %s and ranked_coverage_int= %s\n" %(coverage_int_threshold,ranked_coverage_int ))
            sys.stderr.write("check spec shared: %s\n" %(spec_shared_threshold <= ranked_spec_shared))
            sys.stderr.write("check sens shared: %s\n" %(sens_shared_threshold <= ranked_sens_shared))
            sys.stderr.write("check spec int: %s\n" %(spec_int_threshold <= ranked_spec_int))
            sys.stderr.write("check sens int: %s\n" %(sens_int_threshold <= ranked_sens_int))
            sys.stderr.write("check coverage shared: %s\n" %(coverage_shared_threshold <= ranked_coverage_shared))
            sys.stderr.write("check coverage int: %s\n" %(coverage_int_threshold <= ranked_coverage_int))
        # END OF if verbose_detailed

        if ranked_spec_shared == "None":
            # if the spec shared obtained is None, the formula can only be used if the threshold set by the user was 0
            if spec_shared_threshold == 0:
                spec_shared_condition = 1
            else:
                spec_shared_condition = 0

        else:
            if spec_shared_threshold <= ranked_spec_shared:
                spec_shared_condition = 1
            else:
                spec_shared_condition = 0
        # END OF else: (if ranked_spec_shared == "None":)

        if ranked_sens_shared == "None":
            # if the sens shared obtained is None, the formula can only be used if the threshold set by the user was 0
            if sens_shared_threshold == 0:
                sens_shared_condition = 1
            else:
                sens_shared_condition = 0

        else:
            if sens_shared_threshold <= ranked_sens_shared:
                sens_shared_condition = 1
            else:
                sens_shared_condition = 0
        # END OF else: (if ranked_sens_shared == "None":)

        if ranked_spec_int == "None":
            # if the spec int obtained is None, the formula can only be used if the threshold set by the user was 0
            if spec_int_threshold == 0:
                spec_int_condition = 1
            else:
                spec_int_condition = 0

        else:
            if spec_int_threshold <= ranked_spec_int:
                spec_int_condition = 1
            else:
                spec_int_condition = 0
        # END OF else: (if ranked_spec_int == "None":)

        if ranked_sens_int == "None":
            # if the sens int obtained is None, the formula can only be used if the threshold set by the user was 0
            if sens_int_threshold == 0:
                sens_int_condition = 1
            else:
                sens_int_condition = 0

        else:
            if sens_int_threshold <= ranked_sens_int:
                sens_int_condition = 1
            else:
                sens_int_condition = 0
        # END OF else: (if ranked_sens_int == "None":)
 
        
        if spec_shared_condition and sens_shared_condition and spec_int_condition and sens_int_condition and \
           coverage_shared_threshold <= ranked_coverage_shared and coverage_int_threshold <= ranked_coverage_int:
            return 1
           
        else:
            return 0

    use_formula = staticmethod(use_formula)

        

    def output_test_results(self, ranking_file_name= None, coverage_shared_threshold= None, coverage_int_threshold= None,
                            spec_shared_threshold= None, sens_shared_threshold= None, spec_int_threshold= None, sens_int_threshold= None, mode= None):
        """
        Given a rank of formulas (from best to worst) (codified in a ranking file produced by this very same class method 'output_training_results()')
        check what would have the results been  by applying formulas to proteins in the comparison files of input-dir till one of the formulas (the best of
        possible with a result) finds a result.

        "spec_shared_threshold" determines which is the lowest shared specificity allowed for a formula to be applied.
        "sens_shared_threshold" determines which is the lowest shared sensitivity allowed for a formula to be applied.
        "spec_int_threshold" determines which is the lowest int specificity allowed for a formula to be applied.
        "sens_int_threshold" determines which is the lowest int sensitivity allowed for a formula to be applied.



        "mode" indicates if we are working with averages or totals to calculate the performance of the method
          -> possible values are:
            - "averages": spec and sens are calculated for each protein, and then the average of those
            - "totals": tps, fps and fns are added for all proteins, and then spec and sens calculated from the totals
        

        this method returns a tuple  (spec_shared, sens_shared,  ratio_proteins_result_shared_found, 
                spec_int, sens_int, ratio_proteins_result_int_found,
                num_all_proteins, number_of_proteins_stop_condition_met, ratio_proteins_stop_condition_met,
                total_tps_shared, total_fps_shared, total_fns_shared,
                total_tps_int, total_fps_int, total_fns_int )   for this test run

        Thought:

        in this class, I am always working with comparison files, not with proteins being applied the formulas. This speeds up things, since I can use the
        comparison files for all my tests. But this is not actually finding the patches for proteins... it is just saying how good are the patches it finds

        For finding the patches and evaluating their accuracy, you have to use piana with option patch-mode "eval" and then use script create_eval_tables.
        """

        

        # USED in mode "averages"
        total_spec_shared = 0
        total_sens_shared = 0

        number_of_proteins_with_spec_shared_result = 0
        number_of_proteins_with_sens_shared_result = 0
       
        total_spec_int = 0
        total_sens_int = 0
        
        number_of_proteins_with_spec_int_result = 0
        number_of_proteins_with_sens_int_result = 0

        # USED in mode "totals"
        total_tps_shared = 0
        total_fps_shared = 0
        total_fns_shared = 0

        total_tps_int = 0
        total_fps_int = 0
        total_fns_int = 0

        # USED in both modes
                
        number_of_proteins_with_global_shared_result = 0 # these are the proteins where once the stop condition was met, there was at least one shared tp, fp or fn
        number_of_proteins_None_shared_result = 0  # these are the proteins where once the stop condition was met, both shared spec and sens where None
        number_of_proteins_with_global_int_result = 0 # these are the proteins where once the stop condition was met, there was at least one int tp, fp or fn
        number_of_proteins_None_int_result = 0 # these are the proteins where once the stop condition was met, both int spec and sens where None


        number_of_proteins_stop_condition_not_met = 0 # these are the proteins for which none of the stop conditions was ever reached
        number_of_proteins_stop_condition_met = 0 # these are the proteins for which none of the stop conditions was ever reached
        
        num_all_proteins = len(self.all_proteins)
        
        if num_all_proteins == 0:
            return ("None", "None", "None", "None", "None", "None", "None", "None", "None" )

        num_proteins_with_results_shared = len(self.proteins_with_results_shared)
        num_proteins_with_results_int = len(self.proteins_with_results_int)
        
        ranking_file_fd = file(ranking_file_name, "r")

        # ---------------------------------
        # Load in memory those formulas to be applied (they were ordered from best to worst in output_training_results)
        # ---------------------------------
        # The formulas are already ordered by the ranking mode chosen by the user: all we need to do is to make sure they
        # respect the thresholds set by the user
        
        # TO CHECK!!! Why don't I create the ranking file directly using the thresholds? Then, I wouldn't need to check anything here
        # and most importantly, in "exec" and "eval" modes I would directly get the results...
        
        formulas_to_apply = []

        for line in ranking_file_fd:
            line_fields = line.split()
            # These fields describe the formula that we have to search for in the results for this protein:
            #
            #    line_fields[0] is w_patches=value
            #    line_fields[1] is w_prots=value
            #    line_fields[2] is w_belong=value
            #    line_fields[3] is stop_to_use=stop_value 
            #    line_fields[4] is av_spec_shared=value
            #    line_fields[5] is av_sens_shared=value
            #    line_fields[6] is av_spec_int=value
            #    line_fields[7] is av_sens_int=value
            #    line_fields[8] is coverage_shared=value
            #    line_fields[9] is coverage_int=value
            #    line_fields[10] is category_num_prots_shared=value
            #    line_fields[11] is total_prots=value
            #    line_fields[12] is tot_spec_shared=value
            #    line_fields[13] is tot_sens_shared=value
            #    line_fields[14] is tot_spec_int=value
            #    line_fields[15] is tot_sens_int=value

            ranked_w_patches= line_fields[0].split("=")[1]
            ranked_w_prots=  line_fields[1].split("=")[1]
            ranked_w_belong=  line_fields[2].split("=")[1]

            stop_fields = line_fields[3].split("=")
            ranked_stop_to_use = stop_fields[0]
            ranked_stop_value = stop_fields[1]


            # depending on the mode chosen by the user, select the formulas to be used by looking to the averages or the totals
            if mode == "averages":
                temp_spec_shared = line_fields[4].split("=")[1]
                temp_sens_shared = line_fields[5].split("=")[1]
                temp_spec_int = line_fields[6].split("=")[1]
                temp_sens_int = line_fields[7].split("=")[1]
            elif mode == "totals":
                temp_spec_shared = line_fields[12].split("=")[1]
                temp_sens_shared = line_fields[13].split("=")[1]
                temp_spec_int = line_fields[14].split("=")[1]
                temp_sens_int = line_fields[15].split("=")[1]
            else:
                raise ValueError("mode has an invalid value (%s)" %(mode))

            try:
                ranked_spec_shared = float(temp_spec_shared)
            except:
                ranked_spec_shared = "None"

            try:
                ranked_sens_shared = float(temp_sens_shared)
            except:
                ranked_sens_shared = "None"

            try:
                ranked_spec_int = float(temp_spec_int)
            except:
                ranked_spec_int = "None"

            try:
                ranked_sens_int = float(temp_sens_int)
            except:
                ranked_sens_int = "None"
                
            ranked_coverage_shared = float( line_fields[8].split("=")[1] )
            ranked_coverage_int = float( line_fields[9].split("=")[1] )

            # use_this_formula will be 1 if the formula remains within the restrictions. Will be 0 otherwise
            use_this_formula = ScoringFunctionBenchmark.use_formula( spec_shared_threshold= spec_shared_threshold, ranked_spec_shared= ranked_spec_shared,
                                                                     sens_shared_threshold= sens_shared_threshold, ranked_sens_shared= ranked_sens_shared,
                                                                     spec_int_threshold= spec_int_threshold, ranked_spec_int= ranked_spec_int,
                                                                     sens_int_threshold= sens_int_threshold, ranked_sens_int= ranked_sens_int,
                                                               coverage_shared_threshold= coverage_shared_threshold, ranked_coverage_shared= ranked_coverage_shared,
                                                               coverage_int_threshold= coverage_int_threshold, ranked_coverage_int= ranked_coverage_int)


            if use_this_formula:
                # the formula will only be applied if its specificty is higher than the threshold set by the user

                category = ScoringFunctionBenchmark.Category(w_patches=ranked_w_patches, w_prots=ranked_w_prots, w_belong=ranked_w_belong,
                                                             stop_to_use=ranked_stop_to_use, stop_value=ranked_stop_value)
                formulas_to_apply.append( category )
            else:
                if verbose_detailed:
                    sys.stderr.write("skipping formula\n" )
        # END OF for line in ranking_file_fd:
            

        # ------------------------------
        # Calculate the performance of the method using the formulas previosly loaded according the the user criteria
        # ------------------------------
        # for each protein, find the performance stats using the ranked weights and stop conditions (ie categories)
        # 
        for protein in self.all_proteins:

            if verbose_detailed:
                sys.stderr.write("testing protein %s using %s formulas" %(protein, len(formulas_to_apply)))
                
            stop_condition_met = 0
            
            for formula_to_apply_category in formulas_to_apply:
                # Each of the formulas that were loaded (ie. those "good" formulas) is applied until finding a stop condition for the protein
                formula_to_apply_category_string = formula_to_apply_category.return_category_string()
                
                if self.results_by_protein[protein].has_key(formula_to_apply_category_string):

                    
                    stop_condition_met = 1 # if there is a comparison line for this parameters, it means that when applying the method
                                            # a decomposition will be found: this is the line that says how good the method performed
                                            #   (then, we have to check if the line has usable info about how good it was... spec and sens could be None)

                    # this is the comparison line that would be produced by the method in "exec" or "eval" modes: use it to calculate the performance of
                    # the method
                    # TO DO!!! Previous line is not exactly correct: in piana modes "exec" and "eval" I am not applying thresholds to use the formulas in the
                    #          ranked parameters file... I am using all of them in order, regardless of their values: this means that in modes "exec" and
                    #          "eval" I don't let the user maximize using thresholds. THIS MUST BE CHANGED
                    
                    comparison_line_object_to_use = self.results_by_protein[protein][formula_to_apply_category_string]


                    ignore_result = 0
                    # if the number of positives is too high, or the number of false negatives is to high, we consider that the method didn't find an
                    # answer for the protein: see Note in output_training_results for justification
                    total_test_positives = comparison_line_object_to_use.tps_shared + comparison_line_object_to_use.fps_shared
                    if total_test_positives > NUMBER_OF_POSITIVES_ALLOWED:
                        ignore_result = 1

                    if comparison_line_object_to_use.fns_shared > NUMBER_OF_FALSE_NEGATIVES_ALLOWED:
                        ignore_result = 1
                    

                    # Although the number of proteins that had a result is calculated by looking to the spec and sens calculated in the
                    # comparison line (instead of looking if there were tps, fps and fns), we can use this for both modes, since
                    # it is the same to check if spec of sens were None or if there were positives or fns
                    if not ignore_result and (comparison_line_object_to_use.spec_shared != "None" or comparison_line_object_to_use.sens_shared != "None"):
                        number_of_proteins_with_global_shared_result += 1
                    else:
                        number_of_proteins_None_shared_result += 1

                    if not ignore_result and (comparison_line_object_to_use.spec_int != "None" or comparison_line_object_to_use.sens_int != "None"):
                        number_of_proteins_with_global_int_result += 1
                    else:
                        number_of_proteins_None_int_result += 1

                    if mode == "averages":
                        # In mode averages, calculate the averages of the specs and sens
                        
                        if not ignore_result and comparison_line_object_to_use.spec_shared != "None":

                            total_spec_shared += comparison_line_object_to_use.spec_shared
                            number_of_proteins_with_spec_shared_result += 1

                        if not ignore_result and comparison_line_object_to_use.sens_shared != "None":
                            total_sens_shared += comparison_line_object_to_use.sens_shared
                            number_of_proteins_with_sens_shared_result += 1


                        if not ignore_result and comparison_line_object_to_use.spec_int != "None":
                            total_spec_int += comparison_line_object_to_use.spec_int
                            number_of_proteins_with_spec_int_result += 1

                        if not ignore_result and comparison_line_object_to_use.sens_int != "None":
                            total_sens_int += comparison_line_object_to_use.sens_int
                            number_of_proteins_with_sens_int_result += 1

                    # END OF if mode == "averages":
                    
                    elif mode == "totals":
                        if not ignore_result:
                            total_tps_shared += comparison_line_object_to_use.tps_shared
                            total_fps_shared += comparison_line_object_to_use.fps_shared
                            total_fns_shared += comparison_line_object_to_use.fns_shared
                            
                            total_tps_int += comparison_line_object_to_use.tps_int
                            total_fps_int += comparison_line_object_to_use.fps_int
                            total_fns_int += comparison_line_object_to_use.fns_int
                        # END OF if not ignore_result:

                    # END OF elif mode == "totals":

                # END OF if self.results_by_protein[protein].has_key(formula_to_apply_category_key):
                
                if stop_condition_met:
                    # once a result has been found for a comparison line of a category, it means the result for this protein has been found: go to next protein
                    # this doesn't mean the accuracy was different from None, it just means there was a comparison line for those parameters
                    break
                
            # END OF  for formula_to_apply_category in formulas_to_apply:
            
            if not stop_condition_met:
                # if after trying all formulas, no comparison line is found, this a protein for which the method can't return a result
                number_of_proteins_stop_condition_not_met += 1
            else:
                number_of_proteins_stop_condition_met += 1
                
        # END OF for protein in self.proteins_with_results_shared:

        if mode == "averages":

            if number_of_proteins_with_spec_shared_result > 0:
                spec_shared= total_spec_shared/float(number_of_proteins_with_spec_shared_result)
            else:
                spec_shared= "None"

            if number_of_proteins_with_sens_shared_result:
                sens_shared= total_sens_shared/float(number_of_proteins_with_sens_shared_result)
            else:
                sens_shared= "None"

            if number_of_proteins_with_spec_int_result > 0:
                spec_int= total_spec_int/float(number_of_proteins_with_spec_int_result)
            else:
                spec_int= "None"

            if number_of_proteins_with_sens_int_result:
                sens_int= total_sens_int/float(number_of_proteins_with_sens_int_result)
            else:
                sens_int= "None"
        # END OF if mode == "averages":
        elif mode == "totals":
            try:
                spec_shared = total_tps_shared / float(total_tps_shared + total_fps_shared)
            except:
                spec_shared = "None"

            try:
                sens_shared = total_tps_shared / float(total_tps_shared + total_fns_shared)
            except:
                sens_shared = "None"

            try:
                spec_int = total_tps_int / float(total_tps_int + total_fps_int)
            except:
                spec_int = "None"

            try:
                sens_int = total_tps_int / float(total_tps_int + total_fns_int)
            except:
                sens_int = "None"
        # END OF elif mode == "totals":
                
        ratio_proteins_stop_condition_met = number_of_proteins_stop_condition_met/ float(num_all_proteins)
        
        if number_of_proteins_stop_condition_met:
            ratio_proteins_result_shared_found = number_of_proteins_with_global_shared_result/ float(number_of_proteins_stop_condition_met)
            ratio_proteins_result_int_found = number_of_proteins_with_global_int_result/ float(number_of_proteins_stop_condition_met)
        else:
            ratio_proteins_result_shared_found = 0
            ratio_proteins_result_int_found = 0
       
        return (spec_shared, sens_shared,  ratio_proteins_result_shared_found, 
                spec_int, sens_int, ratio_proteins_result_int_found,
                num_all_proteins, number_of_proteins_stop_condition_met, ratio_proteins_stop_condition_met,
                total_tps_shared, total_fps_shared, total_fns_shared,
                total_tps_int, total_fps_int, total_fns_int )

    class Parameters(object):
        """
        groups in a class all the parameters that can be passed to ScoringFunctionBenchmark
        """

        def __init__(self, coverage_shared_threshold= None ,
                     coverage_int_threshold= None ,
                     spec_shared_threshold= None ,
                     sens_shared_threshold= None ,
                     spec_int_threshold= None ,
                     sens_int_threshold= None,
                     ranking_mode = None       ):
            
            self.coverage_shared_threshold= coverage_shared_threshold
            self.coverage_int_threshold= coverage_int_threshold
            self.spec_shared_threshold= spec_shared_threshold
            self.sens_shared_threshold= sens_shared_threshold
            self.spec_int_threshold= spec_int_threshold
            self.sens_int_threshold= sens_int_threshold
            
            self.ranking_mode= ranking_mode

        def get_coverage_shared_threshold(self):
            return self.coverage_shared_threshold
        
        def get_coverage_int_threshold(self):
            return self.coverage_int_threshold

        def get_spec_shared_threshold(self):
            return self.spec_shared_threshold
        
        def get_sens_shared_threshold(self):
            return self.sens_shared_threshold

        def get_spec_int_threshold(self):
            return self.spec_int_threshold
        
        def get_sens_int_threshold(self):
            return self.sens_int_threshold
        
        def get_ranking_mode(self):
            return self.ranking_mode

    class Results(object):
        """
        Used to keep the average results (and standard deviation) from a n_fold cross-validation
        
        It also keeps which were the parameters (thresholds and ranking mode) used to obtain those results
        """
        def __init__(self, spec_shared, spec_shared_std_deviation, sens_shared, sens_shared_std_deviation,
                     spec_int, spec_int_std_deviation, sens_int, sens_int_std_deviation,
                     num_proteins_tested, ratio_proteins_stop_met,
                     ratio_proteins_with_shared_result,
                     ratio_proteins_with_int_result,parameters_object):
            """

            ratio_proteins_stop_met is the ratio between how many proteins met the stop condition over the total number of proteins being tested

            ratio_proteins_with_result is the ratio between how many proteins had a result != None over the total number of proteins being tested
               -> obviously, a protein can only have a result != None if its condition stop was met
            """
            self.spec_shared= spec_shared
            self.spec_shared_std_deviation = spec_shared_std_deviation
            self.sens_shared= sens_shared
            self.sens_shared_std_deviation = sens_shared_std_deviation
            self.spec_int= spec_int
            self.spec_int_std_deviation = spec_int_std_deviation
            self.sens_int= sens_int
            self.sens_int_std_deviation = sens_int_std_deviation
            self.num_proteins_tested= num_proteins_tested
            self.ratio_proteins_stop_met= ratio_proteins_stop_met
            self.ratio_proteins_with_shared_result= ratio_proteins_with_shared_result
            self.ratio_proteins_with_int_result= ratio_proteins_with_int_result
            self.parameters_object= parameters_object
            
            
        
# ----------------------
# Function usage()
# ----------------------
def usage():
    print "--------------------------------------------------------------------------------------------------------------"
    print "This program processes comparison files and finds statistics from them. Can be called from command line or as an object. \n"
    print "Usage: python ScoringFunctionBenchmark.py  --input-dir=input_dir --output-prefix=output_prefix --ranking-mode=ranking_mode"
    print "                                           --coverage-shared-threshold=coverage_shared_threshold --coverage-int-threshold=coverage_int_threshold"
    print "                                           --spec-shared-threshold=spec_shared_threshold --sens-shared-threshold=spec_shared_threshold"
    print "                                           --spec-int-threshold=spec_int_threshold --sens-int-threshold=spec_int_threshold"
    print "                                           --n-fold=n_fold --test-many-thresholds --just-rank-params [--help] [--verbose]"
    print "\nwhere:"
    print "     output_prefix  : prefix that will be added to result files"
    print "     --test-many-thresholds  : if flag set, many built-in parameters will be tested to find the best ones"
    print "     --just-rank-params  : if flag set, once ranked parameters are generated for the first run, exit (used for getting ranked params)"
    print "     input_dir      : directory with comparison files (cannot be used at the same time than input-file) (must only have comparison files...)"
    print "     n_fold         : number of subsets used for N fold cross validation (n_fold subsets will be created)"
    print "                       -> if n_fold = 0, then do a leave-one-out validation"
    print "     if testing-many-parameters, the following are not required:"
    print "     ranking_mode   : use to train the system for maximizing a certain score"
    print "                      valid values are: "
    print "                       - av_spec_shared: to maximize specificity for shared binding sites looking at protein spec average "
    print "                       - av_sens_shared: to maximize sensitivity for shared binding sites "
    print "                       - av_spec_int: to maximize specificity for int binding sites "
    print "                       - av_sens_int: to maximize sensitivity for int binding sites "
    print "                       - tot_spec_shared: to maximize specificity for shared binding sites looking at global TP,FP, FNs average"
    print "                       - tot_sens_shared: to maximize sensitivity for shared binding sites "
    print "                       - tot_spec_int: to maximize specificity for int binding sites "
    print "                       - tot_sens_int: to maximize sensitivity for int binding sites "
    print "     coverage_shared_threshold : lowest shared coverage allowed  for producing the trained ranking "
    print "     coverage_int_threshold : lowest int coverage allowed  for producing the trained ranking "
    print "     spec_shared_threshold : lowest specificity allowed for producing the trained ranking (formulas with lower spec shared will not be applied)"
    print "     sens_shared_threshold : lowest sensitivity allowed for producing the trained ranking (formulas with lower sens shared will not be applied)"
    print "     spec_int_threshold : lowest specificity allowed for producing the trained ranking (formulas with lower spec int will not be applied)"
    print "     sens_int_threshold : lowest sensitivity allowed for producing the trained ranking (formulas with lower sens int will not be applied)"
    print "     --help         : prints this message and exits"
    print "     --verbose      : prints process info to stdout"
    print "--------------------------------------------------------------------------------------------------------------"

def parse_arguments():
    
    global input_dir
    global output_prefix
    global n_fold

    global coverage_shared_threshold
    global coverage_int_threshold
    
    global spec_shared_threshold
    global sens_shared_threshold
    
    global spec_int_threshold
    global sens_int_threshold
    
    global ranking_mode

    global test_many_thresholds
    global just_rank_params
    
    try:
        opts, args = getopt.getopt(sys.argv[1:], "", ["verbose","help","input-dir=","output-prefix=", "ranking-mode=",
                                                      "coverage-shared-threshold=", "coverage-int-threshold=",
                                                      "spec-shared-threshold=", "sens-shared-threshold=","spec-int-threshold=", "sens-int-threshold=",
                                                      "n-fold=", "test-many-thresholds", "just-rank-params"])
    except getopt.GetoptError, error:
        # print help information and exit:
        sys.stderr.write("Command line incorrect: %s\n" %error)
        usage()
        sys.exit(2)
     
    for option,value in opts:
        
         if option == "--input-dir":
             input_dir = value
             
         elif option == "--input-file":
             input_file = value
             
         elif option == "--output-prefix":
             output_prefix = value
             
         elif option == "--n-fold":
             n_fold = int(value)
             
         elif option == "--ranking-mode":
             ranking_mode = value
             
         elif option == "--test-many-thresholds":
             test_many_thresholds = 1
             
         elif option == "--just-rank-params":
             just_rank_params = 1
             
         elif option == "--coverage-shared-threshold":
             coverage_shared_threshold = float(value)
             
         elif option == "--coverage-int-threshold":
             coverage_int_threshold = float(value)
             
         elif option == "--spec-shared-threshold":
             spec_shared_threshold = float(value)
             
         elif option == "--sens-shared-threshold":
             sens_shared_threshold = float(value)
             
         elif option == "--spec-int-threshold":
             spec_int_threshold = float(value)
             
         elif option == "--sens-int-threshold":
             sens_int_threshold = float(value)

         elif option == "--verbose":
             verbose = 1

         elif option == "--help":
             # print help information and exit
             usage()
             sys.exit(2)   

    # END OF for option,value in opts:

    if output_prefix is None:
        sys.stderr.write("output-prefix cannot be None\n")
        usage()
        sys.exit(2)  

    if n_fold is None:
        sys.stderr.write("n-fold cannot be None\n")
        usage()
        sys.exit(2)

    if input_dir is None:
        sys.stderr.write("input-dir cannot be None\n")
        usage()
        sys.exit(2) 

    if test_many_thresholds == 0:
        # if not testing many parameters, these have to be set

        if ranking_mode is None:
            sys.stderr.write("ranking-mode cannot be None\n")
            usage()
            sys.exit(2)

        if coverage_shared_threshold is None or coverage_int_threshold is None:
            sys.stderr.write("coverage_shared_threshold and coverage_int_threshold cannot be None\n")
            usage()
            sys.exit(2)

        if spec_shared_threshold is None or  sens_shared_threshold is None or spec_int_threshold is None or sens_int_threshold is None:
            sys.stderr.write("spec_shared_threshold and sens_shared_threshold and spec_int_threshold and sens_int_threshold cannot be None\n")
            usage()
            sys.exit(2)



# -------------------------------------------
# MAIN: execute  when called from command line
# -------------------------------------------
if __name__ == "__main__":
    
    input_dir= None
    output_prefix = None

    test_many_thresholds = 0
    just_rank_params = 0

    coverage_shared_threshold = None
    coverage_int_threshold = None
    
    spec_shared_threshold= None
    sens_shared_threshold= None
    spec_int_threshold= None
    sens_int_threshold= None
    
    n_fold= None

    ranking_mode = None

    all_n_fold_results = []    # A list of Results object (one result for each test with different parameters)

    parse_arguments()



    # ----------------------------------------------------------------------------
    # create the parameters (ie. thresholds and ranking modes) that will be tested 
    # ----------------------------------------------------------------------------

    # --
    # this is the list that establishes which are the stats to maximize (i.e give order to ranked parameters)
    # --

    # use only tot_* modes, because the other way of calculating the spec and sens is not correct. Well, it is correct, but if I use
    # averages instead of totals, then I would need to give the standard deviation... Therefore, it is better to use total TPs, FPs and FNs
    # to calculate spec and sens
    
    #list_ranking_modes = ["av_spec_shared", "av_sens_shared", "av_spec_int", "av_sens_int"]

    list_ranking_modes = ["tot_spec_shared", "tot_sens_shared", "tot_spec_int", "tot_sens_int"]           #1728
    #list_ranking_modes = ["tot_spec_shared", "tot_sens_shared", "tot_spec_int", "tot_sens_int"]            #128
    #list_ranking_modes = ["tot_spec_shared"]           #3
    #list_ranking_modes = ["tot_sens_int"]           #1

    # --
    # this is the list of coverages that will be used for testing. A coverage represents the ratio of proteins
    # that must have an answer for a given formula ir order for it to appear in the ranked parameters file
    # --
    list_coverages_shared_to_test = [0.01, 0.1]            #1728
    list_coverages_int_to_test = [0.01, 0.1]               #1728
    
    #list_coverages_shared_to_test = [0.05]             #128
    #list_coverages_int_to_test = [0.05]                #128
    
    #list_coverages_shared_to_test = [0.01]            #3
    #list_coverages_int_to_test = [0.01]               #3
    
    #list_coverages_shared_to_test = [0.01]            #1
    #list_coverages_int_to_test = [0.01]               #1

    # --
    # this is the list of thresholds that will be used for minimum spec and sens allowed for a formula to appear
    # in the ranked parameters file
    # --    
    list_stats_spec_shared_to_test =  [0, 0.3, 0.5, 0.7]     #1728 
    list_stats_sens_shared_to_test =  [0, 0.3, 0.5]         #1728 
    
    list_stats_spec_int_to_test =  [0, 0.3, 0.5]            #1728 
    list_stats_sens_int_to_test =  [0, 0.3, 0.5]            #1728
    
    #list_stats_spec_shared_to_test =  [0, 0.3, 0.5, 0.7]     #128 
    #list_stats_sens_shared_to_test =  [0, 0.3]               #128 
    
    #list_stats_spec_int_to_test =  [0, 0.3]                  #128
    #list_stats_sens_int_to_test =  [0, 0.3]                  #128
    
    #list_stats_spec_shared_to_test =  [0.3, 0.5, 0.7]   #3
    #list_stats_sens_shared_to_test =  [0]               #3
    #list_stats_spec_int_to_test =  [0]                  #3
    #list_stats_sens_int_to_test =  [0]                  #3
    
    #list_stats_spec_shared_to_test =  [0.3]   #1
    #list_stats_sens_shared_to_test =  [0]     #1
    #list_stats_spec_int_to_test =  [0.3]      #1
    #list_stats_sens_int_to_test =  [0]        #1



    list_of_parameters_to_use = []
    
    if test_many_thresholds:
        # create list of parameters that will be used
        if verbose:
            sys.stderr.write("Creating list parameters to use\n")
            number_of_iterations_to_do = len(list_ranking_modes) * len(list_coverages_shared_to_test) *  len(list_coverages_int_to_test) * len(list_stats_spec_shared_to_test) * len(list_stats_sens_shared_to_test) * len(list_stats_spec_int_to_test) * len(list_stats_sens_int_to_test) 
            j = 0

        # create the parameters to be used when the user choses "--test-many-thresholds"
        for one_coverage_shared_threshold in list_coverages_shared_to_test:
            for one_coverage_int_threshold in list_coverages_int_to_test:
                for one_spec_shared_threshold in list_stats_spec_shared_to_test:
                    for one_sens_shared_threshold in list_stats_sens_shared_to_test:
                        for one_spec_int_threshold in list_stats_spec_int_to_test:
                            for one_sens_int_threshold in list_stats_sens_int_to_test:
                                for one_ranking_mode in list_ranking_modes:

                                    if verbose:
                                        sys.stderr.write("%sof%s -- " %(j,number_of_iterations_to_do ))
                                        j += 1

                                    parameters_object = ScoringFunctionBenchmark.Parameters(coverage_shared_threshold= one_coverage_shared_threshold,
                                                                                            coverage_int_threshold= one_coverage_int_threshold,
                                                                                            spec_shared_threshold= one_spec_shared_threshold,
                                                                                            sens_shared_threshold= one_sens_shared_threshold,
                                                                                            spec_int_threshold= one_spec_int_threshold,
                                                                                            sens_int_threshold= one_sens_int_threshold,
                                                                                            ranking_mode = one_ranking_mode)
                                    
                                    list_of_parameters_to_use.append(parameters_object)
        # END OF (all the fors)
    # END OF if test_many_thresholds:
        
    else:
        # use command line parameters (only one run with a fixed set of thresholds and one ranking mode)
        if verbose:
            sys.stderr.write("Using parameters from command line\n")
        parameters_object = ScoringFunctionBenchmark.Parameters(coverage_shared_threshold= coverage_shared_threshold,
                                                                coverage_int_threshold= coverage_int_threshold,
                                                                spec_shared_threshold= spec_shared_threshold,
                                                                sens_shared_threshold= sens_shared_threshold,
                                                                spec_int_threshold= spec_int_threshold,
                                                                sens_int_threshold= sens_int_threshold,
                                                                ranking_mode = ranking_mode)
        list_of_parameters_to_use.append(parameters_object)
    # END OF else:(if test_many_thresholds:)


    # ---------------------------------------------------------------------------------------------------------
    # Do the train/test for each of the parameters created (only one if they were set through the command line)
    # ---------------------------------------------------------------------------------------------------------

    iter_counter = 0
    num_different_thresholds = len(list_of_parameters_to_use)
    for parameters_to_use in list_of_parameters_to_use:
        # for each parameters created previously, do the train/test n-fold
        iter_counter += 1
        coverage_shared_threshold = parameters_to_use.get_coverage_shared_threshold()
        coverage_int_threshold = parameters_to_use.get_coverage_int_threshold()
        spec_shared_threshold = parameters_to_use.get_spec_shared_threshold()
        sens_shared_threshold = parameters_to_use.get_sens_shared_threshold()
        spec_int_threshold = parameters_to_use.get_spec_int_threshold()
        sens_int_threshold = parameters_to_use.get_sens_int_threshold()
        ranking_mode = parameters_to_use.get_ranking_mode()

        thresholds_description = "\n==========================================================================\n"
        thresholds_description += "                    NEW ITERATION %s of %s WITH thresholds                \n" %(iter_counter, num_different_thresholds)
        thresholds_description += "coverage_shared_threshold= %s\n" %coverage_shared_threshold
        thresholds_description += "coverage_int_threshold= %s\n" %coverage_int_threshold
        thresholds_description += "spec_shared_threshold= %s\n"  %spec_shared_threshold
        thresholds_description += "sens_shared_threshold= %s\n"  %sens_shared_threshold
        thresholds_description += "spec_int_threshold= %s\n"  %spec_int_threshold
        thresholds_description += "sens_int_threshold= %s\n"  %sens_int_threshold
        thresholds_description += "ranking_mode = %s\n"  %ranking_mode
        thresholds_description += "==========================================================================\n"

        if verbose:
            sys.stderr.write(thresholds_description)
        # END OF if verbose:


        # file that will hold n-fold results
        if test_many_thresholds:
            all_tests_file_name =  "/home/raragues/phd/piana/code/evaluation/Benchmark/results_many_parameters/all_tests.%s.ranking_%s.%s_fold.%s" %(output_prefix,
                                                                                                                                                    ranking_mode,
                                                                                                                                                    n_fold,
                                                                                                                                                    iter_counter)
            all_tests_file_fd = file(all_tests_file_name, "w")
        else:
            all_tests_file_name =  "%s.ranking_%s.%s_fold_results.txt" %(output_prefix, ranking_mode, n_fold)
            all_tests_file_fd = file(all_tests_file_name, "w")

        # describe the thresholds that are being used in these pool of n-fold tests
        
        all_tests_file_fd.write(thresholds_description)
        
        # ---------------
        # 1. Do N runs of train/test
        #    in each run, a different comparison file is used as test set, and the remaining files as training set
        # ---------------
        total_test_runs = 0   # keeps ids for the test being performed

        # These variables are:
        #
        #   all_* are lists that keep the results from the tests (one element per test)
        #   total_* are the addition of all the results from the tests
        #   total_test_with_* keep the number of tests where  that stat was found

        all_test_runs = []

        all_test_tps_shared = []
        all_test_fps_shared = []
        all_test_fns_shared = []

        total_test_tps_shared = 0
        total_test_fps_shared = 0
        total_test_fns_shared = 0
        
        all_test_spec_shared = []
        all_test_sens_shared = []
        all_ratio_proteins_result_shared_found = []


        total_test_spec_shared = 0
        total_test_sens_shared = 0
        total_ratio_proteins_result_shared_found = 0

        total_test_with_spec_shared_result = 0
        total_test_with_sens_shared_result = 0
        total_test_with_ratio_proteins_result_shared_found = 0


        all_test_tps_int = []
        all_test_fps_int = []
        all_test_fns_int = []
        
        total_test_tps_int = 0
        total_test_fps_int = 0
        total_test_fns_int = 0
        
        all_test_spec_int = []
        all_test_sens_int = []
        all_ratio_proteins_result_int_found = []

        total_test_spec_int = 0
        total_test_sens_int = 0
        total_ratio_proteins_result_int_found = 0

        total_test_with_spec_int_result = 0
        total_test_with_sens_int_result = 0
        total_test_with_ratio_proteins_result_int_found = 0

        all_num_all_proteins = []
        all_number_of_proteins_stop_condition_met = []
        all_ratio_proteins_stop_condition_met = []

        total_test_num_all_proteins = 0
        total_number_of_proteins_stop_condition_met = 0
        total_ratio_proteins_stop_condition_met = 0

        total_test_with_num_all_proteins = 0
        total_test_with_number_of_proteins_stop_condition_met = 0
        total_test_with_ratio_proteins_stop_condition_met = 0

        global_total_training_information_found_shared = 0
        global_size_training_set = 0


        # N Fold cross validation process:
        # divide the input comparison files in n_fold subsets
        # for each subset, do the train(other subsets)/test(this_subset)

        
        all_files_in_directory = {}
        for file_in_directory in utilities.GlobDirectoryWalker(input_dir , "*.*"):
            all_files_in_directory[file_in_directory] = None
        # END OF for file_in_directory in utilities.GlobDirectoryWalker(input_dir , "*.*"):

        number_of_proteins_with_comparison_file = len(all_files_in_directory)

        files_to_sample = copy.deepcopy(all_files_in_directory)
        subsets_dic = {} # subsets_dic is a dictionary that contains a list of files for each subset
                         # follows structure:
                         #                     { 0: [file1, file2, ...],
                         #                       1: [file3, file4, ...],
                         #                       .....................,
                         #                       n_fold-1:[file5, file6, ...]
                         #                     }


        if n_fold == 0:
            # if user chose 0 as n_fold, he meant to do a leave-one-out (ie. each set is just one file)
            n_fold = len(files_to_sample)

        # Initialize subsets to empty lists
        for i in range(n_fold):
            subsets_dic[i] = {}

        # Create random groups of equal size
        end = 0
        random.seed()
        while not end:
            for i in range(n_fold):
                try:
                    file_names = files_to_sample.keys()
                    file_to_insert = random.choice(file_names)
                    subsets_dic[i][file_to_insert] = None
                    file_names.remove(file_to_insert)
                    del files_to_sample[file_to_insert]
                except:
                    # when there are no more items an exception will be raised and we will exit the loop: all items have been assigned to a group
                    end = 1
                    break
            # END OF for i in range(n_fold):
        # while not end:


        if verbose:
            sys.stderr.write("\nSizes of subsets are: ")
            for i in range(n_fold):
                sys.stderr.write("subset %s of lenght %s-" %(i, len(subsets_dic[i])))
            sys.stderr.write("\n")

        for subset_number in subsets_dic:
            # this is the loop in charge of running a test for each subset (using the other subsets for training)

            test_scoring_function_stats = ScoringFunctionBenchmark(output_target_prefix= output_prefix)
            total_test_information_found_shared = 0
            total_test_information_found_int = 0
            test_information_found = 0

            training_scoring_function_stats = ScoringFunctionBenchmark(output_target_prefix= output_prefix)
            total_training_information_found_shared = 0
            total_training_information_found_int = 0
            training_information_found = 0

            # ---------------
            # 1.1. Fill the training and test objects with information from their comparison files
            # ---------------
            number_of_prots_used_for_training = 0
            for file_name in all_files_in_directory:

                protein_name = file_name.split("/")[-1].split(".")[0]

                if subsets_dic[subset_number].has_key(file_name):
                    # if it is a file from the subset being tested, fill the test_scoring_function_stats object
                    test_information_found = test_scoring_function_stats.get_information(comparison_file_name= file_name, protein_name = protein_name )

                    total_test_information_found_shared += test_information_found[0]
                    total_test_information_found_int += test_information_found[1]

                else:
                    number_of_prots_used_for_training += 1
                    # if current subset is not the one to test, reads comparison file and populates training_scoring_function_stats object
                    training_information_found = training_scoring_function_stats.get_information(comparison_file_name= file_name, protein_name = protein_name )

                    total_training_information_found_shared += training_information_found[0]
                    total_training_information_found_int += training_information_found[1]
                # END OF else: (if file_name == comparison_file_to_test:)

            # END OF for file_name in files_to_parse:
            global_size_training_set += number_of_prots_used_for_training
            global_total_training_information_found_shared += total_training_information_found_shared

            # ---------------
            # 1.2. Do the train/test for this n-fold run
            # ---------------

            if total_training_information_found_shared:

                # 2.1 Train
                ranking_file_name = training_scoring_function_stats.output_training_results(ranking_mode=ranking_mode)


                if just_rank_params:
                    sys.stdout.write("ranked parameters were written to file %s\n" %(ranking_file_name) )
                    sys.exit()

                if ranking_mode == "av_spec_shared" or ranking_mode == "av_sens_shared" or ranking_mode == "av_spec_int" or ranking_mode == "av_sens_int":
                    test_mode = "averages"
                elif ranking_mode == "tot_spec_shared" or ranking_mode == "tot_sens_shared" or ranking_mode == "tot_spec_int" or ranking_mode == "tot_sens_int":
                    test_mode = "totals"
                
                # 2.2 Test    
                results = test_scoring_function_stats.output_test_results(ranking_file_name= ranking_file_name,
                                                                          coverage_shared_threshold = coverage_shared_threshold,
                                                                          coverage_int_threshold = coverage_int_threshold,
                                                                          spec_shared_threshold= spec_shared_threshold,
                                                                          sens_shared_threshold= sens_shared_threshold,
                                                                          spec_int_threshold= spec_int_threshold,
                                                                          sens_int_threshold= sens_int_threshold,
                                                                          mode = test_mode)


                # 2.3 Store results: this are the results obtained by training on all subsets except for one
                #    --> add the stats for this test to the totals for the n_fold

                # results is a list where:
                #                         [0] is spec_shared
                #                         [1] is sens_shared
                #                         [2] is ratio_proteins_result_shared_found   (over those where stop condition was met, how many had !=None spec and sens)
                #                         [3] is spec_int
                #                         [4] is sens_int
                #                         [5] is ratio_proteins_result_int_found (over those where stop condition was met, how many had !=None spec and sens)
                #                         [6] is num_all_proteins
                #                         [7] is number_of_proteins_stop_condition_met 
                #                         [8] is ratio_proteins_stop_condition_met (over total num of prots with comp files, how many stop condition was met)
                #                         [9] total tps shared
                #                         [10] total fps shared
                #                         [11] total fns shared 
                #                         [12] total tps int
                #                         [13] total fps int
                #                         [14] total fns int 

                this_test_spec_shared = results[0]
                this_test_sens_shared = results[1]
                this_test_ratio_proteins_result_shared_found = results[2]

                this_test_spec_int = results[3]
                this_test_sens_int = results[4]
                this_test_ratio_proteins_result_int_found = results[5]

                this_test_num_all_proteins = results[6]
                this_test_number_of_proteins_stop_condition_met = results[7]
                this_test_ratio_proteins_stop_condition_met = results[8]

                this_test_total_tps_shared =  results[9]
                this_test_total_fps_shared =  results[10]
                this_test_total_fns_shared =  results[11]
                
                this_test_total_tps_int =  results[12]
                this_test_total_fps_int =  results[13]
                this_test_total_fns_int =  results[14]
                
            # END OF if total_training_information_found_shared:
            
            else:
                # Training files did not produce a list of ranked parameters: this test is all set to None
                this_test_spec_shared = "None"
                this_test_sens_shared = "None"
                this_test_ratio_proteins_result_shared_found = 0
                this_test_spec_int = "None"
                this_test_sens_int = "None"
                this_test_ratio_proteins_result_int_found = 0
                this_test_num_all_proteins = 0
                this_test_number_of_proteins_stop_condition_met = 0
                this_test_ratio_proteins_stop_condition_met = 0
                this_test_total_tps_shared = 0
                this_test_total_fps_shared = 0
                this_test_total_fns_shared = 0
                this_test_total_tps_int = 0 
                this_test_total_fps_int = 0
                this_test_total_fns_int = 0
            # END OF else: (if total_training_information_found_shared:)

            if verbose:
                sys.stderr.write("------------------------TEST %s-------------------------------------------\n" %(total_test_runs))
                sys.stderr.write("Total number of proteins in training set with info: %s (out of %s)\n" %(total_training_information_found_shared,
                                                                                                          number_of_prots_used_for_training))
                sys.stderr.write("shared: tps=%s\tfps=%s\tfns=%s\n" %(this_test_total_tps_shared,this_test_total_fps_shared,this_test_total_fns_shared  ))
                sys.stderr.write("this_test_spec_shared= %s\n" %(this_test_spec_shared))
                sys.stderr.write("this_test_sens_shared=  %s\n" %(this_test_sens_shared))
                sys.stderr.write("this_test_ratio_proteins_result_shared_found=  %s\n" %(this_test_ratio_proteins_result_shared_found))
                sys.stderr.write("int: tps=%s\tfps=%s\tfns=%s\n" %(this_test_total_tps_int,this_test_total_fps_int,this_test_total_fns_int  ))
                sys.stderr.write("this_test_spec_int=  %s\n" %(this_test_spec_int))
                sys.stderr.write("this_test_sens_int=  %s\n" %(this_test_sens_int))
                sys.stderr.write("this_test_ratio_proteins_result_int_found=  %s\n" %(this_test_ratio_proteins_result_int_found))
                sys.stderr.write("this_test_num_all_proteins=  %s\n" %(this_test_num_all_proteins))
                sys.stderr.write("this_test_number_of_proteins_stop_condition_met=  %s\n" %(this_test_number_of_proteins_stop_condition_met))
                sys.stderr.write("this_test_ratio_proteins_stop_condition_met=  %s\n" %(this_test_ratio_proteins_stop_condition_met))
                sys.stderr.write("---------------------------------------------------------------------------\n")

            all_test_runs.append(total_test_runs) # keeps an id for each run performed during testing
                                                  # We performe n_fold test runs, each obtains an average result for that test. Then, we do the average of the
                                                  # averages.
                                                  # This id is used to access the averages of each test

            all_test_tps_shared.append(this_test_total_tps_shared)
            all_test_fps_shared.append(this_test_total_fps_shared)
            all_test_fns_shared.append(this_test_total_fns_shared)

            all_test_spec_shared.append(this_test_spec_shared)
            all_test_sens_shared.append(this_test_sens_shared)
            all_ratio_proteins_result_shared_found.append(this_test_ratio_proteins_result_shared_found)

            all_test_tps_int.append(this_test_total_tps_int)
            all_test_fps_int.append(this_test_total_fps_int)
            all_test_fns_int.append(this_test_total_fns_int)

            all_test_spec_int.append(this_test_spec_int)
            all_test_sens_int.append(this_test_sens_int)
            all_ratio_proteins_result_int_found.append(this_test_ratio_proteins_result_int_found)

            all_num_all_proteins.append(this_test_num_all_proteins)
            all_number_of_proteins_stop_condition_met.append(this_test_number_of_proteins_stop_condition_met)
            all_ratio_proteins_stop_condition_met.append(this_test_ratio_proteins_stop_condition_met)

            total_test_runs += 1

            total_test_tps_shared += this_test_total_tps_shared
            total_test_fps_shared += this_test_total_fps_shared
            total_test_fns_shared += this_test_total_fns_shared

            total_test_tps_int += this_test_total_tps_int
            total_test_fps_int += this_test_total_fps_int
            total_test_fns_int += this_test_total_fns_int

            if this_test_spec_shared != "None":
                total_test_spec_shared += this_test_spec_shared
                total_test_with_spec_shared_result +=1

            if this_test_sens_shared != "None":
                total_test_sens_shared += this_test_sens_shared
                total_test_with_sens_shared_result +=1

            if this_test_ratio_proteins_result_shared_found != "None":
                total_ratio_proteins_result_shared_found += this_test_ratio_proteins_result_shared_found
                total_test_with_ratio_proteins_result_shared_found += 1


            if this_test_spec_int != "None":  
                total_test_spec_int += this_test_spec_int
                total_test_with_spec_int_result +=1

            if this_test_sens_int != "None":  
                total_test_sens_int += this_test_sens_int
                total_test_with_sens_int_result +=1

            if this_test_ratio_proteins_result_int_found != "None":
                total_ratio_proteins_result_int_found += this_test_ratio_proteins_result_int_found
                total_test_with_ratio_proteins_result_int_found += 1


            if this_test_num_all_proteins != "None":
                total_test_num_all_proteins += this_test_num_all_proteins
                total_test_with_num_all_proteins += 1

            if this_test_number_of_proteins_stop_condition_met != "None":
                total_number_of_proteins_stop_condition_met += this_test_number_of_proteins_stop_condition_met
                total_test_with_number_of_proteins_stop_condition_met += 1

            if this_test_ratio_proteins_stop_condition_met!= "None":
                total_ratio_proteins_stop_condition_met += this_test_ratio_proteins_stop_condition_met
                total_test_with_ratio_proteins_stop_condition_met += 1



        # END OF for subset_number in subsets_dic:

        """

        One test has been done for each subset, using the other subsets for training.

        all_test_* contain the stats for each of these tests

        total_test_* contain the sum of all stats for the n-fold tests
        total_test_with_* contain the number of test occurrences that had a specific stat


        Note: the test_modes "averages" and "totals" are used to set how to calculate the spec and sens within a particular test
           - averages will calculate the spec and sens for each protein, and then do the average of those to assing a spec and sens to the n-fold test
           - totals will calculate the tps, fps and fns for all proteins, and then divide them to assign a spec and sens to the n-fold test


        Then, we do the averages between all the n-fold tests: here, we do want to do the averages because it is a cross-validation. Therefore, we must calculate
        the standard deviation between the stats obtained for each of the tests

        
        """

        
        sys.stderr.write("\n")

        # printing the stats for each of the tests
        for i in range(len(all_test_spec_shared)):
            all_tests_file_fd.write("run=%s\ttps_shared=%s\tfps_shared=%s\tfns_shared=%s\tspec_shared=%s\tsens_shared=%s\tratio_proteins_result_shared_found=%s\ttps_int=%s\tfps_int=%s\tfns_int=%s\tspec_int=%s\tsens_int=%s\tratio_proteins_result_int_found=%s\tnum_all_proteins=%s\tnumber_of_proteins_stop_condition_met=%s\tratio_proteins_stop_condition_met=%s\n" %(
                all_test_runs[i],
                all_test_tps_shared[i],
                all_test_fps_shared[i],
                all_test_fns_shared[i],
                all_test_spec_shared[i],
                all_test_sens_shared[i],
                all_ratio_proteins_result_shared_found[i],
                all_test_tps_int[i],
                all_test_fps_int[i],
                all_test_fns_int[i],
                all_test_spec_int[i],
                all_test_sens_int[i],
                all_ratio_proteins_result_int_found[i],
                all_num_all_proteins[i],
                all_number_of_proteins_stop_condition_met[i],
                all_ratio_proteins_stop_condition_met[i]
                ))
        # END OF for i in range(len(all_test_spec_shared)):


        # Now, calculate the average stats obtained (by adding stats for each test and dividing by the total number of tests)
        number_of_tests = len(all_test_runs)

        av_ratio_num_prots_with_shared_result = total_ratio_proteins_result_shared_found / float(total_test_with_ratio_proteins_result_shared_found)
        av_ratio_num_prots_with_int_result = total_ratio_proteins_result_int_found / float(total_test_with_ratio_proteins_result_int_found)

        av_num_proteins_tested = total_test_num_all_proteins/float(total_test_with_num_all_proteins)
        av_num_proteins_stop_met = total_number_of_proteins_stop_condition_met/float(total_test_with_number_of_proteins_stop_condition_met)
        av_ratio_proteins_stop_met = total_ratio_proteins_stop_condition_met/float(total_test_with_ratio_proteins_stop_condition_met) 


        # --
        # these are the global stats: calculated using the totals from all tests
        #  --> I think this is not mathematically correct... the good way of doing an n-fold is calculating the average and the std deviation from all tests
        # --
        try:
            n_fold_global_spec_shared = total_test_tps_shared ( float(total_test_tps_shared + total_test_fps_shared))
        except:
            n_fold_global_spec_shared = "None"
            
        try:
            n_fold_global_sens_shared = total_test_tps_shared ( float(total_test_tps_shared + total_test_fns_shared))
        except:
            n_fold_global_sens_shared = "None"
            
        try:
            n_fold_global_spec_int = total_test_tps_int ( float(total_test_tps_int + total_test_fps_int))
        except:
            n_fold_global_spec_int = "None"
            
        try:
            n_fold_global_sens_int = total_test_tps_int ( float(total_test_tps_int + total_test_fns_int))
        except:
            n_fold_global_sens_int = "None"

        # --
        # these are the averages from all tests: this is the correct way of doing an n-fold
        # --

        if total_test_with_spec_shared_result:
            n_fold_average_spec_shared = total_test_spec_shared/float(total_test_with_spec_shared_result)
            n_fold_spec_shared_std_deviation = utilities.calculate_std_deviation( all_values= all_test_spec_shared)
                                                                        
        else:
            n_fold_average_spec_shared = "None"
            n_fold_spec_shared_std_deviation = 0

        if total_test_with_sens_shared_result:
            n_fold_average_sens_shared = total_test_sens_shared/float(total_test_with_sens_shared_result)
            n_fold_sens_shared_std_deviation = utilities.calculate_std_deviation( all_values= all_test_sens_shared)
        else:
            n_fold_average_sens_shared = "None"
            n_fold_sens_shared_std_deviation = 0

        if total_test_with_spec_int_result:
            n_fold_average_spec_int = total_test_spec_int/float(total_test_with_spec_int_result)
            n_fold_spec_int_std_deviation = utilities.calculate_std_deviation( all_values= all_test_spec_int)
        else:
            n_fold_average_spec_int = "None"
            n_fold_spec_int_std_deviation = 0

        if total_test_with_sens_int_result:
            n_fold_average_sens_int = total_test_sens_int/float(total_test_with_sens_int_result)
            n_fold_sens_int_std_deviation = utilities.calculate_std_deviation( all_values= all_test_sens_int)
        else:
            n_fold_average_sens_int = "None"
            n_fold_sens_int_std_deviation = 0

        string_with_results = "\n-----------------------------------------\n"
        string_with_results += "Number of proteins with comparison file: %s \n" %(number_of_proteins_with_comparison_file)
        string_with_results += "Size of random subset: %s\n" %(len(subsets_dic[0]))  
        string_with_results += "Average size of training set: %s\n" %(global_size_training_set/float(number_of_tests))  
        string_with_results += "Average number of prots in training with shared info: %s\n" %(global_total_training_information_found_shared/float(number_of_tests))
        string_with_results += "------\n"
        string_with_results += "Number_of_tests: %s \n" %(number_of_tests) 
        string_with_results += "test runs with spec shared result: %s\n" %(total_test_with_spec_shared_result)
        string_with_results += "Average ratio of num of prots with shared result per test: %.2f\n" %(av_ratio_num_prots_with_shared_result)
        string_with_results += "test runs with sens shared result: %s\n" %(total_test_with_sens_shared_result)
        string_with_results += "Average ratio of num of prots with int result per test: %.2f\n" %(av_ratio_num_prots_with_int_result)
        string_with_results += "------\n"
        string_with_results += "Average %s-fold num of prots processed per test (%s tests had proteins): %.2f\n" %(n_fold,total_test_with_num_all_proteins,
                                                                                                                   av_num_proteins_tested )
        string_with_results += "Average %s-fold num of prots stop condition met (%s tests had proteins with proteins that found stop condition): %.2f\n" %(n_fold,
                                                                                                    total_test_with_number_of_proteins_stop_condition_met,  
                                                                                                     av_num_proteins_stop_met )
        string_with_results += "Average %s-fold ratio prots stop condition met (%s tests): %.2f\n" %(n_fold,
                                                                                                    total_test_with_ratio_proteins_stop_condition_met,
                                                                                                     av_ratio_proteins_stop_met   )
        string_with_results += "------\n"
        string_with_results += "------\n"
        string_with_results += "Average %s-fold shared specificity (%s tests had spec shared result): %s (global %s) (std_deviation=%s)\n" %(n_fold,
                                                                                                                          total_test_with_spec_shared_result,
                                                                                                                          n_fold_average_spec_shared,
                                                                                                                          total_test_spec_shared,
                                                                                                                          n_fold_spec_shared_std_deviation       )
        string_with_results += "Average %s-fold shared sensibility (%s tests has sens shared result): %s (global %s) (std_deviation=%s)\n" %(n_fold,
                                                                                                                                  total_test_with_sens_shared_result,
                                                                                                                                  n_fold_average_sens_shared,
                                                                                                                          total_test_sens_shared,
                                                                                                                          n_fold_sens_shared_std_deviation )
        string_with_results += "------\n" 
        string_with_results += "Average %s-fold int specificity (%s tests had spec int result): %s (global %s) (std_deviation=%s)\n" %(n_fold,
                                                                                                                                    total_test_with_spec_int_result,
                                                                                                                                    n_fold_average_spec_int,
                                                                                                                                    total_test_spec_int,
                                                                                                                                    n_fold_spec_int_std_deviation)
        string_with_results += "Average %s-fold int sensibility (%s tests has sens int result): %s (global %s) (std_deviation=%s)\n" %(n_fold,
                                                                                                                                  total_test_with_sens_int_result,
                                                                                                                                  n_fold_average_sens_int,
                                                                                                                                  total_test_sens_int,
                                                                                                                                  n_fold_sens_int_std_deviation)
        string_with_results += "------\n"
        string_with_results += "Global %s-fold shared specificity (tps=%s, fps=%s, fns=%s): %s\n" %(n_fold,
                                                                                                    total_test_tps_shared,total_test_fps_shared,total_test_fns_shared,
                                                                                                    n_fold_global_spec_shared)
        string_with_results += "Global %s-fold shared sensitivity (tps=%s, fps=%s, fns=%s): %s\n" %(n_fold,
                                                                                                    total_test_tps_shared,total_test_fps_shared,total_test_fns_shared,
                                                                                                    n_fold_global_sens_shared)
        string_with_results += "------\n"
        string_with_results += "Global %s-fold int specificity (tps=%s, fps=%s, fns=%s): %s\n" %(n_fold,
                                                                                                    total_test_tps_int,total_test_fps_int,total_test_fns_int,
                                                                                                    n_fold_global_spec_int)
        string_with_results += "Global %s-fold int sensitivity (tps=%s, fps=%s, fns=%s): %s\n" %(n_fold,
                                                                                                    total_test_tps_int,total_test_fps_int,total_test_fns_int,
                                                                                                    n_fold_global_sens_int)
        string_with_results += "\n-----------------------------------------\n"

        # print results to screen
        sys.stderr.write(string_with_results)

        # in the file that has the individual stats for each n-fold test, print as well the averages for all the tests
        all_tests_file_fd.write("%s" %string_with_results )
        all_tests_file_fd.close()

        this_n_fold_results = ScoringFunctionBenchmark.Results(spec_shared= n_fold_average_spec_shared,
                                                               spec_shared_std_deviation= n_fold_spec_shared_std_deviation,
                                                               sens_shared= n_fold_average_sens_shared,
                                                               sens_shared_std_deviation= n_fold_sens_shared_std_deviation,
                                                               spec_int= n_fold_average_spec_int,
                                                               spec_int_std_deviation= n_fold_spec_int_std_deviation,
                                                               sens_int= n_fold_average_sens_int,
                                                               sens_int_std_deviation= n_fold_sens_int_std_deviation,
                                                               num_proteins_tested= av_num_proteins_tested,
                                                               ratio_proteins_stop_met= av_ratio_proteins_stop_met,
                                                               ratio_proteins_with_shared_result= av_ratio_num_prots_with_shared_result,
                                                               ratio_proteins_with_int_result= av_ratio_num_prots_with_int_result,
                                                               parameters_object=  parameters_to_use )
        all_n_fold_results.append(this_n_fold_results)
    # END OF for parameters_to_use in list_of_parameters_to_use

    roc_results_file_name = output_prefix + "." + str(n_fold) + "_fold" + "." + str(num_different_thresholds) + "_num_thresholds" + ".roc_table"
    roc_results_file_fd = file(roc_results_file_name, "w")
    
    roc_results_file_fd.write("spec_shared\tspec_shared_std_deviation\tsens_shared\tsens_shared_std_deviation\tspec_int\tspec_int_std_deviation\tsens_int\tsens_int_std_deviation\tnum_prots_tested\tratio_proteins_stop_met\tratio_proteins_with_shared_result\tratio_proteins_with_int_result\tcoverage_shared_threshold\tcoverage_int_threshold\tspec_shared_threshold\tsens_shared_threshold\tspec_int_threshold\tsens_int_threshold\tranking_mode\n")
    for results in all_n_fold_results:
        roc_results_file_fd.write("%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n" %(results.spec_shared,
                                                                                                                   results.spec_shared_std_deviation,
                                                                                                                   results.sens_shared,
                                                                                                                   results.sens_shared_std_deviation,
                                                                                                                   results.spec_int,
                                                                                                                   results.spec_int_std_deviation,
                                                                                                                   results.sens_int,
                                                                                                                   results.sens_int_std_deviation,
                                                                                                                   results.num_proteins_tested,
                                                                                                                   results.ratio_proteins_stop_met,
                                                                                                                   results.ratio_proteins_with_shared_result,
                                                                                                                   results.ratio_proteins_with_int_result,
                                                                                                   results.parameters_object.get_coverage_shared_threshold(),
                                                                                                   results.parameters_object.get_coverage_int_threshold(),
                                                                                                   results.parameters_object.get_spec_shared_threshold(),
                                                                                                   results.parameters_object.get_sens_shared_threshold(),
                                                                                                   results.parameters_object.get_spec_int_threshold(),
                                                                                                   results.parameters_object.get_sens_int_threshold(),
                                                                                                   results.parameters_object.get_ranking_mode()                ) )
       
    roc_results_file_fd.close()
# END OF if __name__ = "__main__":

