"""
File        : GoGraph.py
Author      : Pablo Boixeda & Ramon Aragues
Creation    : 7.2005
Contents    : implementation of graph handling GO (subclass of Graph.py)
Called from : programs/classes that need to use GO graphs

=======================================================================================================


GoGraph is used to cluster a protein-protein interaction network by looking at similarities between GO terms.

In order to make a good cluster, a goGraph object is required, because one proteinPiana
can have more than one term_id for one term_type
the process of clustering goes as follows:

   pianaGraph -> goGraph -> clustering -> graphclustered
   
"""

# GoGraph.py :  implementation of graph handling GO (subclass of Graph.py)
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues


from Graph import *
from PianaDBaccess import *
from GoGraphNodeAttribute import *

#----------------------------------------------------------------------------------------------
class GoGraph(Graph):
#----------------------------------------------------------------------------------------------

    """
    Implements a Graph where nodes are GO (Gene Ontology) terms
    """
    def __init__(self,graphID = None, piana_access_object = None, term_type_value = None):

        self.graph_id = graphID
        self.piana_access = piana_access_object
        self.node_id_counter = 0
        self.term_type=term_type_value

        Graph.__init__(self,self.graph_id)


    def _get_new_node_id(self):
        """
        This method is created to give an id to new nodes, if two
        nodes has the same id some problems can occur so we insurance
        that no two nodes id are equal.        
        """
        new_node_id = self.node_id_counter
        self.node_id_counter += 1
        return new_node_id


    
    def initialize_GoGraph_from_pianaGraph(self, pianaGraph = None):
        """
        Method that initialize a GoGraph from a PianaGraph
        Creates a node for each term_id associated to a proteinPiana value.
        If a proteinPiana doesn't have any term type associated, no node will be created.

        "pianaGraph": pianaGraph to extract information about the network
        """

        if pianaGraph is None:
            raise ValueError ("No pianaGraph to transform to GO")

        else:

            piana_node_list = pianaGraph.get_node_object_list()
            #Getting node objects from pianaGraph
            new_node2old_node_dic={}
            old_node2new_node_dic={}
            new_node_list=[]
            #Initializing dictionaries to save the association between old nodes and new nodes that will be created

            for piana_node in piana_node_list:
                piana_node_attribute = piana_node.get_node_attribute_object() #Getting the node_attribute
                protein_piana = piana_node_attribute.get_proteinPiana() #getting the proteinPiana ID

                
                go_term_list = self.piana_access.get_protein_go_term_id(proteinPiana_value = protein_piana,
                                                                        term_type_value = self.term_type)
                # query to database to extract the possible GO terms associated to this proteinPiana

                if go_term_list :
             
                    for go_term in go_term_list :
                        node_id= self._get_new_node_id()
                        go_node= GraphNode(nodeID= node_id)
                        
                        #Creating GoGraphNodeAttribute object with the information needed
                        #                              ->proteinPiana ID
                        #                              ->term id  (sprecific GO ID code)
                        #                              ->name is the term name i.e. ejemplo
                        #                              ->is_root if the go term is associated to a root protein in the network
                        go_attribute= GoGraphNodeAttribute(proteinPiana_value=protein_piana,
                                                           term_id_value = go_term,
                                                           name=self.piana_access.get_protein_go_name(go_term_id_value=go_term),
                                                           is_root_value=piana_node.is_root())
                        
                        go_node.set_attribute(go_attribute)
                        new_node_list.append(go_node)
                        new_node2old_node_dic[node_id] = protein_piana                        

                        #saving the relationship between old nodes and new nodes
                        if old_node2new_node_dic.has_key(protein_piana):
                            old_node2new_node_dic[protein_piana].append(node_id)
                          
                        else :
                            old_node2new_node_dic[protein_piana] = [node_id]

                        #END of if old_node...
                        
                    #END of for go_term...
                #END of if go_term_list
                elif not go_term_list and piana_node.is_root():

                    #some cases a protein doesn't have a Go term associated. If one protein doesn't have a term associated
                    #we represent these protein in the GO Graph only in case of a root protein
                    node_id= self._get_new_node_id()
                    go_node= GraphNode(nodeID= node_id)
                    go_attribute= GoGraphNodeAttribute(proteinPiana_value=protein_piana,
                                                       term_id_value = 1,
                                                       name= "root: %s" %protein_piana,
                                                       is_root_value=piana_node.is_root())
                    go_node.set_attribute(go_attribute)
                    new_node_list.append(go_node)
                    new_node2old_node_dic[node_id]=protein_piana
                    
                    #saving the relationship between old nodes and new nodes
                    old_node2new_node_dic[protein_piana]= [node_id]
                #END of elif not go_term_list...
                    
            #END of piana_node in piana_node_list

            # Now we have the new Node created but not inserted in the Graph. Let's build the Graph using the dictionaries to set the edges.

            edge_dictionary=pianaGraph.get_nodes_connectivities()

            # Testing dictionaries and node list
            """
            print "Edge dictionary is: %s" % edge_dictionary
            print "***********************"
            print "Old_node2new_node_dic is: %s" %old_node2new_node_dic
            print "***********************"
            print "New_node2old_node_dic is : %s" %new_node2old_node_dic
            print "***********************"
            print "END OF DICTIONARY & LIST TEST"
            """
            #end of dictionary test
            
            return Graph.build_expanded_graph(node_list=new_node_list,
                                              old_edges_dictionary=edge_dictionary,
                                              old_node2new_node_dic=old_node2new_node_dic,
                                              new_node2old_node_dic= new_node2old_node_dic,
                                              newGraph=self)

    def print_go_node_values(self,go_node = None):
        """
        prints the label of one GO node
        
        "go_node": node from a go graph
        """

        node_attribute=go_node.get_node_attribute_object()

        return "%s        proteinPiana %s : GoTerm %s"  %(node_attribute.get_go_name(),
                                                         node_attribute.get_proteinPiana(),
                                                         node_attribute.get_term_id())

    def print_go_dot_file(self, output_target= None):
        """
        method that prints GoGraph in dot format to a file object "output_target"
        """
        output_target.write("graph G { graph [orientation=portrait, pack=true, overlap=scale]\n")
        # uncomment this line for best presentation
        output_target.write(" node [shape=box,fontsize=12,width=0.15,height=0.15,style=filled,fillcolor=lightblue];\n")

        for node in self.get_node_object_list():
            if node.get_node_attribute_object().get_is_root():
                output_target.write(""""%s" [fillcolor = %s]\n""" %(self.print_go_node_values(go_node = node),"yellow"))

        for edge in self.get_edge_object_list():
            start_node= edge.get_start_node_id()
            end_node= edge.get_end_node_id()
            output_target.write(""" "%s" -- "%s" [len=2];\n""" %(self.print_go_node_values(go_node = self.get_node(start_node)),
                                                                 self.print_go_node_values(go_node = self.get_node(end_node))))
        # END OF for edge in self.edges:
                                 
        # print graph termination    
        output_target.write( "}\n")   

        
                    

                        
        
    


