"""
File        : CirGraphCluster.py
Author      : Ramon Aragues
Creation    : 7.2.2006
Contents    : implements a clustered graph where the elements of the cluster are CirGraphNodeAttribute objects
Called from : 

=======================================================================================================
"""

# CirGraphCluster.py: implements a clustered graph where the elements of the cluster are CirGraphNodeAttribute objects
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

from GraphCluster import *

class CirGraphCluster(GraphCluster):

    def __init__(self, graphID= None, piana_access=None):

        self.piana_access= piana_access
        

        Graph.__init__(self, graphID= graphID)

    def do_action(self):
        """
        This is the generic method called after each step of the clustering.

        """
        pass # not doing anything...


    def print_pairs_same_cluster(self, output_target, root_protein):
        """
        prints to "output_target" pairs of proteins (ie. proteins with elements in that cluster)  that appear in the same cluster

        "root_protein" is used to uniquely identify the clusters (clusters have unique ids if they were all obtained from the
                       same network... but since we are doing multiple calls from run_piana_protein_by_protein, clusters
                       need to be distinguished by their id in the network and the root protein that was used to generate them 
        """

        already_printed_pairs = {}  # to avoid printing repetitions
        
        nodes= self.get_node_object_list()
        
        for node in nodes:
            elements_list= node.get_node_attribute_object().get_list_elements()
            number_of_elements = len(elements_list)
            for i in range(number_of_elements):
                for j in range(i+1, number_of_elements):
                    proteinPiana_i = elements_list[i].get_proteinPiana()
                    proteinPiana_j = elements_list[j].get_proteinPiana()
                    if proteinPiana_i != proteinPiana_j:

                        # make sure that no duplication of pairs occurs...
                        if proteinPiana_i <= proteinPiana_j:
                            p1 = proteinPiana_i
                            p2 = proteinPiana_j
                        else:
                            p1 = proteinPiana_j
                            p2 = proteinPiana_i
                        
                        key = "%s.%s" %(p1, p2)
                        if already_printed_pairs.has_key(key):
                            continue
                        already_printed_pairs[key] = None

                        cluster_name = "%s.%s" %(root_protein, node.get_node_id())
                        
                        output_target.write("share\t%s\t%s\t%s\n" %(cluster_name , p1, p2) )
                # END OF for j in range(i+1, number_of_elements):
            # END OF for i in range(number_of_elements):
        # END OF for node in nodes:
            

    def print_pairs_interactions(self, output_target, root_protein):
        """
        prints  to "output_target" pairs of proteins whose clusters interact
        
        "root_protein" is used to uniquely identify the clusters (clusters have unique ids if they were all obtained from the
                       same network... but since we are doing multiple calls from run_piana_protein_by_protein, clusters
                       need to be distinguished by their id in the network and the root protein that was used to generate them 
        """
        already_printed_pairs = {}  # to avoid printing repetitions
        
        for edge in self.get_edge_object_list():

            node_id_start = edge.get_start_node_id()
            node_start = self.get_node(identifier= node_id_start, get_mode="error") 
            list_elements_start = node_start.get_node_attribute_object().get_list_elements()
            
            node_id_end = edge.get_end_node_id()
            node_end = self.get_node(identifier= node_id_end, get_mode="error") 
            list_elements_end = node_end.get_node_attribute_object().get_list_elements()

            for one_element_start in list_elements_start:
                for one_element_end in list_elements_end:

                    proteinPiana_start = one_element_start.get_proteinPiana()
                    proteinPiana_end = one_element_end.get_proteinPiana()
                    
                    # make sure that no duplication of pairs occurs...
                    if proteinPiana_start <= proteinPiana_end:
                        p1 = proteinPiana_start
                        cluster_1_name = "%s.%s" %(root_protein, node_start.get_node_id())
                        p2 = proteinPiana_end
                        cluster_2_name = "%s.%s" %(root_protein, node_end.get_node_id())
                    else:
                        p1 = proteinPiana_end
                        cluster_1_name = "%s.%s" %(root_protein, node_end.get_node_id())
                        p2 = proteinPiana_start
                        cluster_2_name = "%s.%s" %(root_protein, node_start.get_node_id())
                        
                    key = "%s.%s" %(p1, p2)
                    if already_printed_pairs.has_key(key):
                        continue
                    already_printed_pairs[key] = None

                    output_target.write("int\t%s-%s\t%s\t%s\n" %(cluster_1_name, cluster_2_name, p1, p2) )
                # END OF for one_element_end in list_elements_end:
            # END OF for one_element_start in list_elements_start:
        # END OF for edge in self.get_edge_object_list():

