"""
File     : GraphNodeAttribute.py
Author   : Daniel Jaeggi and Ramon Aragues
Creation : 2003
Contents : class NodeAttribute

=======================================================================================================

Skeleton attribute classes for nodes in a graph.

nodes in a graph are just abstract containers and only have
"real world" meanings by containing attributes. For each particular
problem, custon edge and node attributes must be defined.

For example, we may be modeling the network formed by Hollywood actors.
Actors form the nodes and films in which both actors have starred form
the edges. We could then define custom node attributes thus:

class Actor(NodeAttribute):
    # class defining an movie actor
    def __init__(self, name, n_oscars=0, speciality='hollywood_blockbusters'):
        self.name = name
        self.n_oscars = int(n_oscars)
        self.speciality = speciality

Attributes in this way can be arbitrarily complex 
"""

# GraphNodeAttribute.py: implements a class for node attribute contained in a GraphNode object
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues




class GraphNodeAttribute:
    """
    The master class for all GraphNode attributes. It is envisaged that this
    class will form the basis for custom attribute classes.
    """

    def get_node_id(self):
        """
        must be overwritten with a method that returns an identifier of this attribute
        """
        raise ValueError("This GraphNodeAttribute has no identifier because it has not been implemented\n")
    
    def __eq__(self,other):
        """
        When defining new Attribute classes, the equality operator MUST be overridden.
        Otherwise, adding further attributes to existing nodes in the graph may cause
        duplicated attributes.
        """
        if self == other:
            return 1
        else:
            return 0

    def merge_attribute(self, attribute_object, ignore_ids):
        """
        Must be overwritten by a method specific to the kind of node being used for the Graph
        
        merges existing attribute with atribute_object passed as argument
        """
        pass
