"""
File     : PianaGraphFilters.py
Author   : Ramon Aragues
Creation : 25.02.2004
Contents : classes implementing filters that can be applied to PianaGraph
Comments :

=======================================================================================================


A filter can either be a node or and edge filter and must contain a method
called "test" which returns either a 1, if the entity passes the filter, or
a 0 otherwise

"""

# PianaGraphFilters.py: classes implementing filters that can be applied to PianaGraph
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

from Filter import *

verbose = 0

class PianaNodeFilter(NodeFilter):
    """
    Not implemented
    """
    pass

class PianaEdgeFilter(EdgeFilter):
    """
    Not implemented
    """
    pass


class FilterInteractionSourceDB(PianaEdgeFilter):
    """
    Filter used for hiding interactions that belong (or do not belong) to a certain database
    """
    
    def __init__(self, list_dbs, belong_mode ):
        """

        "belong_mode" can be:
          - "filter_not_belonging": hides interactions of databases which are not in "list_dbs"
          - "filter_belonging": hides interactions of databases which are in list_dbs


        if an interaction appears in two different databases, and a filter is applied on one of them, the interaction remains unhidden, since it
        still appears in a database that is not filtered


        """
        # TO DO!!!! The edge_attribute must somehow know that a specifific source database has been filtered: otherwise, when visualizing information
        # the edge_attribute will say that it can be found on a specific database. And this database is supposed to be filtered!!!! The edge_attribute must
        # know that it is unhidden because one of its source databases was not filtered... how do I keep track of this???? Maybe I could do it at the
        # visualization level... maybe have an attribute inside the edge_attribute... don know how to do it!!! TO DO!!!!! And this is true for all filters!!!!

        if not list_dbs:
            raise ValueError("Strange filter... list_dbs is empty")
        else:
            self.list_dbs = list_dbs

        if belong_mode is None:
            raise ValueError("belong_mode cannot be None for this filter")
        
        elif belong_mode != "filter_not_belonging" and belong_mode != "filter_belonging": 
            raise ValueError("incorrect value given to belong_mode for this filter: belong_mode must be 'filter_not_belonging' or 'filter_belonging' ")
        else:
            self.belong_mode = belong_mode

    
    def test(self,edge):
        """
        performs the test on edge to check the conditions set for the filter

        checks if the edge source db (obtained from the edge attributes) is in the list of dbs passed to the filter, and depending on the belong_mode
        returns 1 (edge must be hidden) or 0 (edge remains unhidden)
        """
        keep_unhidden = 0
        
        edge_attribute = edge.get_edge_attribute_object()
        edge_source_dbs = edge_attribute.get_list_source_db()

        if verbose:
            sys.stderr.write("Inside Filter, interactionPiana is %s of db %s  " %(edge_attribute.get_interactionPiana(),
                                                                                  edge_source_dbs ))

        # depending on the belong_mode, the process is different
        if self.belong_mode == "filter_belonging":

            # belong_mode filter_belonging makes sure the lenght of the list of source_dbs is bigger than 0
            # after removing all those databases that have to be filtered
            for filter_db in self.list_dbs:

                if filter_db in edge_source_dbs:
                    index = edge_source_dbs.index(filter_db)
                    edge_source_dbs.pop(index)

                if edge_source_dbs:
                    keep_unhidden = 1

        elif self.belong_mode == "filter_not_belonging":

            # belong_mode filter_not_belonging makes sure if the list of source_dbs contains any of the databases in the filter
            # the edge will be kept unhidden
            for edge_source_db in edge_source_dbs:

                if edge_source_db in self.list_dbs:
                    keep_unhidden = 1

        return keep_unhidden

