"""
File        : delete_interactions_from_db.py
Author      : Ramon Aragues
Creation    : 18.05.2004
Contents    : script that deletes specific interactions from piana databases
Called from : 

=======================================================================================================

This program deletes interaction entries from piana databases, according to conditions imposed from command line

"""

import sys
import getopt

import re
import readline

import MySQLdb

from PianaDBaccess import *

verbose = 1

# ---------------------------------------------------------------
# Set here the default values for command line arguments
# ---------------------------------------------------------------
#
# These will be the values taken by the program when called directly from build_database.py




# ----------------------
# Function usage()
# ----------------------
def usage():
    print "--------------------------------------------------------------------------------------------------------------"
    print "This program deletes specific interactions from piana databases\n"
    print "Usage: python delete_interactions_from_db.py  --db-to-delete=db_to_delete --protein-to-delete=protein_to_delete"
    print "           --piana-dbname=piana_dbname --piana-dbhost=piana_dbhost --piana-dbuser=piana_dbuser --piana-dbpass=piana_dbpass  [--help] [--verbose]"
    print "\nwhere:"
    print "     piana_dbname : name of database piana to be used (required)"
    print "     piana_dbhost : name of host where database piana to be used is placed (required)"
    print "     piana_dbuser : username accessing the database (not required in most systems)"
    print "     piana_dbpass : password of username accessing the database (not required in most systems)"
    print "     db_to_delete   : name of the database whose interactions are to be deleted"
    print "     protein_to_delete   : proteinPiana of the protein whose interactions are to be deleted"
    print "     --help         : prints this message and exits"
    print "     --verbose      : prints process info to stdout"
    print "--------------------------------------------------------------------------------------------------------------"
        

   
# ---------------------------
# Function parseArguments()                                               
# --------------------------- 

def parseArguments():
    
    global piana_dbname
    global piana_dbhost
    global piana_dbuser
    global piana_dbpass
    
    global db_to_delete
    global protein_to_delete

    global verbose
    
    try:
        opts, args = getopt.getopt(sys.argv[1:], "vhf:d:t:n:o:u:w:", ["verbose","help", "db-to-delete=", "protein-to-delete=",
                                                                   "piana-dbname=", "piana-dbhost=", "piana-dbuser=", "piana-dbpass="])
    except getopt.GetoptError:
        # print help information and exit:
        usage()
        sys.exit(2)
     
    for option,value in opts:
        
        if option in ("-n", "--piana-dbname"):
            piana_dbname = value
            
        elif option in ("-o", "--piana-dbhost"):
            piana_dbhost = value
            
        elif option in ("-u", "--piana-dbuser"):
            piana_dbuser = value
            
        elif option in ("-w", "--piana-dbpass"):
            piana_dbpass = value
            
        elif option in ("-d", "--db-to-delete"):
            db_to_delete = value
            
        elif option in ("-t", "--protein-to-delete"):
            protein_to_delete = value
            
        elif option in ("-x", "--verbose"):
            verbose = 1
            
        elif option in ("-h", "--help"):
            # print help information and exit
            usage()
            sys.exit(2)
             
    # END OF for option,value in opts:

    if db_to_delete is None and protein_to_delete is None:
        sys.stderr.write("You must set a database or a protein to delete\n")
        usage()
        sys.exit(2)
        
        
        
# --------
# --------
#  Main()               
# --------                               
# --------
db_to_delete = None
protein_to_delete = None

piana_dbname = None
piana_dbuser = None
piana_dbhost = None
piana_dbpass = None

# parsing arguments from the command line
parseArguments()

# Initialisating connection to piana
piana_access = PianaDBaccess(dbname=piana_dbname, dbhost=piana_dbhost, dbuser=piana_dbuser, dbpassword= piana_dbpass)

number_of_interactions_deleted = 0

if db_to_delete is not None:
    # Deleting interactions inserted by a specific database involves deleting records from 3 tables
    # interactionMethod, interactionSourceDB and interaction. However, entries in table interaction
    # will only be deleted if there were no other databases describing that specific interaction between two proteins
    #
    # Process of deletion of interactions that were inserted from database DB
    # --> 1. delete entries in table interactionMethod where sourceDBID = DB
    #        delete from interactionMethod where sourceDBID= DB
    # --> delete entries in table interactionSourceDB where sourceDBID = DB
    #        delete from interactionSourceDB where sourceDBID= DB (keep list of deleted interactionPiana)
    # --> delete entries in table interaction that do not have a corresponding interactionSourceDB entry
    #       for deleted_interactionPiana in deleted_interactionPianas:
    #            if deleted_interactionPiana not in table interactionSourceDB:
    #                  delete deleted_interactionPiana from table interaction

    sourceDBID= piana_access.get_sourceDBID(sourceDBDescription_value= db_to_delete)
    
    piana_access.delete_interaction_scores_for_sourceDBID(sourceDBID_value =  sourceDBID)
    piana_access.delete_interaction_protein_source_for_sourceDBID(sourceDBID_value = sourceDBID )
    piana_access.delete_interaction_method_for_sourceDBID(sourceDBID_value = sourceDBID )
    
    deleted_interactionPianas = piana_access.delete_interaction_sourceDB_for_sourceDBID( sourceDBID_value = sourceDBID )

    
    for deleted_interactionPiana in deleted_interactionPianas:
        if not piana_access.get_interaction_methodID_list(interactionPiana_value = deleted_interactionPiana):
            # if there are no methods associated to this interactionPiana it means the deleted entry was the only appearance of this interactionPiana
            piana_access.delete_interaction(interactionPiana_value = deleted_interactionPiana)
            number_of_interactions_deleted += 1
        # END OF if not piana_access.get_interaction_methodID_list(interactionPiana_value = deleted_interactionPiana):
    # END OF for deleted_interactionPiana in deleted_interactionPianas:

sys.stderr.write("Number of interactions deleted: %s\n" %(number_of_interactions_deleted))
    
# END OF if db_to_delete is not None:

if protein_to_delete is not None:
    sys.stderr.write("Warning! Function still not implemented")



