"""
File        : GOApi.py
Author      : Ramon Aragues
Creation    : 27.09.2005
Contents    : methods used to access GO information 
Called from : command line

=======================================================================================================


"""

# GOApi.py: methods used to access GO information 
#
# Copyright (C) 2005  Ramon Aragues
# author email: ramon.aragues@upf.edu and boliva@imim.es
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#    http://www.gnu.org/copyleft/gpl.html
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# University Pompeu Fabra, hereby disclaims all copyright
# interest in the program 'PIANA'
# (software for working with protein-protein interaction networks) written 
# by Ramon Aragues

import sys
import getopt

import re
import readline
import copy

import MySQLdb


def get_fathers(gocursor=None,term_id=None):
   """
   get all the GO fathers of term_id

   returns a list of term ids
   """
   
   sqlquery="select term1_id from term2term where term2_id=%s"%term_id
   gocursor.execute(sqlquery)
   fathers=gocursor.fetchall()
   fathers_list=[]
   for father in fathers:
      fathers_list.append(father[0])


   return fathers_list


def get_parents_at_level_N(list_go_terms, level_desired, dic_gos_at_level, gocursor, piana_access):
    """
    returns parents of list_go_terms at a certain level "level_desired"


    This method is used to get the upper terms at a given level of a list of GO terms


    "list_go_terms" is the list of go terms for which you want to get the parents

    "level_desired" is the level number you want the parents to be

    "dic_gos_at_level" is a dictionary with terms already found for that level (set to {} when calling... it is used for recursion purposes)

    "gocursor" is a mysql cursor pointing to a GO database

    
    """
    temp_dic_gos_at_level = dic_gos_at_level
    
    if not list_go_terms:
       # if no more go terms, end recursion
       return temp_dic_gos_at_level
    
    else:
       new_list_to_check = []
       
       for go_term in list_go_terms:

          distance_to_root = piana_access.get_go_depth(term_id_value= go_term)
          
          if distance_to_root == level_desired:
             # level chosen by user reached... return that go term
             temp_dic_gos_at_level[go_term] = None
             
          elif distance_to_root > level_desired:
             new_list_to_check.extend(get_fathers(gocursor=gocursor, term_id=go_term)) # next "generation" gos to check
             
       # END OF for go_term in list_go_terms:

       return get_parents_at_level_N(list_go_terms=new_list_to_check, level_desired=level_desired,
                                       dic_gos_at_level= temp_dic_gos_at_level, gocursor= gocursor, piana_access= piana_access)
       
    # END OF else: (if not list_go_terms:)


def get_all_parents(list_go_terms, gocursor, piana_access, dic_gos_at_level={}, stop_level = 2):
    """
    This method is used to get the terms for all lower levels (ie. less specific) of a list of GO terms

    returns a a list with all parents of list_go_terms until reaching a certain level "stop_level"

      -> does not return list_go_terms inside the list, only the parents

    "list_go_terms" is the list of go terms for which you want to get all the parents

    "gocursor" is a mysql cursor pointing to a GO database

    "piana_access"  is a connection to a PIANA database

    "dic_gos_at_level" is a dictionary with terms already found  (set to {} when calling... it is used for recursion purposes)

    "stop_level" sets on which level will this method stop returning parents (default is 2)
       --> GO levels are:  root (0) -- category_root (1) (eg. molecular function) -- first particularity of function (2) -- (3) -- (4) -- ...
       --> levels start being descriptive at GO level 3-4

    
    """
    temp_dic_gos_at_level = copy.deepcopy(dic_gos_at_level)
    
    if not list_go_terms:
       # if no more go terms, end recursion
       return temp_dic_gos_at_level.keys()
    
    else:
       new_list_to_check = []
       
       for go_term in list_go_terms:

          distance_to_root = piana_access.get_go_depth(term_id_value= go_term)
          
          if distance_to_root >= stop_level:
             # only considering parents that are not root or category root (eg. molecular_function)
             temp_dic_gos_at_level[go_term] = None
             new_list_to_check.extend(get_fathers(gocursor=gocursor, term_id=go_term)) # next "level" gos to check
             
       # END OF for go_term in list_go_terms:

       return get_all_parents(list_go_terms=new_list_to_check, 
                              dic_gos_at_level= temp_dic_gos_at_level, gocursor= gocursor, piana_access= piana_access)
       
    # END OF else: (if not list_go_terms:)
    
